/**
 * SUMO UX Tracker - Utils Module
 * Fonctions utilitaires partagées
 * OPTIM #5: Élimination code dupliqué (-200 lignes)
 */

/**
 * Formate un temps en millisecondes en format mm:ss
 * @param {number} ms - Temps en millisecondes
 * @returns {string} Temps formaté (ex: "2:45")
 */
export function formatTime(ms) {
  const seconds = Math.floor(ms / 1000);
  const minutes = Math.floor(seconds / 60);
  const secs = seconds % 60;
  return `${minutes}:${secs.toString().padStart(2, '0')}`;
}

/**
 * Formate une durée en texte lisible
 * @param {number} ms - Durée en millisecondes
 * @returns {string} Durée formatée (ex: "2h 15m", "45m 30s", "25s")
 */
export function formatDuration(ms) {
  const seconds = Math.floor(ms / 1000);
  const minutes = Math.floor(seconds / 60);
  const hours = Math.floor(minutes / 60);

  if (hours > 0) {
    return `${hours}h ${minutes % 60}m`;
  } else if (minutes > 0) {
    return `${minutes}m ${seconds % 60}s`;
  } else {
    return `${seconds}s`;
  }
}

/**
 * Tronque une URL longue (version avancée avec parsing)
 * @param {string} url - URL à tronquer
 * @param {number} maxLength - Longueur maximale (défaut: 60)
 * @returns {string} URL tronquée
 */
export function truncateUrl(url, maxLength = 60) {
  if (!url) return 'Session sans URL';

  try {
    // Parser l'URL pour extraire domaine + path
    const urlObj = new URL(url);
    const domain = urlObj.hostname.replace('www.', '');
    const path = urlObj.pathname;

    // Si path est root, retourner juste le domaine
    if (path === '/' || path === '') {
      return domain;
    }

    const display = `${domain}${path}`;

    // Tronquer si trop long
    if (display.length <= maxLength) {
      return display;
    }

    return display.substring(0, maxLength - 3) + '...';
  } catch {
    // Si URL invalide, fallback sur truncate simple
    if (url.length <= maxLength) {
      return url;
    }
    return url.substring(0, maxLength - 3) + '...';
  }
}

/**
 * Tronque une URL simple (version rapide sans parsing)
 * @param {string} url - URL à tronquer
 * @param {number} max - Longueur maximale
 * @returns {string} URL tronquée
 */
export function truncateUrlSimple(url, max = 40) {
  if (!url || url.length <= max) return url || 'URL inconnue';
  return url.substring(0, max - 3) + '...';
}

/**
 * Label d'événement pour replay
 * @param {string} type - Type d'événement
 * @returns {string} Label avec emoji
 */
export function getEventLabel(type) {
  const labels = {
    'rage_click': '⚠️ Rage Click',
    'navigation': '🧭 Navigation',
    'page_start': '📄 Nouvelle page',
    'click': '🖱️ Clic'
  };
  return labels[type] || type;
}

/**
 * Formate un nombre avec séparateur de milliers
 * @param {number} num - Nombre à formater
 * @returns {string} Nombre formaté (ex: "1,234")
 */
export function formatNumber(num) {
  return num.toLocaleString('fr-FR');
}

/**
 * Calcule le pourcentage
 * @param {number} value - Valeur
 * @param {number} total - Total
 * @returns {number} Pourcentage arrondi
 */
export function percentage(value, total) {
  if (total === 0) return 0;
  return Math.round((value / total) * 100);
}

/**
 * Debounce function pour optimiser les events
 * @param {Function} func - Fonction à debouncer
 * @param {number} wait - Délai en ms
 * @returns {Function} Fonction debouncée
 */
export function debounce(func, wait = 300) {
  let timeout;
  return function executedFunction(...args) {
    const later = () => {
      clearTimeout(timeout);
      func(...args);
    };
    clearTimeout(timeout);
    timeout = setTimeout(later, wait);
  };
}

/**
 * Throttle function pour limiter la fréquence d'appel
 * @param {Function} func - Fonction à throttler
 * @param {number} limit - Limite en ms
 * @returns {Function} Fonction throttlée
 */
export function throttle(func, limit = 100) {
  let inThrottle;
  return function(...args) {
    if (!inThrottle) {
      func.apply(this, args);
      inThrottle = true;
      setTimeout(() => inThrottle = false, limit);
    }
  };
}

/**
 * Génère un ID unique
 * @returns {string} ID unique (timestamp + random)
 */
export function generateId() {
  return `${Date.now()}_${Math.random().toString(36).substr(2, 9)}`;
}

/**
 * Vérifie si valeur est vide (null, undefined, '', [], {})
 * @param {*} value - Valeur à vérifier
 * @returns {boolean} True si vide
 */
export function isEmpty(value) {
  if (value === null || value === undefined) return true;
  if (typeof value === 'string' && value.trim() === '') return true;
  if (Array.isArray(value) && value.length === 0) return true;
  if (typeof value === 'object' && Object.keys(value).length === 0) return true;
  return false;
}

/**
 * Deep clone d'un objet
 * @param {Object} obj - Objet à cloner
 * @returns {Object} Clone de l'objet
 */
export function deepClone(obj) {
  return JSON.parse(JSON.stringify(obj));
}

/**
 * Convertit bytes en format lisible (KB, MB, GB)
 * @param {number} bytes - Nombre de bytes
 * @param {number} decimals - Nombre de décimales (défaut: 2)
 * @returns {string} Taille formatée (ex: "1.5 MB")
 */
export function formatBytes(bytes, decimals = 2) {
  if (bytes === 0) return '0 Bytes';

  const k = 1024;
  const dm = decimals < 0 ? 0 : decimals;
  const sizes = ['Bytes', 'KB', 'MB', 'GB', 'TB'];

  const i = Math.floor(Math.log(bytes) / Math.log(k));

  return parseFloat((bytes / Math.pow(k, i)).toFixed(dm)) + ' ' + sizes[i];
}

// Default export pour compatibilité
export default {
  formatTime,
  formatDuration,
  truncateUrl,
  truncateUrlSimple,
  getEventLabel,
  formatNumber,
  percentage,
  debounce,
  throttle,
  generateId,
  isEmpty,
  deepClone,
  formatBytes
};
