/**
 * SUMO UX Tracker - Sessions Manager
 * Gestion de la liste des sessions (VERSION AMÉLIORÉE)
 */

import db from './db.js';
import importService from './import.js';
import { truncateUrl } from './utils.js';
import foldersManager from './folders-manager.js';

class SessionsManager {
  constructor() {
    this.sessions = [];
    this.filteredSessions = [];
    this.currentSort = 'date-desc';
    this.thumbnailCache = new Map(); // Cache pour les vignettes
    this.sessionsMetadata = new Map(); // Cache pour les métadonnées (titres + dossiers)

    // Mode comparaison
    this.compareMode = false;
    this.selectedSessions = [];

    // Filtre par dossier
    this.currentFolderId = null;

    // Flag pour savoir si les listeners globaux sont attachés
    this.globalListenersAttached = false;
  }

  /**
   * Charge toutes les sessions
   */
  async loadSessions() {
    console.log('[Sessions] Loading sessions...');

    try {
      this.sessions = await db.getAllSessions();

      // Charger les métadonnées (titres + dossiers)
      this.sessionsMetadata = await db.getAllSessionsMetadata();
      console.log('[Sessions] Loaded metadata for', this.sessionsMetadata.size, 'sessions');

      this.filteredSessions = [...this.sessions];

      // Appliquer filtre par dossier si actif
      if (this.currentFolderId) {
        this.filterByFolder(this.currentFolderId);
      }

      // Appliquer le tri actuel
      this.sortSessions(this.currentSort);

      // Afficher
      await this.render();

      // Mettre à jour les compteurs de dossiers
      if (foldersManager) {
        foldersManager.updateFolderCounts();
      }

    } catch (error) {
      console.error('[Sessions] Error loading sessions:', error);
    }
  }

  /**
   * Filtre les sessions par dossier
   */
  filterByFolder(folderId) {
    this.currentFolderId = folderId;

    if (!folderId) {
      // "Toutes les sessions"
      this.filteredSessions = [...this.sessions];
    } else {
      // Filtrer par dossier
      this.filteredSessions = this.sessions.filter(session => {
        const metadata = this.sessionsMetadata.get(session.id);
        return metadata && metadata.folderId === folderId;
      });
    }

    console.log('[Sessions] Filtered by folder:', folderId, '->', this.filteredSessions.length, 'sessions');
    this.render();
  }

  /**
   * Filtre les sessions par URL ou titre
   */
  filterSessions(searchTerm) {
    const term = searchTerm.toLowerCase().trim();

    if (!term) {
      // Réappliquer filtre par dossier si actif
      if (this.currentFolderId) {
        this.filterByFolder(this.currentFolderId);
      } else {
        this.filteredSessions = [...this.sessions];
        this.render();
      }
      return;
    }

    // Recherche dans URL, title original et titre personnalisé
    const baseList = this.currentFolderId
      ? this.sessions.filter(session => {
          const metadata = this.sessionsMetadata.get(session.id);
          return metadata && metadata.folderId === this.currentFolderId;
        })
      : this.sessions;

    this.filteredSessions = baseList.filter(session => {
      const url = session.metadata?.initialUrl?.toLowerCase() || '';
      const title = session.metadata?.initialTitle?.toLowerCase() || '';
      const metadata = this.sessionsMetadata.get(session.id);
      const customTitle = metadata?.customTitle?.toLowerCase() || '';

      return url.includes(term) || title.includes(term) || customTitle.includes(term);
    });

    this.render();
  }

  /**
   * Trie les sessions
   */
  sortSessions(sortBy) {
    this.currentSort = sortBy;
    this.filteredSessions = db.sortSessions(this.filteredSessions, sortBy);
    this.render();
  }

  /**
   * Affiche la liste des sessions
   */
  async render() {
    const container = document.getElementById('sessions-list');
    if (!container) return;

    // Si aucune session
    if (this.filteredSessions.length === 0) {
      container.innerHTML = `
        <div class="sessions-empty">
          <div class="sessions-empty-icon">🔭</div>
          <h3>Aucune session trouvée</h3>
          <p>Importez vos premiers fichiers .uxtrack pour commencer l'analyse</p>
          <button class="btn-primary" onclick="window.sumoApp.showView('import')">
            📥 Importer une session
          </button>
        </div>
      `;
      return;
    }

    // Générer les cards (sans attendre les thumbnails)
    container.innerHTML = this.filteredSessions
      .map(session => this.createSessionCard(session))
      .join('');

    // Attacher les event listeners
    this.attachCardListeners();

    // Charger les thumbnails en arrière-plan
    this.loadThumbnails();
  }

  /**
   * Charge les thumbnails des sessions visibles
   */
  async loadThumbnails() {
    // ✅ Charger tous les thumbnails en parallèle pour de meilleures performances
    const loadPromises = this.filteredSessions.map(async (session) => {
      // Si déjà en cache, utiliser le cache et mettre à jour
      if (this.thumbnailCache.has(session.id)) {
        const cachedDataUrl = this.thumbnailCache.get(session.id);
        this.updateThumbnail(session.id, cachedDataUrl);
        return;
      }

      try {
        // Récupérer le premier screenshot
        const thumbnailDataUrl = await db.getFirstScreenshot(session.id);

        if (thumbnailDataUrl) {
          // Mettre en cache
          this.thumbnailCache.set(session.id, thumbnailDataUrl);

          // Mettre à jour l'affichage
          this.updateThumbnail(session.id, thumbnailDataUrl);
        }
      } catch (error) {
        console.error('[Sessions] Error loading thumbnail for', session.id, error);
      }
    });

    // Attendre que tous les thumbnails soient chargés
    await Promise.all(loadPromises);
  }

  /**
   * Met à jour la vignette d'une session
   */
  updateThumbnail(sessionId, dataUrl) {
    const card = document.querySelector(`.session-card[data-session-id="${sessionId}"]`);
    if (!card) return;

    const thumbnail = card.querySelector('.session-card-thumbnail');
    if (!thumbnail) return;

    // Vérifier si l'image existe déjà pour éviter de re-render
    const existingImg = thumbnail.querySelector('img');
    if (existingImg && existingImg.src === dataUrl) {
      return;
    }

    // Créer l'image
    const img = document.createElement('img');
    img.src = dataUrl;
    img.alt = 'Session thumbnail';

    // Remplacer le placeholder par l'image (en préservant les badges)
    const placeholder = thumbnail.querySelector('.session-card-thumbnail-placeholder');
    if (placeholder) {
      // Insérer l'image avant le placeholder
      thumbnail.insertBefore(img, placeholder);
      // Supprimer le placeholder
      placeholder.remove();
    } else {
      // Si pas de placeholder, l'ajouter au début (avant les badges)
      const badges = thumbnail.querySelector('.session-card-badges');
      if (badges) {
        thumbnail.insertBefore(img, badges);
      } else {
        thumbnail.appendChild(img);
      }
    }
  }

  /**
   * Obtient le badge du type de session
   */
  getSessionTypeBadge(sessionType) {
    const types = {
      'figma-prototype': { label: 'Figma Prototype', color: 'badge-info' },
      'figma-design': { label: 'Figma Design', color: 'badge-info' },
      'figma': { label: 'Figma', color: 'badge-info' },
      'web': { label: 'Web', color: 'badge-secondary' }
    };

    const type = types[sessionType] || types.web;
    return `<span class="badge ${type.color}">🎨 ${type.label}</span>`;
  }

  /**
   * Crée une card de session
   */
  createSessionCard(session) {
    const meta = session.metadata;
    const badges = [];

    // Récupérer les métadonnées (titre personnalisé)
    const sessionMetadata = this.sessionsMetadata.get(session.id);
    const customTitle = sessionMetadata?.customTitle;

    // Détecter le type de session si absent
    const sessionType = meta.sessionType || db.detectSessionType(meta.initialUrl);

    // Badge type de session
    badges.push(this.getSessionTypeBadge(sessionType));

    // Badge rage clicks
    if (meta.rageClicks > 0) {
      badges.push(`<span class="badge badge-danger">⚠️ ${meta.rageClicks} rage clicks</span>`);
    }

    // Badge longue session
    if (meta.duration > 10 * 60 * 1000) {
      badges.push(`<span class="badge badge-success">✓ Session longue</span>`);
    }

    // Titre de la session : priorité au titre personnalisé, sinon initialTitle, sinon domaine
    const sessionTitle = customTitle || meta.initialTitle || this.extractDomain(meta.initialUrl);
    const hasCustomTitle = !!customTitle;

    return `
      <div class="session-card ${this.compareMode ? 'compare-mode' : ''}" data-session-id="${session.id}" draggable="true">
        ${this.compareMode ? `
          <div class="session-checkbox" data-session-id="${session.id}">
            ${this.selectedSessions.includes(session.id) ? '✓' : ''}
          </div>
        ` : ''}
        <div class="session-card-thumbnail">
          <div class="session-card-thumbnail-placeholder">📸</div>
          <div class="session-card-badges">
            ${badges.join('')}
          </div>
        </div>

        <div class="session-card-content">
          <div class="session-card-title ${hasCustomTitle ? 'has-custom-title' : ''}">${sessionTitle}</div>
          <div class="session-card-url">${truncateUrl(meta.initialUrl, 60)}</div>

          <div class="session-card-meta">
            <div class="session-card-stat">
              <div class="session-card-stat-label">Durée</div>
              <div class="session-card-stat-value">${importService.formatDuration(meta.duration)}</div>
            </div>

            <div class="session-card-stat">
              <div class="session-card-stat-label">Interactions</div>
              <div class="session-card-stat-value">${meta.totalEvents}</div>
            </div>

            <div class="session-card-stat">
              <div class="session-card-stat-label">Clics</div>
              <div class="session-card-stat-value">${meta.clicks}</div>
            </div>

            <div class="session-card-stat">
              <div class="session-card-stat-label">Screenshots</div>
              <div class="session-card-stat-value">${meta.screenshots}</div>
            </div>
          </div>

          <div class="session-card-footer">
            <div class="session-card-date">${importService.formatDate(session.importDate)}</div>
            <div class="session-card-actions">
              <button class="session-card-action" data-action="rename" data-session-id="${session.id}" title="Renommer">
                ✏️
              </button>
              <button class="session-card-action" data-action="delete" data-session-id="${session.id}" title="Supprimer">
                🗑️
              </button>
            </div>
          </div>
        </div>
      </div>
    `;
  }

  /**
   * Attache les listeners globaux (une seule fois)
   */
  attachGlobalListeners() {
    if (this.globalListenersAttached) return;

    // Bouton "Comparer 2 sessions"
    const btnCompareMode = document.getElementById('btn-compare-mode');
    if (btnCompareMode) {
      btnCompareMode.addEventListener('click', (e) => {
        e.preventDefault();
        e.stopPropagation();
        console.log('[Sessions] Click on btn-compare-mode');
        this.toggleCompareMode();
      });
    }

    // Bouton "Comparer" (exécuter)
    const btnCompareExecute = document.getElementById('btn-compare-execute');
    if (btnCompareExecute) {
      btnCompareExecute.addEventListener('click', (e) => {
        e.preventDefault();
        e.stopPropagation();
        this.executeComparison();
      });
    }

    // Bouton "Annuler"
    const btnCompareCancel = document.getElementById('btn-compare-cancel');
    if (btnCompareCancel) {
      btnCompareCancel.addEventListener('click', (e) => {
        e.preventDefault();
        e.stopPropagation();
        this.toggleCompareMode();
      });
    }

    this.globalListenersAttached = true;
    console.log('[Sessions] Global listeners attached');
  }

  /**
   * Attache les listeners sur les cards
   */
  attachCardListeners() {
    // Attacher les listeners globaux (une seule fois)
    this.attachGlobalListeners();

    // Checkboxes en mode comparaison
    if (this.compareMode) {
      document.querySelectorAll('.session-checkbox').forEach(checkbox => {
        checkbox.addEventListener('click', (e) => {
          e.stopPropagation();
          const sessionId = checkbox.dataset.sessionId;
          this.selectSessionForComparison(sessionId);
        });
      });
    }

    // Click sur card
    document.querySelectorAll('.session-card').forEach(card => {
      card.addEventListener('click', (e) => {
        // Ignorer si click sur bouton action
        if (e.target.closest('.session-card-action')) return;

        // Ignorer si click sur checkbox
        if (e.target.closest('.session-checkbox')) return;

        const sessionId = card.dataset.sessionId;

        if (this.compareMode) {
          // En mode comparaison, sélectionner la session
          this.selectSessionForComparison(sessionId);
        } else {
          // En mode normal, ouvrir la session
          window.sumoApp.openSession(sessionId);
        }
      });
    });

    // Bouton rename
    document.querySelectorAll('.session-card-action[data-action="rename"]').forEach(btn => {
      btn.addEventListener('click', async (e) => {
        e.stopPropagation();

        const sessionId = btn.dataset.sessionId;
        const session = this.sessions.find(s => s.id === sessionId);
        const sessionMetadata = this.sessionsMetadata.get(sessionId);
        const currentTitle = sessionMetadata?.customTitle || session?.metadata?.initialTitle || '';

        // Ouvrir modal de renommage
        if (foldersManager) {
          foldersManager.openRenameSessionModal(sessionId, currentTitle);
        }
      });
    });

    // Bouton delete
    document.querySelectorAll('.session-card-action[data-action="delete"]').forEach(btn => {
      btn.addEventListener('click', async (e) => {
        e.stopPropagation();

        const sessionId = btn.dataset.sessionId;

        if (confirm('Êtes-vous sûr de vouloir supprimer cette session ?')) {
          await this.deleteSession(sessionId);
        }
      });
    });

    // Drag & drop pour déplacer sessions vers dossiers
    document.querySelectorAll('.session-card[draggable="true"]').forEach(card => {
      card.addEventListener('dragstart', (e) => {
        const sessionId = card.dataset.sessionId;
        e.dataTransfer.effectAllowed = 'move';
        e.dataTransfer.setData('text/plain', sessionId);
        card.classList.add('dragging');

        // Notifier folders manager
        if (foldersManager) {
          foldersManager.startDragSession(sessionId);
        }
      });

      card.addEventListener('dragend', (e) => {
        card.classList.remove('dragging');
      });
    });
  }

  /**
   * Supprime une session
   */
  async deleteSession(sessionId) {
    try {
      console.log('[Sessions] Deleting session:', sessionId);

      await db.deleteSession(sessionId);

      // Supprimer les metadata associées
      await db.deleteSessionMetadata(sessionId);

      // Supprimer du cache
      this.thumbnailCache.delete(sessionId);

      // Recharger la liste
      await this.loadSessions();

      // Mettre à jour les compteurs de dossiers
      if (foldersManager) {
        await foldersManager.updateFolderCounts();
      }

    } catch (error) {
      console.error('[Sessions] Error deleting session:', error);
      alert('Erreur lors de la suppression');
    }
  }

  /**
   * Extrait le domaine d'une URL
   */
  extractDomain(url) {
    if (!url) return 'Session sans titre';
    
    try {
      const urlObj = new URL(url);
      return urlObj.hostname.replace('www.', '');
    } catch {
      return 'Session sans titre';
    }
  }

  // OPTIM #5: Removed duplicated utility function
  // Now using utils.js: truncateUrl

  /**
   * Active/désactive le mode comparaison
   */
  toggleCompareMode() {
    console.log('[Sessions] toggleCompareMode() called, current mode:', this.compareMode);

    this.compareMode = !this.compareMode;
    this.selectedSessions = [];

    console.log('[Sessions] Compare mode NOW:', this.compareMode);

    if (this.compareMode) {
      // Afficher selection bar
      const selectionBar = document.getElementById('comparison-selection-bar');
      if (selectionBar) {
        selectionBar.style.display = 'flex';
        console.log('[Sessions] Selection bar shown');
      }

      // Re-render cards avec checkboxes
      this.render();
    } else {
      // Masquer selection bar
      const selectionBar = document.getElementById('comparison-selection-bar');
      if (selectionBar) {
        selectionBar.style.display = 'none';
        console.log('[Sessions] Selection bar hidden');
      }

      // Re-render sans checkboxes
      this.render();
    }
  }

  /**
   * Sélectionne/désélectionne une session pour comparaison
   */
  selectSessionForComparison(sessionId) {
    const index = this.selectedSessions.indexOf(sessionId);

    if (index > -1) {
      // Désélectionner
      this.selectedSessions.splice(index, 1);
      console.log('[Sessions] Deselected:', sessionId);
    } else if (this.selectedSessions.length < 2) {
      // Sélectionner (max 2)
      this.selectedSessions.push(sessionId);
      console.log('[Sessions] Selected:', sessionId);
    } else {
      // Déjà 2 sélectionnées, remplacer la première
      this.selectedSessions.shift();
      this.selectedSessions.push(sessionId);
      console.log('[Sessions] Replaced selection:', sessionId);
    }

    // Mettre à jour l'UI
    this.updateCompareSelection();
  }

  /**
   * Met à jour l'affichage de la sélection
   */
  updateCompareSelection() {
    // Update count
    const countEl = document.getElementById('comparison-selection-count');
    if (countEl) {
      countEl.textContent = `${this.selectedSessions.length}/2`;
    }

    // Update checkboxes
    document.querySelectorAll('.session-checkbox').forEach(checkbox => {
      const sessionId = checkbox.dataset.sessionId;
      const isSelected = this.selectedSessions.includes(sessionId);
      checkbox.classList.toggle('selected', isSelected);
      checkbox.innerHTML = isSelected ? '✓' : '';
    });

    // Enable/disable compare button
    const compareBtn = document.getElementById('btn-compare-execute');
    if (compareBtn) {
      compareBtn.disabled = this.selectedSessions.length !== 2;
    }
  }

  /**
   * Exécute la comparaison
   */
  executeComparison() {
    if (this.selectedSessions.length !== 2) {
      alert('Veuillez sélectionner 2 sessions');
      return;
    }

    const [sessionA, sessionB] = this.selectedSessions;
    console.log('[Sessions] Comparing:', sessionA, 'vs', sessionB);

    // Ouvrir vue comparaison
    window.sumoApp.openComparison(sessionA, sessionB);

    // Reset mode comparaison
    this.toggleCompareMode();
  }

  /**
   * Obtient les stats globales
   */
  getGlobalStats() {
    if (this.sessions.length === 0) {
      return null;
    }

    const totalDuration = this.sessions.reduce((sum, s) => sum + s.metadata.duration, 0);
    const totalClicks = this.sessions.reduce((sum, s) => sum + s.metadata.clicks, 0);
    const totalRageClicks = this.sessions.reduce((sum, s) => sum + s.metadata.rageClicks, 0);
    
    return {
      totalSessions: this.sessions.length,
      totalDuration: totalDuration,
      avgDuration: totalDuration / this.sessions.length,
      totalClicks: totalClicks,
      totalRageClicks: totalRageClicks
    };
  }
}

// Export singleton
const sessionsManager = new SessionsManager();
export default sessionsManager;
