/**
 * SUMO UX Tracker - Session Viewer
 * Page dédiée pour visualiser une session dans une nouvelle fenêtre
 */

import db from './db.js';
import replayManager from './replay.js';
import heatmapManager from './heatmap.js';
import analyticsManager from './analytics.js';
import feedbackViewer from './feedback-viewer.js';
import dataAnonymizer from './data-anonymizer.js';
import aiAnalyzer from './ai-analyzer.js';

class SessionViewer {
  constructor() {
    this.currentSession = null;
    this.sessionId = null;
    this.elements = {};
  }

  /**
   * Initialise le viewer
   */
  async init() {
    console.log('[SessionViewer] Initializing...');

    try {
      // Récupérer le sessionId depuis l'URL
      const urlParams = new URLSearchParams(window.location.search);
      this.sessionId = urlParams.get('id');

      if (!this.sessionId) {
        throw new Error('No session ID provided');
      }

      // Initialiser la base de données
      await db.init();
      console.log('[SessionViewer] Database initialized');

      // Récupérer les éléments DOM
      this.cacheElements();

      // Attacher les event listeners
      this.attachEventListeners();

      // Charger la session
      await this.loadSession();

      console.log('[SessionViewer] Initialization complete');
    } catch (error) {
      console.error('[SessionViewer] Initialization error:', error);
      this.showError('Erreur lors du chargement de la session');
    }
  }

  /**
   * Cache les éléments DOM
   */
  cacheElements() {
    this.elements.sessionTitle = document.getElementById('session-title');
    this.elements.analysisTitle = document.getElementById('analysis-title');
    this.elements.btnBackToMain = document.getElementById('btn-back-to-main');
    this.elements.btnCloseWindow = document.getElementById('btn-close-window');
    this.elements.tabBtns = document.querySelectorAll('.tab-btn');
    this.elements.tabContents = document.querySelectorAll('.tab-content');

    console.log('[SessionViewer] Elements cached');
  }

  /**
   * Attache les event listeners
   */
  attachEventListeners() {
    // Bouton retour à la liste
    this.elements.btnBackToMain?.addEventListener('click', () => {
      window.close();
      // Si la fenêtre ne se ferme pas (ouverture directe), rediriger
      setTimeout(() => {
        window.location.href = 'index.html';
      }, 100);
    });

    // Bouton fermer fenêtre
    this.elements.btnCloseWindow?.addEventListener('click', () => {
      window.close();
    });

    // Tab buttons
    this.elements.tabBtns.forEach(btn => {
      btn.addEventListener('click', () => {
        const tabName = btn.dataset.tab;
        this.switchTab(tabName);
      });
    });

    console.log('[SessionViewer] Event listeners attached');
  }

  /**
   * Charge la session
   */
  async loadSession() {
    console.log('[SessionViewer] Loading session:', this.sessionId);

    try {
      // Récupérer la session avec lazy loading des screenshots
      const session = await db.getSessionWithScreenshotsLazy(this.sessionId);
      if (!session) {
        throw new Error('Session not found');
      }

      this.currentSession = session;

      // Mettre à jour le titre
      const url = session.metadata?.initialUrl || 'Session';
      this.elements.sessionTitle.textContent = `🎯 SUMO UX Tracker - ${url}`;
      this.elements.analysisTitle.textContent = `Session - ${url}`;

      // Initialiser le replay par défaut
      this.switchTab('replay');

      console.log('[SessionViewer] Session loaded successfully');
    } catch (error) {
      console.error('[SessionViewer] Error loading session:', error);
      this.showError('Erreur lors du chargement de la session');
    }
  }

  /**
   * Change d'onglet dans l'analyse
   */
  switchTab(tabName) {
    console.log('[SessionViewer] Switching to tab:', tabName);

    // Mettre à jour les boutons
    this.elements.tabBtns.forEach(btn => {
      if (btn.dataset.tab === tabName) {
        btn.classList.add('active');
      } else {
        btn.classList.remove('active');
      }
    });

    // Mettre à jour les contenus
    this.elements.tabContents.forEach(content => {
      if (content.id === `tab-${tabName}`) {
        content.classList.add('active');
      } else {
        content.classList.remove('active');
      }
    });

    // Initialiser le contenu de l'onglet
    if (this.currentSession) {
      switch (tabName) {
        case 'replay':
          replayManager.init(this.currentSession);
          break;
        case 'heatmap':
          heatmapManager.init(this.currentSession);
          break;
        case 'analytics':
          analyticsManager.init(this.currentSession);
          break;
        case 'feedback':
          feedbackViewer.init(this.currentSession);
          break;
      }
    }
  }

  /**
   * Affiche une erreur
   */
  showError(message) {
    console.error('[SessionViewer] Error:', message);
    alert(message);
  }
}

// Initialiser le viewer au chargement
const viewer = new SessionViewer();

document.addEventListener('DOMContentLoaded', () => {
  viewer.init();
});

// Exposer globalement pour les autres modules
window.sessionViewer = viewer;
window.dataAnonymizer = dataAnonymizer;
window.aiAnalyzer = aiAnalyzer;

export default viewer;
