/**
 * SUMO UX Tracker - Replay Manager
 * Lecteur de replay de session (version avec trail souris)
 */

import db from './db.js';
import { formatTime, formatDuration, truncateUrl, getEventLabel } from './utils.js';
import pdfExport from './pdfExport.js';

class ReplayManager {
  constructor() {
    this.session = null;
    this.currentTime = 0;
    this.duration = 0;
    this.isPlaying = false;
    this.playbackSpeed = 1;
    this.screenshots = [];
    this.currentScreenshotIndex = 0;

    // requestAnimationFrame properties
    this.animationFrameId = null;
    this.lastFrameTime = 0;

    // Canvas pour les overlays
    this.overlayCanvas = null;
    this.overlayCtx = null;

    // Contrôles overlay
    this.overlays = {
      clicks: { enabled: true, opacity: 0.7, radius: 40 },
      mouseMoves: { enabled: false, opacity: 0.7 },
      scroll: { enabled: true, opacity: 0.7 }
    };
    this.isFullscreen = false;
  }

  /**
   * Initialise le replay
   */
  async init(session) {
    console.log('[Replay] Initializing replay for session:', session.id);

    this.session = session;
    this.currentTime = 0;
    this.duration = session.metadata?.duration || session.duration || 0;
    this.isPlaying = false;
    this.currentScreenshotIndex = -1; // ✅ Forcer à -1 pour que le premier screenshot se charge

    // Charger les screenshots
    this.loadScreenshots();

    // Afficher l'interface
    this.render();

    // Afficher le premier screenshot
    await this.updateDisplay();
  }

  /**
   * Charge tous les screenshots depuis la session
   */
  loadScreenshots() {
    this.screenshots = [];
    
    if (!this.session.screenshots || this.session.screenshots.length === 0) {
      console.warn('[Replay] No screenshots in session');
      return;
    }

    this.screenshots = this.session.screenshots.map((screenshot, index) => ({
      id: screenshot.filename || `screenshot_${index}`,
      timestamp: screenshot.timestamp,
      dataUrl: screenshot.dataUrl,
      type: screenshot.type
    }));
    
    // Trier par timestamp
    this.screenshots.sort((a, b) => a.timestamp - b.timestamp);
    
    console.log('[Replay] Screenshots loaded:', this.screenshots.length);
  }

  /**
   * Affiche l'interface de replay
   */
  render() {
    const container = document.getElementById('replay-container');
    if (!container) return;

    const startTime = this.session.metadata?.startTime || this.session.startTime || 0;
    const url = this.session.metadata?.initialUrl || this.session.url || 'URL inconnue';
    const clicks = this.session.metadata?.clicks || this.session.clicks || 0;

    container.innerHTML = `
      <div class="replay-player">
        <div class="replay-viewport" id="replay-viewport">
          <img class="replay-screenshot" id="replay-screenshot" alt="Screenshot">
          <canvas class="replay-overlay" id="replay-overlay-canvas"></canvas>
          <div class="replay-overlay" id="replay-overlay">
            <div class="replay-cursor" id="replay-cursor" style="display: none;"></div>
          </div>
        </div>
        
        <div class="replay-controls">
          <div class="replay-timeline">
            <div class="replay-timeline-track" id="replay-timeline">
              <div class="replay-timeline-progress" id="replay-progress">
                <div class="replay-timeline-handle"></div>
              </div>
              <div class="replay-timeline-events" id="replay-events"></div>
            </div>
          </div>
          
          <div class="replay-time-display">
            <span id="replay-current-time">0:00</span>
            <span>/</span>
            <span id="replay-total-time">${formatTime(this.duration)}</span>
          </div>
          
          <div class="replay-controls-buttons">
            <button class="replay-btn" id="replay-prev" title="Précédent">◀️</button>
            <button class="replay-btn primary" id="replay-play" title="Play">▶️</button>
            <button class="replay-btn" id="replay-next" title="Suivant">▶️</button>
            <button class="replay-btn" id="replay-enlarge" title="Agrandir">🖼️</button>
            <button class="replay-btn" id="replay-export-pdf" title="Export PDF">📄</button>

            <div class="replay-speed">
              <button class="replay-speed-btn ${this.playbackSpeed === 0.5 ? 'active' : ''}" data-speed="0.5">0.5x</button>
              <button class="replay-speed-btn ${this.playbackSpeed === 1 ? 'active' : ''}" data-speed="1">1x</button>
              <button class="replay-speed-btn ${this.playbackSpeed === 2 ? 'active' : ''}" data-speed="2">2x</button>
              <button class="replay-speed-btn ${this.playbackSpeed === 4 ? 'active' : ''}" data-speed="4">4x</button>
            </div>
          </div>
        </div>
      </div>
      
      <div class="replay-sidebar">
        <!-- Contrôles Overlay -->
        <div class="replay-control-section">
          <h3>🎨 Overlays</h3>
          
          <div class="replay-layer-toggle" data-layer="clicks">
            <div class="replay-layer-label">
              <span class="replay-layer-icon" style="color: #ef4444;">🔴</span>
              <span>Clics (${clicks})</span>
            </div>
            <div class="replay-layer-checkbox ${this.overlays.clicks.enabled ? 'checked' : ''}"></div>
          </div>
          
          <div class="replay-layer-toggle" data-layer="moves">
            <div class="replay-layer-label">
              <span class="replay-layer-icon" style="color: #3b82f6;">🔵</span>
              <span>Mouvements souris (${this.session.metadata?.mouseMoves || 0})</span>
            </div>
            <div class="replay-layer-checkbox ${this.overlays.mouseMoves.enabled ? 'checked' : ''}"></div>
          </div>
          
          <div class="replay-layer-toggle" data-layer="scroll">
            <div class="replay-layer-label">
              <span class="replay-layer-icon" style="color: #8b5cf6;">🟣</span>
              <span>Scroll (${this.session.metadata?.scrolls || 0})</span>
            </div>
            <div class="replay-layer-checkbox ${this.overlays.scroll.enabled ? 'checked' : ''}"></div>
          </div>
        </div>
        
        <!-- Paramètres -->
        <div class="replay-control-section">
          <h3>⚙️ Paramètres</h3>
          
          <div class="replay-slider">
            <div class="replay-slider-label">
              <span>Opacité</span>
              <span class="replay-slider-value" id="replay-opacity-value">${Math.round(this.overlays.clicks.opacity * 100)}%</span>
            </div>
            <input type="range" class="replay-slider-input" id="replay-opacity-slider" 
                   min="0" max="100" value="${this.overlays.clicks.opacity * 100}">
          </div>
          
          <div class="replay-slider">
            <div class="replay-slider-label">
              <span>Rayon</span>
              <span class="replay-slider-value" id="replay-radius-value">${this.overlays.clicks.radius}px</span>
            </div>
            <input type="range" class="replay-slider-input" id="replay-radius-slider" 
                   min="20" max="80" value="${this.overlays.clicks.radius}">
          </div>
        </div>
        
        <!-- Informations -->
        <div class="replay-info">
          <h3>📊 Informations</h3>
          <div class="replay-info-item">
            <span class="replay-info-label">URL</span>
            <span class="replay-info-value">${truncateUrl(url, 40)}</span>
          </div>
          <div class="replay-info-item">
            <span class="replay-info-label">Durée</span>
            <span class="replay-info-value">${formatDuration(this.duration)}</span>
          </div>
          <div class="replay-info-item">
            <span class="replay-info-label">Screenshots</span>
            <span class="replay-info-value">${this.screenshots.length}</span>
          </div>
        </div>
        
        <!-- Plein écran -->
        <div class="replay-fullscreen">
          <button class="replay-fullscreen-btn" id="replay-fullscreen">
            🖥️ Mode plein écran
          </button>
        </div>
        
        <!-- Événements -->
        <div class="replay-events">
          <h3>📋 Événements clés</h3>
          <div id="replay-events-list">
            ${this.renderEventsList()}
          </div>
        </div>
      </div>
    `;

    this.attachListeners();
    this.renderTimelineEvents();
  }

  /**
   * Affiche la liste des événements
   */
  renderEventsList() {
    const events = this.session.events || this.session.data?.events || [];
    const startTime = this.session.metadata?.startTime || this.session.startTime || 0;
    
    const keyEvents = events.filter(e => 
      e.type === 'rage_click' || 
      e.type === 'navigation' || 
      e.type === 'page_start'
    );

    if (keyEvents.length === 0) {
      return '<p style="color: var(--text-secondary); font-size: 0.875rem;">Aucun événement clé</p>';
    }

    return keyEvents.map(event => {
      const time = event.timestamp - startTime;
      return `
        <div class="replay-event" data-time="${time}">
          <div class="replay-event-time">${formatTime(time)}</div>
          <div class="replay-event-type">${getEventLabel(event.type)}</div>
        </div>
      `;
    }).join('');
  }

  /**
   * Attache les listeners
   */
  attachListeners() {
    // Play/pause
    const playBtn = document.getElementById('replay-play');
    playBtn?.addEventListener('click', () => {
      if (this.isPlaying) {
        this.pause();
      } else {
        this.play();
      }
    });

    // Prev/Next
    document.getElementById('replay-prev')?.addEventListener('click', () => {
      this.previousEvent();
    });

    document.getElementById('replay-next')?.addEventListener('click', () => {
      this.nextEvent();
    });

    // Speed
    document.querySelectorAll('.replay-speed-btn').forEach(btn => {
      btn.addEventListener('click', () => {
        const speed = parseFloat(btn.dataset.speed);
        this.setSpeed(speed);
        
        // Update active state
        document.querySelectorAll('.replay-speed-btn').forEach(b => {
          b.classList.toggle('active', b === btn);
        });
      });
    });

    // Timeline
    const timeline = document.getElementById('replay-timeline');
    timeline?.addEventListener('click', (e) => {
      const rect = timeline.getBoundingClientRect();
      const percent = (e.clientX - rect.left) / rect.width;
      this.seek(percent * this.duration);
    });

    // Layer toggles
    document.querySelectorAll('.replay-layer-toggle').forEach(toggle => {
      toggle.addEventListener('click', () => {
        const layer = toggle.dataset.layer;
        
        if (layer === 'clicks') {
          this.overlays.clicks.enabled = !this.overlays.clicks.enabled;
        } else if (layer === 'moves') {
          this.overlays.mouseMoves.enabled = !this.overlays.mouseMoves.enabled;
        } else if (layer === 'scroll') {
          this.overlays.scroll.enabled = !this.overlays.scroll.enabled;
        }
        
        const checkbox = toggle.querySelector('.replay-layer-checkbox');
        checkbox.classList.toggle('checked');
        
        this.updateDisplay();
      });
    });

    // Sliders
    const opacitySlider = document.getElementById('replay-opacity-slider');
    opacitySlider?.addEventListener('input', (e) => {
      const value = parseFloat(e.target.value) / 100;
      this.overlays.clicks.opacity = value;
      this.overlays.mouseMoves.opacity = value;
      this.overlays.scroll.opacity = value;
      
      document.getElementById('replay-opacity-value').textContent = Math.round(value * 100) + '%';
      this.updateDisplay();
    });

    const radiusSlider = document.getElementById('replay-radius-slider');
    radiusSlider?.addEventListener('input', (e) => {
      const value = parseFloat(e.target.value);
      this.overlays.clicks.radius = value;
      
      document.getElementById('replay-radius-value').textContent = value + 'px';
      this.updateDisplay();
    });

    // Event list
    document.querySelectorAll('.replay-event').forEach(eventEl => {
      eventEl.addEventListener('click', () => {
        const time = parseFloat(eventEl.dataset.time);
        this.seek(time);
      });
    });

    // Fullscreen
    document.getElementById('replay-fullscreen')?.addEventListener('click', () => {
      this.toggleFullscreen();
    });

    // Enlarge
    document.getElementById('replay-enlarge')?.addEventListener('click', () => {
      this.toggleEnlarge();
    });

    // Export PDF
    document.getElementById('replay-export-pdf')?.addEventListener('click', () => {
      this.exportToPDF();
    });
  }

  /**
   * Affiche les événements sur la timeline
   */
  renderTimelineEvents() {
    const container = document.getElementById('replay-events');
    if (!container) return;

    const events = this.session.events || this.session.data?.events || [];
    const startTime = this.session.metadata?.startTime || this.session.startTime || 0;

    container.innerHTML = events
      .filter(e => 
        e.type === 'click' || 
        e.type === 'rage_click' || 
        e.type === 'navigation' || 
        e.type === 'page_start'
      )
      .map(event => {
        const time = event.timestamp - startTime;
        const percent = (time / this.duration) * 100;
        
        let className = 'replay-timeline-event';
        if (event.type === 'click') className += ' click';
        if (event.type === 'rage_click') className += ' rage-click';
        if (event.type === 'navigation') className += ' navigation';
        if (event.type === 'page_start') className += ' navigation';

        return `<div class="${className}" style="left: ${percent}%"></div>`;
      })
      .join('');
  }

  /**
   * Lance la lecture
   */
  play() {
    if (this.isPlaying) return;

    this.isPlaying = true;
    const playBtn = document.getElementById('replay-play');
    if (playBtn) playBtn.innerHTML = '⏸️';

    // Initialiser le timestamp pour requestAnimationFrame
    this.lastFrameTime = performance.now();

    // Démarrer la boucle d'animation
    this.animationLoop();
  }

  /**
   * Boucle d'animation avec requestAnimationFrame
   * @private
   */
  animationLoop() {
    if (!this.isPlaying) return;

    const now = performance.now();
    const deltaTime = now - this.lastFrameTime;
    this.lastFrameTime = now;

    // Avancer le temps en fonction du delta (60 FPS smooth)
    this.currentTime += deltaTime * this.playbackSpeed;

    // Vérifier si on a atteint la fin
    if (this.currentTime >= this.duration) {
      this.currentTime = this.duration;
      this.pause();
      this.updateDisplay(); // Dernier update avant pause
      return;
    }

    // Mettre à jour l'affichage
    this.updateDisplay();

    // Planifier la prochaine frame
    this.animationFrameId = requestAnimationFrame(() => this.animationLoop());
  }

  /**
   * Met en pause
   */
  pause() {
    if (!this.isPlaying) return;

    this.isPlaying = false;
    const playBtn = document.getElementById('replay-play');
    if (playBtn) playBtn.innerHTML = '▶️';

    // Annuler requestAnimationFrame au lieu de clearInterval
    if (this.animationFrameId) {
      cancelAnimationFrame(this.animationFrameId);
      this.animationFrameId = null;
    }
  }

  /**
   * Change la vitesse
   */
  setSpeed(speed) {
    this.playbackSpeed = speed;
    
    if (this.isPlaying) {
      this.pause();
      this.play();
    }
  }

  /**
   * Cherche un temps spécifique
   */
  seek(time) {
    this.currentTime = Math.max(0, Math.min(time, this.duration));
    this.updateDisplay();
  }

  /**
   * Va à l'événement précédent
   */
  previousEvent() {
    const events = this.session.events || this.session.data?.events || [];
    const startTime = this.session.metadata?.startTime || this.session.startTime || 0;
    
    const prevEvent = events
      .filter(e => (e.timestamp - startTime) < this.currentTime)
      .sort((a, b) => b.timestamp - a.timestamp)[0];
    
    if (prevEvent) {
      this.seek(prevEvent.timestamp - startTime);
    }
  }

  /**
   * Va à l'événement suivant
   */
  nextEvent() {
    const events = this.session.events || this.session.data?.events || [];
    const startTime = this.session.metadata?.startTime || this.session.startTime || 0;
    
    const nextEvent = events
      .filter(e => (e.timestamp - startTime) > this.currentTime)
      .sort((a, b) => a.timestamp - b.timestamp)[0];
    
    if (nextEvent) {
      this.seek(nextEvent.timestamp - startTime);
    }
  }

  /**
   * Met à jour l'affichage
   */
  updateDisplay() {
    // Mettre à jour le temps
    const currentTimeEl = document.getElementById('replay-current-time');
    if (currentTimeEl) {
      currentTimeEl.textContent = formatTime(this.currentTime);
    }

    // Mettre à jour la progress bar
    const progress = (this.currentTime / this.duration) * 100;
    const progressEl = document.getElementById('replay-progress');
    if (progressEl) {
      progressEl.style.width = `${progress}%`;
    }

    // Mettre à jour le screenshot
    this.updateScreenshot();

    // Mettre à jour les overlays
    this.updateOverlays();
  }

  /**
   * Met à jour le screenshot affiché (OPTIM #4: lazy loading)
   */
  async updateScreenshot() {
    if (this.screenshots.length === 0) return;

    const startTime = this.session.metadata?.startTime || this.session.startTime || 0;
    const absoluteTime = startTime + this.currentTime;

    // Trouver le screenshot correspondant
    let screenshotIndex = 0;
    for (let i = 0; i < this.screenshots.length; i++) {
      if (this.screenshots[i].timestamp <= absoluteTime) {
        screenshotIndex = i;
      } else {
        break;
      }
    }

    if (screenshotIndex !== this.currentScreenshotIndex) {
      this.currentScreenshotIndex = screenshotIndex;
      const screenshot = this.screenshots[screenshotIndex];

      const screenshotEl = document.getElementById('replay-screenshot');
      if (!screenshotEl) return;

      // OPTIM #4: Lazy loading - charger le screenshot à la demande
      if (!screenshot.loaded && !screenshot.dataUrl) {
        try {
          const dataUrl = await db.loadScreenshotDataUrl(
            this.session.id,
            screenshot.id
          );

          // Update screenshot object with dataUrl
          screenshot.dataUrl = dataUrl;
          screenshot.loaded = true;

          console.log(`[Replay] Lazy loaded screenshot ${screenshot.id}`);
        } catch (error) {
          console.error('[Replay] Error loading screenshot:', error);
          return;
        }
      }

      // Display screenshot
      if (screenshot.dataUrl) {
        screenshotEl.src = screenshot.dataUrl;

        // Setup canvas après chargement de l'image
        screenshotEl.onload = () => {
          this.setupOverlayCanvas();
        };
      }
    }
  }

  /**
   * Configure le canvas d'overlay
   */
  setupOverlayCanvas() {
    const canvas = document.getElementById('replay-overlay-canvas');
    const screenshotEl = document.getElementById('replay-screenshot');
    
    if (!canvas || !screenshotEl || !screenshotEl.complete) return;
    
    const viewport = document.getElementById('replay-viewport');
    if (!viewport) return;
    
    const rect = viewport.getBoundingClientRect();
    
    canvas.width = rect.width;
    canvas.height = rect.height;
    
    this.overlayCanvas = canvas;
    this.overlayCtx = canvas.getContext('2d');
    
    this.updateOverlays();
  }

  /**
   * Met à jour les overlays (canvas)
   */
  updateOverlays() {
    if (!this.overlayCanvas || !this.overlayCtx) {
      this.setupOverlayCanvas();
      return;
    }
    
    // Effacer le canvas
    this.overlayCtx.clearRect(0, 0, this.overlayCanvas.width, this.overlayCanvas.height);
    
    const screenshotEl = document.getElementById('replay-screenshot');
    if (!screenshotEl || !screenshotEl.complete) return;
    
    // Calculer les dimensions d'affichage
    const viewport = this.getViewportDimensions();
    
    // Filtrer les événements jusqu'au temps actuel
    const startTime = this.session.metadata?.startTime || this.session.startTime || 0;
    const absoluteTime = startTime + this.currentTime;
    const events = (this.session.events || this.session.data?.events || [])
      .filter(e => e.timestamp <= absoluteTime);
    
    // Dessiner les overlays
    if (this.overlays.mouseMoves.enabled) {
      this.drawMouseMoves(absoluteTime, viewport);
    }
    
    if (this.overlays.clicks.enabled) {
      this.drawClicks(events, viewport);
    }
    
    if (this.overlays.scroll.enabled) {
      this.drawScrolls(events, viewport);
    }
  }

  /**
   * Calcule les dimensions du viewport
   */
  getViewportDimensions() {
    const screenshotEl = document.getElementById('replay-screenshot');
    const viewport = document.getElementById('replay-viewport');
    
    if (!screenshotEl || !viewport) return null;
    
    const imgNaturalWidth = screenshotEl.naturalWidth;
    const imgNaturalHeight = screenshotEl.naturalHeight;
    const imgNaturalRatio = imgNaturalWidth / imgNaturalHeight;
    
    const containerRect = viewport.getBoundingClientRect();
    const containerWidth = containerRect.width;
    const containerHeight = containerRect.height;
    const containerRatio = containerWidth / containerHeight;
    
    let displayedWidth, displayedHeight, offsetX, offsetY;
    
    if (imgNaturalRatio > containerRatio) {
      displayedWidth = containerWidth;
      displayedHeight = containerWidth / imgNaturalRatio;
      offsetX = 0;
      offsetY = (containerHeight - displayedHeight) / 2;
    } else {
      displayedWidth = containerHeight * imgNaturalRatio;
      displayedHeight = containerHeight;
      offsetX = (containerWidth - displayedWidth) / 2;
      offsetY = 0;
    }
    
    return {
      width: imgNaturalWidth,
      height: imgNaturalHeight,
      displayedWidth,
      displayedHeight,
      offsetX,
      offsetY
    };
  }

  /**
   * Dessine les mouvements de souris (TRAIL comme Hotjar)
   */
  drawMouseMoves(currentTime, viewport) {
    if (!this.overlays.mouseMoves.enabled || !viewport) return;

    const ctx = this.overlayCtx;
    const opacity = this.overlays.mouseMoves.opacity;

    // Filtrer les mouvements jusqu'au temps actuel
    const moves = (this.session.data?.events || [])
      .filter(e => e.type === 'mouse_move' && e.timestamp <= currentTime)
      .slice(-100); // Garder les 100 derniers mouvements

    if (moves.length < 2) return;

    // Dessiner le trail
    ctx.strokeStyle = `rgba(59, 130, 246, ${opacity})`;
    ctx.lineWidth = 3;
    ctx.lineCap = 'round';
    ctx.lineJoin = 'round';

    ctx.beginPath();
    moves.forEach((move, index) => {
      const x = viewport.offsetX + (move.x / viewport.width) * viewport.displayedWidth;
      const y = viewport.offsetY + (move.y / viewport.height) * viewport.displayedHeight;

      if (index === 0) {
        ctx.moveTo(x, y);
      } else {
        ctx.lineTo(x, y);
      }
    });
    ctx.stroke();

    // Dessiner un point au dernier emplacement
    if (moves.length > 0) {
      const lastMove = moves[moves.length - 1];
      const x = viewport.offsetX + (lastMove.x / viewport.width) * viewport.displayedWidth;
      const y = viewport.offsetY + (lastMove.y / viewport.height) * viewport.displayedHeight;

      ctx.fillStyle = `rgba(59, 130, 246, ${opacity})`;
      ctx.beginPath();
      ctx.arc(x, y, 4, 0, Math.PI * 2);
      ctx.fill();
    }
  }

  /**
   * Dessine les clics
   */
  drawClicks(events, viewport) {
    if (!viewport) return;
    
    const ctx = this.overlayCtx;
    const clicks = events.filter(e => e.type === 'click' || e.type === 'rage_click');
    
    clicks.forEach(click => {
      const x = viewport.offsetX + (click.x / viewport.width) * viewport.displayedWidth;
      const y = viewport.offsetY + (click.y / viewport.height) * viewport.displayedHeight;
      
      const isRage = click.type === 'rage_click';
      const radius = this.overlays.clicks.radius / 2;
      
      // Dessiner le cercle
      ctx.fillStyle = isRage 
        ? `rgba(255, 149, 0, ${this.overlays.clicks.opacity * 0.6})`
        : `rgba(239, 68, 68, ${this.overlays.clicks.opacity * 0.5})`;
      ctx.strokeStyle = isRage
        ? `rgba(255, 149, 0, 1)`
        : `rgba(239, 68, 68, 0.9)`;
      ctx.lineWidth = isRage ? 3 : 2;
      
      ctx.beginPath();
      ctx.arc(x, y, radius, 0, Math.PI * 2);
      ctx.fill();
      ctx.stroke();
    });
  }

  /**
   * Dessine les scrolls - Style Hotjar avec zones de pourcentage
   */
  drawScrolls(events, viewport) {
    if (!viewport) return;

    const ctx = this.overlayCtx;
    const scrolls = events.filter(e => e.type === 'scroll');
    const opacity = this.overlays.scroll.opacity;

    // Style Hotjar: diviser en zones et calculer le % d'atteinte
    const numZones = 10;
    const zoneHeight = viewport.displayedHeight / numZones;
    const maxScrollPercentages = [];

    // Collecter les pourcentages de scroll
    scrolls.forEach(scroll => {
      const scrollPercentage = scroll.position?.percentage ?? scroll.scrollPercentage;
      if (scrollPercentage !== undefined) {
        maxScrollPercentages.push(scrollPercentage);
      } else {
        const scrollY = scroll.position?.y ?? scroll.y ?? scroll.scrollY;
        if (scrollY !== undefined) {
          const estimatedPercent = Math.min((scrollY / 5000) * 100, 100);
          maxScrollPercentages.push(estimatedPercent);
        }
      }
    });

    if (maxScrollPercentages.length === 0) return;

    const maxReached = Math.max(...maxScrollPercentages, 0);

    // Dessiner les zones avec gradient de couleur
    for (let i = 0; i < numZones; i++) {
      const zoneStart = i * 10;
      const zoneY = viewport.offsetY + (i * zoneHeight);

      const reachedCount = maxScrollPercentages.filter(p => p >= zoneStart).length;
      const percentage = Math.round((reachedCount / maxScrollPercentages.length) * 100);

      // Couleur basée sur le pourcentage
      let r, g, b;
      if (percentage >= 50) {
        const t = (percentage - 50) / 50;
        r = Math.round(255 * (1 - t));
        g = 200;
        b = 50;
      } else {
        const t = percentage / 50;
        r = 255;
        g = Math.round(200 * t);
        b = 50;
      }

      // Zone colorée
      ctx.fillStyle = `rgba(${r}, ${g}, ${b}, ${0.3 * opacity})`;
      ctx.fillRect(viewport.offsetX, zoneY, viewport.displayedWidth, zoneHeight);

      // Ligne de séparation
      ctx.fillStyle = `rgba(${r}, ${g}, ${b}, ${0.6 * opacity})`;
      ctx.fillRect(viewport.offsetX, zoneY + zoneHeight - 2, viewport.displayedWidth, 2);

      // Pourcentage à gauche
      ctx.font = `bold ${12 * opacity}px sans-serif`;
      const text = `${percentage}%`;
      const textWidth = ctx.measureText(text).width;
      const textX = viewport.offsetX + 10;
      const textY = zoneY + zoneHeight / 2;

      ctx.fillStyle = `rgba(0, 0, 0, ${0.7 * opacity})`;
      ctx.fillRect(textX - 4, textY - 8, textWidth + 12, 16);

      ctx.fillStyle = `rgba(255, 255, 255, ${0.95 * opacity})`;
      ctx.textAlign = 'left';
      ctx.textBaseline = 'middle';
      ctx.fillText(text, textX, textY);
    }

    // Ligne du max scroll
    if (maxReached > 0) {
      const maxY = viewport.offsetY + (maxReached / 100) * viewport.displayedHeight;
      ctx.strokeStyle = `rgba(255, 255, 255, ${0.8 * opacity})`;
      ctx.lineWidth = 2;
      ctx.setLineDash([8, 4]);
      ctx.beginPath();
      ctx.moveTo(viewport.offsetX, maxY);
      ctx.lineTo(viewport.offsetX + viewport.displayedWidth, maxY);
      ctx.stroke();
      ctx.setLineDash([]);
    }
  }

  /**
   * Toggle plein écran
   */
  toggleFullscreen() {
    const viewport = document.getElementById('replay-viewport');
    
    if (!document.fullscreenElement) {
      viewport.requestFullscreen();
      this.isFullscreen = true;
      
      // Ajouter contrôles fullscreen
      this.addFullscreenControls();
    } else {
      document.exitFullscreen();
      this.isFullscreen = false;
    }
  }

  /**
   * Ajoute les contrôles en mode fullscreen
   */
  addFullscreenControls() {
    const viewport = document.getElementById('replay-viewport');
    
    const controls = document.createElement('div');
    controls.id = 'fullscreen-controls';
    controls.className = 'fullscreen-controls';
    controls.innerHTML = `
      <div class="fullscreen-timeline">
        <div class="fullscreen-time-display">
          <span id="fs-current-time">${formatTime(this.currentTime)}</span>
          <span>/</span>
          <span>${formatTime(this.duration)}</span>
        </div>
        
        <div class="fullscreen-timeline-track" id="fs-timeline">
          <div class="fullscreen-timeline-progress" id="fs-progress" style="width: ${(this.currentTime / this.duration) * 100}%">
            <div class="fullscreen-timeline-handle"></div>
          </div>
        </div>
        
        <div class="fullscreen-controls-buttons">
          <button class="fullscreen-btn" id="fs-prev">◀️</button>
          <button class="fullscreen-btn primary" id="fs-play">${this.isPlaying ? '⏸️' : '▶️'}</button>
          <button class="fullscreen-btn" id="fs-next">▶️</button>
          <button class="fullscreen-btn" id="fs-exit">✕</button>
        </div>
      </div>
    `;
    
    viewport.appendChild(controls);
    
    // Attacher les listeners
    document.getElementById('fs-play')?.addEventListener('click', () => {
      if (this.isPlaying) this.pause();
      else this.play();
      
      document.getElementById('fs-play').innerHTML = this.isPlaying ? '⏸️' : '▶️';
    });
    
    document.getElementById('fs-prev')?.addEventListener('click', () => this.previousEvent());
    document.getElementById('fs-next')?.addEventListener('click', () => this.nextEvent());
    
    document.getElementById('fs-exit')?.addEventListener('click', () => {
      document.exitFullscreen();
    });
    
    document.getElementById('fs-timeline')?.addEventListener('click', (e) => {
      const rect = e.currentTarget.getBoundingClientRect();
      const percent = (e.clientX - rect.left) / rect.width;
      this.seek(percent * this.duration);
    });
    
    // Écouter la sortie du fullscreen
    document.addEventListener('fullscreenchange', () => {
      if (!document.fullscreenElement) {
        controls.remove();
        this.isFullscreen = false;
      }
    });
  }

  /**
   * Toggle agrandissement
   */
  toggleEnlarge() {
    const container = document.querySelector('.replay-container');
    if (container) {
      container.classList.toggle('enlarged');
    }
  }

  /**
   * Exporte le replay actuel en PDF
   */
  async exportToPDF() {
    try {
      console.log('[Replay] Exporting to PDF...');

      const filename = await pdfExport.exportReplay(
        this.session,
        this.currentTime
      );

      pdfExport.showNotification(`PDF exporté: ${filename}`, 'success');

    } catch (error) {
      console.error('[Replay] Error exporting PDF:', error);
      pdfExport.showNotification('Erreur lors de l\'export PDF', 'error');
    }
  }

  // OPTIM #5: Removed duplicated utility functions
  // Now using utils.js: formatTime, formatDuration, truncateUrl, getEventLabel

  /**
   * Nettoie les ressources (OPTIM #4: free memory)
   */
  cleanup() {
    this.pause();

    // OPTIM #4: Libérer la mémoire des screenshots loaded
    if (this.screenshots && this.screenshots.length > 0) {
      let freedCount = 0;
      this.screenshots.forEach(screenshot => {
        if (screenshot.loaded && screenshot.dataUrl) {
          // Libérer le dataUrl de la mémoire
          screenshot.dataUrl = null;
          screenshot.loaded = false;
          freedCount++;
        }
      });

      if (freedCount > 0) {
        console.log(`[Replay] Cleanup: freed ${freedCount} screenshots from memory`);
      }
    }
  }
}

// Export pour utilisation en module
export default new ReplayManager();
