/**
 * SUMO UX Tracker - PDF Export Module
 * FEATURE #1: Export Analytics, Heatmap, Replay to PDF
 */

import { formatDuration, formatTime, truncateUrl } from './utils.js';

class PDFExportManager {
  constructor() {
    this.jsPDF = null;
    this.html2canvas = null;
    this.loadingLibraries = false;
  }

  /**
   * Charge les librairies jsPDF et html2canvas depuis CDN
   */
  async loadLibraries() {
    if (this.jsPDF && this.html2canvas) {
      return; // Déjà chargées
    }

    if (this.loadingLibraries) {
      // Attendre que le chargement en cours se termine
      await new Promise(resolve => {
        const check = setInterval(() => {
          if (this.jsPDF && this.html2canvas) {
            clearInterval(check);
            resolve();
          }
        }, 100);
      });
      return;
    }

    this.loadingLibraries = true;

    try {
      // Charger html2canvas
      if (!window.html2canvas) {
        await this.loadScript('https://cdnjs.cloudflare.com/ajax/libs/html2canvas/1.4.1/html2canvas.min.js');
      }
      this.html2canvas = window.html2canvas;

      // Charger jsPDF
      if (!window.jspdf) {
        await this.loadScript('https://cdnjs.cloudflare.com/ajax/libs/jspdf/2.5.1/jspdf.umd.min.js');
      }
      this.jsPDF = window.jspdf.jsPDF;

      console.log('[PDFExport] Libraries loaded successfully');
      this.loadingLibraries = false;
    } catch (error) {
      this.loadingLibraries = false;
      console.error('[PDFExport] Error loading libraries:', error);
      throw new Error('Impossible de charger les librairies PDF. Vérifiez votre connexion internet.');
    }
  }

  /**
   * Charge un script depuis une URL
   */
  loadScript(url) {
    return new Promise((resolve, reject) => {
      const script = document.createElement('script');
      script.src = url;
      script.onload = resolve;
      script.onerror = reject;
      document.head.appendChild(script);
    });
  }

  /**
   * Export Analytics en PDF
   */
  async exportAnalytics(session, metrics) {
    await this.loadLibraries();

    const pdf = new this.jsPDF({
      orientation: 'portrait',
      unit: 'mm',
      format: 'a4'
    });

    const pageWidth = pdf.internal.pageSize.getWidth();
    const pageHeight = pdf.internal.pageSize.getHeight();
    const margin = 15;
    let yPos = margin;

    // Header
    pdf.setFontSize(20);
    pdf.setTextColor(59, 130, 246); // Blue
    pdf.text('SUMO UX Tracker', margin, yPos);
    yPos += 8;

    pdf.setFontSize(16);
    pdf.setTextColor(0, 0, 0);
    pdf.text('Rapport d\'Analytics', margin, yPos);
    yPos += 12;

    // Session Info
    pdf.setFontSize(10);
    pdf.setTextColor(100, 100, 100);
    const sessionInfo = [
      `URL: ${truncateUrl(session.metadata?.initialUrl || session.url || 'N/A', 80)}`,
      `Date: ${new Date(session.metadata?.importDate || Date.now()).toLocaleDateString('fr-FR')}`,
      `Durée: ${formatDuration(metrics.duration)}`,
      `Session ID: ${session.id}`
    ];

    sessionInfo.forEach(line => {
      pdf.text(line, margin, yPos);
      yPos += 5;
    });

    yPos += 5;

    // UX Score (circle)
    const uxScore = metrics.uxScore || 0;
    const circleX = pageWidth / 2;
    const circleY = yPos + 25;
    const circleRadius = 20;

    // Circle background
    pdf.setFillColor(240, 240, 240);
    pdf.circle(circleX, circleY, circleRadius, 'F');

    // Circle border (color based on score)
    let scoreColor = [34, 197, 94]; // Green
    if (uxScore < 70) scoreColor = [251, 146, 60]; // Orange
    if (uxScore < 50) scoreColor = [239, 68, 68]; // Red

    pdf.setDrawColor(...scoreColor);
    pdf.setLineWidth(2);
    pdf.circle(circleX, circleY, circleRadius, 'S');

    // Score text
    pdf.setFontSize(24);
    pdf.setTextColor(...scoreColor);
    pdf.text(`${uxScore}`, circleX, circleY + 2, { align: 'center' });

    pdf.setFontSize(10);
    pdf.setTextColor(100, 100, 100);
    pdf.text('UX Score', circleX, circleY + 10, { align: 'center' });

    yPos = circleY + circleRadius + 15;

    // Metrics Grid
    pdf.setFontSize(12);
    pdf.setTextColor(0, 0, 0);
    pdf.text('Métriques Clés', margin, yPos);
    yPos += 7;

    const metricsData = [
      ['Clics', metrics.clicks || 0],
      ['Rage Clicks', metrics.rageClicks || 0],
      ['Défilements', metrics.scrolls || 0],
      ['Pages vues', metrics.pagesViewed || 0]
    ];

    pdf.setFontSize(10);
    const colWidth = (pageWidth - 2 * margin) / 2;

    metricsData.forEach((metric, index) => {
      const row = Math.floor(index / 2);
      const col = index % 2;
      const x = margin + col * colWidth;
      const y = yPos + row * 12;

      pdf.setTextColor(100, 100, 100);
      pdf.text(metric[0], x, y);
      pdf.setFontSize(14);
      pdf.setTextColor(0, 0, 0);
      pdf.text(String(metric[1]), x, y + 5);
      pdf.setFontSize(10);
    });

    yPos += metricsData.length / 2 * 12 + 10;

    // Insights
    if (metrics.insights && metrics.insights.length > 0) {
      pdf.setFontSize(12);
      pdf.setTextColor(0, 0, 0);
      pdf.text('Insights', margin, yPos);
      yPos += 7;

      pdf.setFontSize(9);
      metrics.insights.slice(0, 3).forEach(insight => {
        // Icon
        pdf.text(insight.icon || '•', margin, yPos);

        // Title
        pdf.setTextColor(0, 0, 0);
        pdf.setFont(undefined, 'bold');
        pdf.text(insight.title, margin + 5, yPos);

        yPos += 4;

        // Description
        pdf.setFont(undefined, 'normal');
        pdf.setTextColor(100, 100, 100);
        const descLines = pdf.splitTextToSize(insight.description, pageWidth - 2 * margin - 5);
        pdf.text(descLines, margin + 5, yPos);
        yPos += descLines.length * 4 + 3;

        // Check if we need new page
        if (yPos > pageHeight - margin - 20) {
          pdf.addPage();
          yPos = margin;
        }
      });
    }

    // Footer
    const footer = `Généré le ${new Date().toLocaleString('fr-FR')} | SUMO UX Tracker`;
    pdf.setFontSize(8);
    pdf.setTextColor(150, 150, 150);
    pdf.text(footer, pageWidth / 2, pageHeight - 10, { align: 'center' });

    // Save
    const filename = `analytics-${session.id}-${Date.now()}.pdf`;
    pdf.save(filename);

    console.log(`[PDFExport] Analytics exported: ${filename}`);
    return filename;
  }

  /**
   * Export Heatmap en PDF
   */
  async exportHeatmap(session, screenshotIndex = 0) {
    await this.loadLibraries();

    // Capture du canvas heatmap
    const heatmapContainer = document.getElementById('heatmap-container');
    if (!heatmapContainer) {
      throw new Error('Heatmap container not found');
    }

    // Créer un élément temporaire pour le contenu
    const exportContainer = document.createElement('div');
    exportContainer.style.position = 'absolute';
    exportContainer.style.left = '-9999px';
    exportContainer.style.width = '1200px';
    exportContainer.style.background = 'white';
    exportContainer.style.padding = '20px';
    document.body.appendChild(exportContainer);

    // Header
    exportContainer.innerHTML = `
      <div style="font-family: Arial, sans-serif;">
        <h1 style="color: #3b82f6; margin: 0 0 10px 0;">SUMO UX Tracker - Heatmap</h1>
        <div style="color: #666; font-size: 12px; margin-bottom: 20px;">
          <div>URL: ${truncateUrl(session.metadata?.initialUrl || session.url || 'N/A', 100)}</div>
          <div>Screenshot: ${screenshotIndex + 1} / ${session.screenshots?.length || 1}</div>
          <div>Date: ${new Date().toLocaleDateString('fr-FR')}</div>
        </div>
        <div id="heatmap-capture" style="position: relative; display: inline-block;">
          <!-- Heatmap will be inserted here -->
        </div>
      </div>
    `;

    // Capture l'image de base + canvas overlay
    const baseImg = document.getElementById('heatmap-base');
    const overlayCanvas = document.getElementById('heatmap-canvas');

    const captureDiv = exportContainer.querySelector('#heatmap-capture');

    if (baseImg && overlayCanvas) {
      // Créer un canvas composite
      const composite = document.createElement('canvas');
      composite.width = baseImg.naturalWidth;
      composite.height = baseImg.naturalHeight;
      const ctx = composite.getContext('2d');

      // Dessiner l'image de base
      ctx.drawImage(baseImg, 0, 0);

      // Dessiner l'overlay heatmap
      ctx.drawImage(overlayCanvas, 0, 0, composite.width, composite.height);

      // Ajouter au container
      composite.style.maxWidth = '100%';
      composite.style.height = 'auto';
      captureDiv.appendChild(composite);
    }

    try {
      // Capture avec html2canvas
      const canvas = await this.html2canvas(exportContainer, {
        scale: 2,
        useCORS: true,
        logging: false
      });

      // Créer PDF
      const pdf = new this.jsPDF({
        orientation: canvas.width > canvas.height ? 'landscape' : 'portrait',
        unit: 'mm',
        format: 'a4'
      });

      const pageWidth = pdf.internal.pageSize.getWidth();
      const pageHeight = pdf.internal.pageSize.getHeight();

      const imgData = canvas.toDataURL('image/jpeg', 0.8);
      const imgWidth = pageWidth - 20; // margins
      const imgHeight = (canvas.height * imgWidth) / canvas.width;

      pdf.addImage(imgData, 'JPEG', 10, 10, imgWidth, Math.min(imgHeight, pageHeight - 20));

      // Footer
      const footer = `Généré le ${new Date().toLocaleString('fr-FR')} | SUMO UX Tracker`;
      pdf.setFontSize(8);
      pdf.setTextColor(150, 150, 150);
      pdf.text(footer, pageWidth / 2, pageHeight - 5, { align: 'center' });

      // Save
      const filename = `heatmap-${session.id}-screenshot-${screenshotIndex + 1}-${Date.now()}.pdf`;
      pdf.save(filename);

      console.log(`[PDFExport] Heatmap exported: ${filename}`);
      return filename;
    } finally {
      // Cleanup
      document.body.removeChild(exportContainer);
    }
  }

  /**
   * Export Replay screenshot en PDF
   */
  async exportReplay(session, currentTime = 0) {
    await this.loadLibraries();

    const pdf = new this.jsPDF({
      orientation: 'portrait',
      unit: 'mm',
      format: 'a4'
    });

    const pageWidth = pdf.internal.pageSize.getWidth();
    const pageHeight = pdf.internal.pageSize.getHeight();
    const margin = 15;
    let yPos = margin;

    // Header
    pdf.setFontSize(20);
    pdf.setTextColor(59, 130, 246);
    pdf.text('SUMO UX Tracker', margin, yPos);
    yPos += 8;

    pdf.setFontSize(16);
    pdf.setTextColor(0, 0, 0);
    pdf.text('Replay Session', margin, yPos);
    yPos += 12;

    // Session Info
    pdf.setFontSize(10);
    pdf.setTextColor(100, 100, 100);
    const sessionInfo = [
      `URL: ${truncateUrl(session.metadata?.initialUrl || session.url || 'N/A', 80)}`,
      `Temps: ${formatTime(currentTime)} / ${formatDuration(session.metadata?.duration || 0)}`,
      `Date: ${new Date(session.metadata?.importDate || Date.now()).toLocaleDateString('fr-FR')}`
    ];

    sessionInfo.forEach(line => {
      pdf.text(line, margin, yPos);
      yPos += 5;
    });

    yPos += 5;

    // Capture screenshot
    const screenshotEl = document.getElementById('replay-screenshot');
    if (screenshotEl && screenshotEl.src) {
      const imgWidth = pageWidth - 2 * margin;
      const imgHeight = (screenshotEl.naturalHeight * imgWidth) / screenshotEl.naturalWidth;

      // Check if fits on page
      if (yPos + imgHeight > pageHeight - margin) {
        pdf.addPage();
        yPos = margin;
      }

      pdf.addImage(screenshotEl.src, 'JPEG', margin, yPos, imgWidth, imgHeight);
      yPos += imgHeight + 10;
    }

    // Footer
    const footer = `Généré le ${new Date().toLocaleString('fr-FR')} | SUMO UX Tracker`;
    pdf.setFontSize(8);
    pdf.setTextColor(150, 150, 150);
    pdf.text(footer, pageWidth / 2, pageHeight - 10, { align: 'center' });

    // Save
    const filename = `replay-${session.id}-${Date.now()}.pdf`;
    pdf.save(filename);

    console.log(`[PDFExport] Replay exported: ${filename}`);
    return filename;
  }

  /**
   * Affiche notification de succès
   */
  showNotification(message, type = 'success') {
    const notification = document.createElement('div');
    notification.className = `pdf-export-notification pdf-export-${type}`;
    notification.textContent = message;
    notification.style.cssText = `
      position: fixed;
      top: 20px;
      right: 20px;
      background: ${type === 'success' ? '#22c55e' : '#ef4444'};
      color: white;
      padding: 15px 20px;
      border-radius: 8px;
      box-shadow: 0 4px 12px rgba(0,0,0,0.15);
      z-index: 10000;
      font-family: Arial, sans-serif;
      font-size: 14px;
      animation: slideInRight 0.3s ease;
    `;

    document.body.appendChild(notification);

    setTimeout(() => {
      notification.style.animation = 'slideOutRight 0.3s ease';
      setTimeout(() => {
        document.body.removeChild(notification);
      }, 300);
    }, 3000);
  }
}

// Export singleton
export default new PDFExportManager();
