/**
 * SUMO UX Tracker - Patterns Detector
 * Analyse les sessions pour détecter patterns UX
 */

class PatternsDetector {
  constructor() {
    this.patterns = {
      rage_clicks: [],
      dead_clicks: [],
      confusion_zones: [],
      form_abandonments: [],
      repetitive_scrolls: []
    };
  }

  /**
   * Analyse une session complète pour détecter tous les patterns
   */
  analyzeSession(session) {
    console.log('[PatternsDetector] Analyzing session for UX patterns...');

    // Reset patterns
    this.patterns = {
      rage_clicks: [],
      dead_clicks: [],
      confusion_zones: [],
      form_abandonments: [],
      repetitive_scrolls: []
    };

    const events = session.data?.events || [];

    // Extraire les patterns déjà détectés par l'extension
    this.extractDetectedPatterns(events);

    // Analyser les events pour détecter patterns manqués
    this.detectRageClicksFromEvents(events);
    this.detectDeadClicksFromEvents(events);
    this.detectConfusionZonesFromEvents(events);
    this.detectFormAbandonmentFromEvents(events);
    this.detectRepetitiveScrollsFromEvents(events);

    console.log('[PatternsDetector] Analysis complete:', this.patterns);

    return this.patterns;
  }

  /**
   * Extrait les patterns déjà détectés par l'extension
   */
  extractDetectedPatterns(events) {
    events.forEach(event => {
      if (event.type === 'pattern_detected') {
        const pattern = event.pattern;

        switch (pattern) {
          case 'rage_click':
            this.patterns.rage_clicks.push({
              timestamp: event.timestamp,
              position: event.position,
              intensity: event.intensity,
              clickCount: event.clickCount,
              duration: event.duration,
              source: 'extension'
            });
            break;

          case 'dead_click':
            this.patterns.dead_clicks.push({
              timestamp: event.timestamp,
              position: event.position,
              element: event.element,
              selector: event.selector,
              source: 'extension'
            });
            break;

          case 'confusion_zone':
            this.patterns.confusion_zones.push({
              timestamp: event.timestamp,
              position: event.position,
              duration: event.duration,
              source: 'extension'
            });
            break;

          case 'form_abandonment':
            this.patterns.form_abandonments.push({
              timestamp: event.timestamp,
              element: event.element,
              fieldType: event.fieldType,
              charactersLost: event.charactersLost,
              source: 'extension'
            });
            break;

          case 'repetitive_scroll':
            this.patterns.repetitive_scrolls.push({
              timestamp: event.timestamp,
              changes: event.changes,
              source: 'extension'
            });
            break;
        }
      }
    });
  }

  /**
   * Détecte rage clicks depuis les events de clic
   */
  detectRageClicksFromEvents(events) {
    const clickEvents = events.filter(e => e.type === 'click');
    const threshold = 3000; // 3 secondes
    const radius = 50; // 50px
    const minClicks = 4;

    for (let i = 0; i < clickEvents.length; i++) {
      const click = clickEvents[i];
      const recentClicks = [click];

      // Chercher clics proches dans le temps et l'espace
      for (let j = i + 1; j < clickEvents.length; j++) {
        const nextClick = clickEvents[j];

        if (nextClick.timestamp - click.timestamp > threshold) break;

        const distance = Math.sqrt(
          Math.pow(nextClick.x - click.x, 2) +
          Math.pow(nextClick.y - click.y, 2)
        );

        if (distance < radius) {
          recentClicks.push(nextClick);
        }
      }

      // Si rage click détecté
      if (recentClicks.length >= minClicks) {
        // Vérifier si pas déjà détecté
        const alreadyDetected = this.patterns.rage_clicks.some(rc =>
          Math.abs(rc.timestamp - click.timestamp) < 1000
        );

        if (!alreadyDetected) {
          this.patterns.rage_clicks.push({
            timestamp: click.timestamp,
            position: {
              x: Math.round(recentClicks.reduce((sum, c) => sum + c.x, 0) / recentClicks.length),
              y: Math.round(recentClicks.reduce((sum, c) => sum + c.y, 0) / recentClicks.length)
            },
            intensity: Math.min(10, Math.round(recentClicks.length / 2)),
            clickCount: recentClicks.length,
            duration: recentClicks[recentClicks.length - 1].timestamp - click.timestamp,
            source: 'analysis'
          });
        }

        i += recentClicks.length - 1; // Skip processed clicks
      }
    }
  }

  /**
   * Détecte dead clicks (clics sur éléments non-interactifs)
   */
  detectDeadClicksFromEvents(events) {
    const clickEvents = events.filter(e => e.type === 'click');

    clickEvents.forEach(click => {
      // Si l'élément cliqué est un DIV, SPAN, P, IMG sans interactivité
      const nonInteractiveTags = ['DIV', 'SPAN', 'P', 'IMG', 'SECTION', 'ARTICLE', 'H1', 'H2', 'H3', 'H4', 'H5', 'H6'];

      if (click.element && nonInteractiveTags.includes(click.element)) {
        // Vérifier si pas déjà détecté
        const alreadyDetected = this.patterns.dead_clicks.some(dc =>
          Math.abs(dc.timestamp - click.timestamp) < 100
        );

        if (!alreadyDetected) {
          this.patterns.dead_clicks.push({
            timestamp: click.timestamp,
            position: { x: click.x, y: click.y },
            element: click.element,
            selector: click.selector || 'unknown',
            source: 'analysis'
          });
        }
      }
    });
  }

  /**
   * Détecte confusion zones (hovers prolongés)
   */
  detectConfusionZonesFromEvents(events) {
    const mouseMoveEvents = events.filter(e => e.type === 'mouse_move');
    const threshold = 3000; // 3 secondes
    const radius = 100; // 100px
    const minMoves = 15;

    for (let i = 0; i < mouseMoveEvents.length - minMoves; i++) {
      const firstMove = mouseMoveEvents[i];
      const recentMoves = [firstMove];

      // Collecter mouvements dans timeframe
      for (let j = i + 1; j < mouseMoveEvents.length; j++) {
        const nextMove = mouseMoveEvents[j];

        if (nextMove.timestamp - firstMove.timestamp > threshold) break;

        const distance = Math.sqrt(
          Math.pow(nextMove.x - firstMove.x, 2) +
          Math.pow(nextMove.y - firstMove.y, 2)
        );

        if (distance < radius) {
          recentMoves.push(nextMove);
        }
      }

      // Si hover prolongé détecté
      if (recentMoves.length >= minMoves) {
        const duration = recentMoves[recentMoves.length - 1].timestamp - firstMove.timestamp;

        if (duration >= threshold) {
          // Vérifier si pas déjà détecté
          const alreadyDetected = this.patterns.confusion_zones.some(cz =>
            Math.abs(cz.timestamp - firstMove.timestamp) < 2000
          );

          if (!alreadyDetected) {
            this.patterns.confusion_zones.push({
              timestamp: firstMove.timestamp,
              position: {
                x: Math.round(recentMoves.reduce((sum, m) => sum + m.x, 0) / recentMoves.length),
                y: Math.round(recentMoves.reduce((sum, m) => sum + m.y, 0) / recentMoves.length)
              },
              duration: duration,
              source: 'analysis'
            });
          }

          i += recentMoves.length - 1;
        }
      }
    }
  }

  /**
   * Détecte form abandonment (champs remplis puis effacés)
   */
  detectFormAbandonmentFromEvents(events) {
    // Chercher événements avec type contenant "input" ou données de formulaire
    const formEvents = events.filter(e =>
      e.type === 'input' || e.element === 'INPUT' || e.element === 'TEXTAREA'
    );

    // Grouper par champ
    const fieldGroups = new Map();

    formEvents.forEach(event => {
      const key = event.selector || event.element || 'unknown';

      if (!fieldGroups.has(key)) {
        fieldGroups.set(key, []);
      }

      fieldGroups.get(key).push(event);
    });

    // Analyser chaque champ
    fieldGroups.forEach((eventsList, fieldKey) => {
      // Chercher pattern: valeur remplie -> valeur vide
      for (let i = 0; i < eventsList.length - 1; i++) {
        const current = eventsList[i];
        const next = eventsList[i + 1];

        // Si valeur présente puis absente
        if (current.value && current.value.length > 3 && next.value === '') {
          this.patterns.form_abandonments.push({
            timestamp: next.timestamp,
            element: current.element || 'INPUT',
            fieldType: current.fieldType || 'text',
            charactersLost: current.value.length,
            source: 'analysis'
          });
        }
      }
    });
  }

  /**
   * Détecte scrolls répétitifs
   */
  detectRepetitiveScrollsFromEvents(events) {
    const scrollEvents = events.filter(e => e.type === 'scroll');

    if (scrollEvents.length < 8) return;

    for (let i = 0; i < scrollEvents.length - 8; i++) {
      const windowEvents = scrollEvents.slice(i, i + 8);
      let changes = 0;

      // Compter changements de direction
      for (let j = 1; j < windowEvents.length; j++) {
        const prev = windowEvents[j - 1].scrollY || 0;
        const curr = windowEvents[j].scrollY || 0;
        const diff = Math.abs(curr - prev);

        if (diff > 100 && j > 1) {
          const prevPrev = windowEvents[j - 2].scrollY || 0;
          const direction1 = curr > prev;
          const direction2 = prev > prevPrev;

          if (direction1 !== direction2) {
            changes++;
          }
        }
      }

      // Si 3+ changements = scroll répétitif
      if (changes >= 3) {
        // Vérifier si pas déjà détecté
        const alreadyDetected = this.patterns.repetitive_scrolls.some(rs =>
          Math.abs(rs.timestamp - windowEvents[0].timestamp) < 5000
        );

        if (!alreadyDetected) {
          this.patterns.repetitive_scrolls.push({
            timestamp: windowEvents[0].timestamp,
            changes: changes,
            source: 'analysis'
          });
        }

        i += 4; // Skip ahead
      }
    }
  }

  /**
   * Obtient les stats globales des patterns
   */
  getStats() {
    return {
      total_patterns: this.getTotalCount(),
      rage_clicks: this.patterns.rage_clicks.length,
      dead_clicks: this.patterns.dead_clicks.length,
      confusion_zones: this.patterns.confusion_zones.length,
      form_abandonments: this.patterns.form_abandonments.length,
      repetitive_scrolls: this.patterns.repetitive_scrolls.length
    };
  }

  /**
   * Compte total de patterns
   */
  getTotalCount() {
    return (
      this.patterns.rage_clicks.length +
      this.patterns.dead_clicks.length +
      this.patterns.confusion_zones.length +
      this.patterns.form_abandonments.length +
      this.patterns.repetitive_scrolls.length
    );
  }

  /**
   * Obtient les patterns par sévérité
   */
  getPatternsBySeverity() {
    const critical = this.patterns.rage_clicks.length + this.patterns.dead_clicks.length;
    const warning = this.patterns.confusion_zones.length + this.patterns.form_abandonments.length;
    const info = this.patterns.repetitive_scrolls.length;

    return { critical, warning, info };
  }

  /**
   * Génère un résumé texte des patterns
   */
  getSummary() {
    const stats = this.getStats();
    const severity = this.getPatternsBySeverity();

    if (stats.total_patterns === 0) {
      return 'Aucun pattern UX problématique détecté - Session fluide';
    }

    const parts = [];

    if (stats.rage_clicks > 0) {
      parts.push(`${stats.rage_clicks} rage click${stats.rage_clicks > 1 ? 's' : ''}`);
    }

    if (stats.dead_clicks > 0) {
      parts.push(`${stats.dead_clicks} dead click${stats.dead_clicks > 1 ? 's' : ''}`);
    }

    if (stats.confusion_zones > 0) {
      parts.push(`${stats.confusion_zones} zone${stats.confusion_zones > 1 ? 's' : ''} de confusion`);
    }

    if (stats.form_abandonments > 0) {
      parts.push(`${stats.form_abandonments} abandon${stats.form_abandonments > 1 ? 's' : ''} de formulaire`);
    }

    if (stats.repetitive_scrolls > 0) {
      parts.push(`${stats.repetitive_scrolls} scroll${stats.repetitive_scrolls > 1 ? 's' : ''} répétitif${stats.repetitive_scrolls > 1 ? 's' : ''}`);
    }

    return parts.join(', ');
  }
}

// Export singleton
const patternsDetector = new PatternsDetector();
export default patternsDetector;
