/**
 * SUMO UX Tracker - Main Application
 * Point d'entrée principal
 */

import db from './db.js';
import importService from './import.js';
import sessionsManager from './sessions.js';
import foldersManager from './folders-manager.js';
import replayManager from './replay.js';
import heatmapManager from './heatmap.js';
import analyticsManager from './analytics.js';
import comparisonManager from './comparison.js';
import feedbackViewer from './feedback-viewer.js';
import dataAnonymizer from './data-anonymizer.js';
import aiAnalyzer from './ai-analyzer.js';
import funnelUI from './funnel-ui.js';

class SumoUXApp {
  constructor() {
    this.currentView = 'import';
    this.currentSession = null;
    this.elements = {};
    this.importManager = null;
  }

  /**
   * Initialise l'application
   */
  async init() {
    console.log('[App] Initializing SUMO UX Tracker...');

    try {
      // Initialiser la base de données
      await db.init();
      console.log('[App] Database initialized');

      // Initialiser le service d’import
      this.initImportManager();

      // Récupérer les éléments DOM
      this.cacheElements();

      // Attacher les event listeners
      this.attachEventListeners();

      // Vérifier s'il y a des sessions
      const sessionCount = await db.getSessionCount();
      console.log('[App] Sessions count:', sessionCount);

      // Initialiser le folders manager
      await foldersManager.init();
      console.log('[App] Folders manager initialized');

      // Initialiser l'UI des funnels
      funnelUI.init();
      console.log('[App] Funnel UI initialized');

      // Toujours afficher la vue sessions
      this.showView('sessions');

      // Si des sessions existent, les charger
      if (sessionCount > 0) {
        await sessionsManager.loadSessions();
      } else {
        // Sinon, ouvrir automatiquement la modal d'import
        this.showImportModal();
      }

      console.log('[App] Initialization complete');
    } catch (error) {
      console.error('[App] Initialization error:', error);
      this.showError('Erreur lors de l\'initialisation de l\'application');
    }
  }

  /**
   * Initialise le service d’import
   */
  initImportManager() {
    this.importManager = importService;
    console.log('[App] ImportService initialisé');
  }

  /**
   * Cache les éléments DOM
   */
  cacheElements() {
    // Vues
    this.elements.views = {
      sessions: document.getElementById('view-sessions'),
      comparison: document.getElementById('view-comparison'),
      analysis: document.getElementById('view-analysis')
    };

    // Modal Import
    this.elements.modalImport = document.getElementById('modal-import');
    this.elements.dropzone = document.getElementById('dropzone');
    this.elements.fileInput = document.getElementById('file-input');
    this.elements.importStatus = document.getElementById('import-status');
    this.elements.btnImport = document.getElementById('btn-import');
    this.elements.btnImportClose = document.getElementById('btn-import-close');

    // Sessions
    this.elements.sessionsList = document.getElementById('sessions-list');
    this.elements.searchSessions = document.getElementById('search-sessions');
    this.elements.sortSessions = document.getElementById('sort-sessions');

    // Analysis
    this.elements.btnBack = document.getElementById('btn-back');
    this.elements.analysisTitle = document.getElementById('analysis-title');
    this.elements.tabBtns = document.querySelectorAll('.tab-btn');
    this.elements.tabContents = document.querySelectorAll('.tab-content');

    console.log('[App] Elements cached');
  }

  /**
   * Attache les event listeners
   */
  attachEventListeners() {
    // Import button (header)
    this.elements.btnImport?.addEventListener('click', () => {
      this.showImportModal();
    });

    // Import modal close button
    this.elements.btnImportClose?.addEventListener('click', () => {
      this.closeImportModal();
    });

    // Import modal backdrop
    this.elements.modalImport?.querySelector('.modal-backdrop')?.addEventListener('click', () => {
      this.closeImportModal();
    });

    // Dropzone
    this.elements.dropzone?.addEventListener('click', () => {
      this.elements.fileInput?.click();
    });

    this.elements.dropzone?.addEventListener('dragover', (e) => {
      e.preventDefault();
      e.stopPropagation();
      this.elements.dropzone.classList.add('drag-over');
    });

    this.elements.dropzone?.addEventListener('dragleave', (e) => {
      e.preventDefault();
      e.stopPropagation();
      this.elements.dropzone.classList.remove('drag-over');
    });

    this.elements.dropzone?.addEventListener('drop', (e) => {
      e.preventDefault();
      e.stopPropagation();
      this.elements.dropzone.classList.remove('drag-over');
      
      const files = Array.from(e.dataTransfer.files);
      this.handleFiles(files);
    });

    // File input
    this.elements.fileInput?.addEventListener('change', (e) => {
      const files = Array.from(e.target.files);
      this.handleFiles(files);
      e.target.value = ''; // Reset input
    });

    // Search sessions
    this.elements.searchSessions?.addEventListener('input', (e) => {
      sessionsManager.filterSessions(e.target.value);
    });

    // Sort sessions
    this.elements.sortSessions?.addEventListener('change', (e) => {
      sessionsManager.sortSessions(e.target.value);
    });

    // Back button
    this.elements.btnBack?.addEventListener('click', () => {
      this.showView('sessions');
      sessionsManager.loadSessions();
    });

    // Tab buttons
    this.elements.tabBtns.forEach(btn => {
      btn.addEventListener('click', () => {
        const tabName = btn.dataset.tab;
        this.switchTab(tabName);
      });
    });

    console.log('[App] Event listeners attached');
  }

  /**
   * Gère les fichiers uploadés
   */
  async handleFiles(files) {
    console.log('[App] Handling files:', files.length);

    // Filtrer les fichiers valides
    const validFiles = files.filter(file => this.importManager.isValidFile(file));
    
    if (validFiles.length === 0) {
      this.showError('Aucun fichier .uxtrack valide détecté');
      return;
    }

    // Afficher le status
    this.elements.importStatus.innerHTML = '';

    // Importer chaque fichier
    for (const file of validFiles) {
      const itemEl = this.createImportItem(file);
      this.elements.importStatus.appendChild(itemEl);

      try {
        // Importer le fichier via ImportService
        const session = await this.importManager.importFile(file);
        this.updateImportItem(itemEl, 'success', 'Importé avec succès');
      } catch (error) {
        console.error('[App] Import error:', error);
        this.updateImportItem(itemEl, 'error', error.message);
      }
    }

    // Fermer la modal et recharger les sessions après un délai
    setTimeout(async () => {
      this.closeImportModal();
      // S'assurer que la vue sessions est affichée
      this.showView('sessions');
      await sessionsManager.loadSessions();
      // Mettre à jour les compteurs
      if (foldersManager) {
        await foldersManager.updateFolderCounts();
      }
    }, 2000);
  }

  /**
   * Crée un élément d'import
   */
  createImportItem(file) {
    const div = document.createElement('div');
    div.className = 'import-item loading';
    div.innerHTML = `
      <div class="import-item-icon">📦</div>
      <div class="import-item-info">
        <div class="import-item-name">${file.name}</div>
        <div class="import-item-status">Import en cours...</div>
        <div class="import-item-progress">
          <div class="import-item-progress-bar" style="width: 0%"></div>
        </div>
      </div>
    `;

    // Animer la progress bar
    const progressBar = div.querySelector('.import-item-progress-bar');
    let progress = 0;
    const interval = setInterval(() => {
      progress += Math.random() * 30;
      if (progress > 90) progress = 90;
      progressBar.style.width = `${progress}%`;
    }, 200);
    div.dataset.interval = interval;

    return div;
  }

  /**
   * Met à jour un élément d'import
   */
  updateImportItem(itemEl, status, message) {
    clearInterval(itemEl.dataset.interval);
    
    itemEl.className = `import-item ${status}`;
    itemEl.querySelector('.import-item-status').textContent = message;
    itemEl.querySelector('.import-item-progress-bar').style.width = '100%';
    
    if (status === 'success') {
      itemEl.querySelector('.import-item-icon').textContent = '✅';
    } else if (status === 'error') {
      itemEl.querySelector('.import-item-icon').textContent = '❌';
    }
  }

  /**
   * Change de vue
   */
  showView(viewName) {
    console.log('[App] Switching to view:', viewName);

    // Cacher toutes les vues (explicitement pour éviter les bugs)
    Object.values(this.elements.views).forEach(view => {
      if (view) {
        view.classList.remove('active');
        view.style.display = 'none'; // Force le masquage
      }
    });

    // Afficher la vue demandée
    if (this.elements.views[viewName]) {
      this.elements.views[viewName].classList.add('active');
      this.elements.views[viewName].style.display = 'block'; // Force l'affichage
      this.currentView = viewName;
      console.log('[App] View switched to:', viewName);
    }
  }

  /**
   * Change d'onglet dans l'analyse
   */
  switchTab(tabName) {
    console.log('[App] Switching to tab:', tabName);

    // Mettre à jour les boutons
    this.elements.tabBtns.forEach(btn => {
      if (btn.dataset.tab === tabName) {
        btn.classList.add('active');
      } else {
        btn.classList.remove('active');
      }
    });

    // Mettre à jour les contenus
    this.elements.tabContents.forEach(content => {
      if (content.id === `tab-${tabName}`) {
        content.classList.add('active');
      } else {
        content.classList.remove('active');
      }
    });

    // Initialiser le contenu de l'onglet
    if (this.currentSession) {
      switch (tabName) {
        case 'replay':
          replayManager.init(this.currentSession);
          break;
        case 'heatmap':
          heatmapManager.init(this.currentSession);
          break;
        case 'analytics':
          analyticsManager.init(this.currentSession);
          break;
        case 'feedback':
          feedbackViewer.init(this.currentSession);
          break;
        case 'funnels':
          // Afficher les funnels de la session
          if (analyticsManager.funnelResults) {
            funnelUI.renderSessionFunnels(this.currentSession.id, analyticsManager.funnelResults);
          }
          break;
      }
    }
  }

  /**
   * Ouvre une session pour analyse
   */
  async openSession(sessionId) {
    console.log('[App] Opening session:', sessionId);

    try {
      // Récupérer la session AVEC metadata screenshots (OPTIM #4: lazy loading)
      const session = await db.getSessionWithScreenshotsLazy(sessionId);
      if (!session) {
        throw new Error('Session not found');
      }

      this.currentSession = session;

      // Mettre à jour le titre
      this.elements.analysisTitle.textContent = `Session - ${session.metadata.initialUrl}`;

      // Afficher la vue d'analyse
      this.showView('analysis');

      // Initialiser le replay par défaut
      this.switchTab('replay');

    } catch (error) {
      console.error('[App] Error opening session:', error);
      this.showError('Erreur lors de l\'ouverture de la session');
    }
  }

  /**
   * Ouvre la comparaison A/B de deux sessions
   */
  async openComparison(sessionIdA, sessionIdB) {
    console.log('[App] Opening comparison:', sessionIdA, 'vs', sessionIdB);

    try {
      // Afficher la vue de comparaison
      this.showView('comparison');

      // Initialiser la comparaison
      await comparisonManager.init(sessionIdA, sessionIdB);

    } catch (error) {
      console.error('[App] Error opening comparison:', error);
      this.showError('Erreur lors de l\'ouverture de la comparaison');
    }
  }

  /**
   * Affiche la modal d'import
   */
  showImportModal() {
    console.log('[App] Opening import modal');
    this.elements.modalImport?.classList.add('active');
    // Vider le statut d'import
    if (this.elements.importStatus) {
      this.elements.importStatus.innerHTML = '';
    }
  }

  /**
   * Ferme la modal d'import
   */
  closeImportModal() {
    console.log('[App] Closing import modal');
    this.elements.modalImport?.classList.remove('active');
  }

  /**
   * Affiche une erreur
   */
  showError(message) {
    console.error('[App] Error:', message);
    alert(message); // TODO: Remplacer par un toast/notification
  }
}

// Initialiser l'app au chargement
const app = new SumoUXApp();

document.addEventListener('DOMContentLoaded', () => {
  app.init();
});

// Exposer globalement pour les autres modules
window.sumoApp = app;
window.sumoApp.sessionsManager = sessionsManager;
window.sumoApp.foldersManager = foldersManager;
window.feedbackViewer = feedbackViewer;
window.dataAnonymizer = dataAnonymizer;
window.aiAnalyzer = aiAnalyzer;

// ✅ FONCTIONS DE TEST v2 (temporaires pour validation)
window.testFoldersV2 = async function() {
  console.log('🧪 Testing Folders v2...');

  try {
    // Test 1: Créer un dossier
    const folder = await db.createFolder('Test Application Client', '#2563eb');
    console.log('✅ Folder created:', folder);

    // Test 2: Récupérer tous les dossiers
    const folders = await db.getAllFolders();
    console.log('✅ All folders:', folders);

    // Test 3: Créer des métadonnées pour une session
    const sessions = await db.getAllSessions();
    if (sessions.length > 0) {
      await db.saveSessionMetadata(sessions[0].id, {
        customTitle: 'Parcours création compte - Test',
        folderId: folder.id,
        tags: ['onboarding', 'mobile']
      });
      console.log('✅ Session metadata saved for:', sessions[0].id);

      // Test 4: Récupérer les métadonnées
      const metadata = await db.getSessionMetadata(sessions[0].id);
      console.log('✅ Session metadata:', metadata);
    }

    console.log('🎉 All tests passed!');
    return { success: true, folder, folders };

  } catch (error) {
    console.error('❌ Test failed:', error);
    return { success: false, error: error.message };
  }
};

console.log('💡 Tip: Run window.testFoldersV2() in console to test the new features');

export default app;
