/**
 * SUMO UX Tracker - Heatmap Manager
 * Génération de heatmaps (version avec trail souris)
 */

import db from './db.js';
import pdfExport from './pdfExport.js';

class HeatmapManager {
  constructor() {
    this.session = null;
    this.canvas = null;
    this.ctx = null;
    this.baseImage = null;
    this.currentScreenshotIndex = 0;
    this.layers = {
      clicks: { enabled: true, intensity: 0.7 },
      mouseMoves: { enabled: false, intensity: 0.7 },
      scroll: { enabled: false, intensity: 0.7 }
    };

    // Offscreen Canvas layers for performance optimization
    this.offscreenLayers = {
      clicks: { canvas: null, ctx: null, cached: false },
      mouseMoves: { canvas: null, ctx: null, cached: false },
      scroll: { canvas: null, ctx: null, cached: false }
    };

    // Track if layers need re-rendering
    this.dirtyLayers = {
      clicks: true,
      mouseMoves: true,
      scroll: true
    };
  }

  /**
   * Initialise la heatmap
   */
  async init(session) {
    console.log('[Heatmap] Initializing heatmap for session:', session.id);

    this.session = session;
    this.currentScreenshotIndex = 0;

    // Afficher l'interface
    this.render();

    // Charger le premier screenshot comme base
    await this.loadBaseImage();

    // Générer la heatmap
    this.generateHeatmap();

    // ✅ Charger les thumbnails en arrière-plan
    this.loadThumbnails();
  }

  /**
   * Charge les thumbnails de la galerie de manière asynchrone
   */
  async loadThumbnails() {
    if (!this.session.screenshots || this.session.screenshots.length <= 1) {
      return;
    }

    for (let i = 0; i < this.session.screenshots.length; i++) {
      const screenshot = this.session.screenshots[i];

      // Passer si déjà chargé
      if (screenshot.loaded && screenshot.dataUrl) {
        this.updateThumbnail(i, screenshot.dataUrl);
        continue;
      }

      try {
        const dataUrl = await db.loadScreenshotDataUrl(
          this.session.id,
          screenshot.id
        );

        screenshot.dataUrl = dataUrl;
        screenshot.loaded = true;

        // Mettre à jour le thumbnail dans la galerie
        this.updateThumbnail(i, dataUrl);

      } catch (error) {
        console.error(`[Heatmap] Error loading thumbnail ${i}:`, error);
      }
    }
  }

  /**
   * Met à jour le thumbnail dans la galerie
   */
  updateThumbnail(index, dataUrl) {
    const thumbnails = document.querySelectorAll('.heatmap-thumbnail');
    if (!thumbnails[index]) return;

    const placeholder = thumbnails[index].querySelector('.heatmap-thumbnail-placeholder');
    if (placeholder) {
      // Remplacer le placeholder par l'image
      const img = document.createElement('img');
      img.src = dataUrl;
      img.alt = `Screenshot ${index + 1}`;
      placeholder.replaceWith(img);
    }
  }
  
  /**
   * Render la navigation par screenshots (style gallery)
   */
  renderScreenshotNavigation() {
    if (!this.session.screenshots || this.session.screenshots.length <= 1) {
      return '';
    }

    return `
      <div class="heatmap-thumbnails-bar">
        <div class="heatmap-thumbnails-scroll">
          ${this.session.screenshots.map((screenshot, index) => `
            <div class="heatmap-thumbnail ${index === this.currentScreenshotIndex ? 'active' : ''}"
                 data-index="${index}"
                 data-screenshot-id="${screenshot.id}"
                 title="Screenshot ${index + 1}">
              <div class="heatmap-thumbnail-placeholder">📸</div>
              <div class="heatmap-thumbnail-overlay"></div>
            </div>
          `).join('')}
        </div>
      </div>
    `;
  }
  
  /**
   * Change le screenshot affiché
   */
  async changeScreenshot(index) {
    if (index < 0 || index >= this.session.screenshots.length) {
      return;
    }

    this.currentScreenshotIndex = index;
    const screenshot = this.session.screenshots[index];

    // OPTIM #4: Lazy loading - charger screenshot à la demande
    if (!screenshot.loaded && !screenshot.dataUrl) {
      try {
        const dataUrl = await db.loadScreenshotDataUrl(
          this.session.id,
          screenshot.id
        );
        screenshot.dataUrl = dataUrl;
        screenshot.loaded = true;
        console.log(`[Heatmap] Lazy loaded screenshot ${screenshot.id}`);
      } catch (error) {
        console.error('[Heatmap] Error loading screenshot:', error);
        return;
      }
    }

    // Mettre à jour l'image de base
    const baseEl = document.getElementById('heatmap-base');
    if (baseEl && screenshot.dataUrl) {
      baseEl.src = screenshot.dataUrl;

      // Attendre le chargement
      await new Promise((resolve) => {
        const img = new Image();
        img.onload = () => {
          this.baseImage = img;
          resolve();
        };
        img.src = screenshot.dataUrl;
      });
      
      // Mettre à jour l'UI (thumbnails)
      document.querySelectorAll('.heatmap-thumbnail').forEach((thumb, i) => {
        if (i === index) {
          thumb.classList.add('active');
        } else {
          thumb.classList.remove('active');
        }
      });
      
      // Mettre à jour le numéro affiché
      const numEl = document.getElementById('current-screenshot-num');
      if (numEl) {
        numEl.textContent = index + 1;
      }
      
      // Reconfigurer le canvas et régénérer
      this.setupCanvas();
      this.generateHeatmap();
    }
  }

  /**
   * Charge l'image de base (OPTIM #4: lazy loading)
   */
  async loadBaseImage() {
    if (!this.session.screenshots || this.session.screenshots.length === 0) {
      console.warn('[Heatmap] No screenshots available');
      return;
    }

    const firstScreenshot = this.session.screenshots[0];

    // OPTIM #4: Lazy loading - charger le premier screenshot à la demande
    if (!firstScreenshot.loaded && !firstScreenshot.dataUrl) {
      try {
        const dataUrl = await db.loadScreenshotDataUrl(
          this.session.id,
          firstScreenshot.id
        );
        firstScreenshot.dataUrl = dataUrl;
        firstScreenshot.loaded = true;
        console.log(`[Heatmap] Lazy loaded first screenshot ${firstScreenshot.id}`);
      } catch (error) {
        console.error('[Heatmap] Error loading first screenshot:', error);
        return;
      }
    }

    if (firstScreenshot.dataUrl) {
      const img = new Image();

      return new Promise((resolve) => {
        img.onload = () => {
          this.baseImage = img;

          // Mettre à jour l'affichage
          const baseEl = document.getElementById('heatmap-base');
          if (baseEl) {
            baseEl.src = firstScreenshot.dataUrl;
          }

          // Setup canvas avec les bonnes dimensions
          this.setupCanvas();

          resolve();
        };
        img.onerror = () => {
          console.error('[Heatmap] Error loading image');
          resolve();
        };
        img.src = firstScreenshot.dataUrl;
      });
    }
  }

  /**
   * Affiche l'interface
   */
  render() {
    const container = document.getElementById('heatmap-container');
    if (!container) return;

    const events = this.session.events || this.session.data?.events || [];
    const clicksCount = events.filter(e => e.type === 'click' || e.type === 'rage_click').length;
    const movesCount = events.filter(e => e.type === 'mouse_move').length;
    const scrollsCount = events.filter(e => e.type === 'scroll').length;

    container.innerHTML = `
      <div class="heatmap-viewer">
        <!-- Grande image principale avec heatmap -->
        <div class="heatmap-main-display">
          <div class="heatmap-canvas-container">
            <img class="heatmap-base-image" id="heatmap-base" alt="Base">
            <canvas class="heatmap-canvas" id="heatmap-canvas"></canvas>
          </div>
          
          <!-- Info overlay -->
          <div class="heatmap-current-info">
            <span class="heatmap-current-number">Screenshot <strong id="current-screenshot-num">1</strong> / ${this.session.screenshots?.length || 1}</span>
          </div>
        </div>
        
        <!-- Barre de thumbnails en bas -->
        ${this.renderScreenshotNavigation()}
      </div>
      
      <div class="heatmap-controls">
        <div class="heatmap-control-section">
          <h3>🎨 Couches</h3>
          
          <div class="heatmap-layer-toggle" data-layer="clicks">
            <div class="heatmap-layer-label">
              <span class="heatmap-layer-icon">🔴</span>
              <span>Clics (${clicksCount})</span>
            </div>
            <div class="heatmap-layer-checkbox ${this.layers.clicks.enabled ? 'checked' : ''}"></div>
          </div>
          
          <div class="heatmap-layer-toggle" data-layer="moves">
            <div class="heatmap-layer-label">
              <span class="heatmap-layer-icon">🔵</span>
              <span>Mouvements souris (${movesCount})</span>
            </div>
            <div class="heatmap-layer-checkbox ${this.layers.mouseMoves.enabled ? 'checked' : ''}"></div>
          </div>
          
          <div class="heatmap-layer-toggle" data-layer="scroll">
            <div class="heatmap-layer-label">
              <span class="heatmap-layer-icon">🟣</span>
              <span>Scroll (${scrollsCount})</span>
            </div>
            <div class="heatmap-layer-checkbox ${this.layers.scroll.enabled ? 'checked' : ''}"></div>
          </div>
        </div>
        
        <div class="heatmap-control-section">
          <h3>⚙️ Intensité</h3>
          
          <div class="heatmap-slider">
            <div class="heatmap-slider-label">
              <span>Clics</span>
              <span class="heatmap-slider-value" id="clicks-intensity-value">${Math.round(this.layers.clicks.intensity * 100)}%</span>
            </div>
            <input type="range" class="heatmap-slider-input" id="clicks-intensity-slider" 
                   min="0" max="100" value="${this.layers.clicks.intensity * 100}">
          </div>
          
          <div class="heatmap-slider">
            <div class="heatmap-slider-label">
              <span>Mouvements</span>
              <span class="heatmap-slider-value" id="moves-intensity-value">${Math.round(this.layers.mouseMoves.intensity * 100)}%</span>
            </div>
            <input type="range" class="heatmap-slider-input" id="moves-intensity-slider" 
                   min="0" max="100" value="${this.layers.mouseMoves.intensity * 100}">
          </div>
          
          <div class="heatmap-slider">
            <div class="heatmap-slider-label">
              <span>Scroll</span>
              <span class="heatmap-slider-value" id="scroll-intensity-value">${Math.round(this.layers.scroll.intensity * 100)}%</span>
            </div>
            <input type="range" class="heatmap-slider-input" id="scroll-intensity-slider" 
                   min="0" max="100" value="${this.layers.scroll.intensity * 100}">
          </div>
        </div>
        
        <div class="heatmap-control-section">
          <h3>📊 Statistiques</h3>
          <div class="heatmap-stats">
            <div class="heatmap-stat">
              <div class="heatmap-stat-label">Clics totaux</div>
              <div class="heatmap-stat-value clicks">${clicksCount}</div>
            </div>
            <div class="heatmap-stat">
              <div class="heatmap-stat-label">Mouvements</div>
              <div class="heatmap-stat-value moves">${movesCount}</div>
            </div>
            <div class="heatmap-stat">
              <div class="heatmap-stat-label">Scroll events</div>
              <div class="heatmap-stat-value scroll">${scrollsCount}</div>
            </div>
          </div>
        </div>
        
        <div class="heatmap-export">
          <button class="heatmap-export-btn" id="heatmap-export-png">
            📥 Export PNG
          </button>
          <button class="heatmap-export-btn" id="heatmap-export-pdf">
            📄 Export PDF
          </button>
        </div>
      </div>
    `;

    this.attachListeners();
  }

  /**
   * Attache les listeners
   */
  attachListeners() {
    // Layer toggles
    document.querySelectorAll('.heatmap-layer-toggle').forEach(toggle => {
      toggle.addEventListener('click', () => {
        const layer = toggle.dataset.layer;
        
        if (layer === 'clicks') {
          this.layers.clicks.enabled = !this.layers.clicks.enabled;
        } else if (layer === 'moves') {
          this.layers.mouseMoves.enabled = !this.layers.mouseMoves.enabled;
        } else if (layer === 'scroll') {
          this.layers.scroll.enabled = !this.layers.scroll.enabled;
        }
        
        const checkbox = toggle.querySelector('.heatmap-layer-checkbox');
        checkbox.classList.toggle('checked');
        
        this.generateHeatmap();
      });
    });

    // Intensity sliders
    const clicksSlider = document.getElementById('clicks-intensity-slider');
    clicksSlider?.addEventListener('input', (e) => {
      const value = parseFloat(e.target.value) / 100;
      this.layers.clicks.intensity = value;
      document.getElementById('clicks-intensity-value').textContent = Math.round(value * 100) + '%';

      // Mark layer as dirty (needs re-rendering)
      this.dirtyLayers.clicks = true;
      this.generateHeatmap();
    });

    const movesSlider = document.getElementById('moves-intensity-slider');
    movesSlider?.addEventListener('input', (e) => {
      const value = parseFloat(e.target.value) / 100;
      this.layers.mouseMoves.intensity = value;
      document.getElementById('moves-intensity-value').textContent = Math.round(value * 100) + '%';

      // Mark layer as dirty (needs re-rendering)
      this.dirtyLayers.mouseMoves = true;
      this.generateHeatmap();
    });

    const scrollSlider = document.getElementById('scroll-intensity-slider');
    scrollSlider?.addEventListener('input', (e) => {
      const value = parseFloat(e.target.value) / 100;
      this.layers.scroll.intensity = value;
      document.getElementById('scroll-intensity-value').textContent = Math.round(value * 100) + '%';

      // Mark layer as dirty (needs re-rendering)
      this.dirtyLayers.scroll = true;
      this.generateHeatmap();
    });

    // Export PNG
    document.getElementById('heatmap-export-png')?.addEventListener('click', () => {
      this.exportHeatmapPNG();
    });

    // Export PDF
    document.getElementById('heatmap-export-pdf')?.addEventListener('click', () => {
      this.exportHeatmapPDF();
    });

    // Thumbnails
    document.querySelectorAll('.heatmap-thumbnail').forEach(thumb => {
      thumb.addEventListener('click', () => {
        const index = parseInt(thumb.dataset.index);
        this.changeScreenshot(index);
      });
    });
  }

  /**
   * Configure le canvas
   */
  setupCanvas() {
    this.canvas = document.getElementById('heatmap-canvas');
    if (!this.canvas) return;

    const baseImg = document.getElementById('heatmap-base');
    if (!baseImg || !baseImg.complete) return;

    const container = this.canvas.parentElement;
    const containerRect = container.getBoundingClientRect();

    this.canvas.width = containerRect.width;
    this.canvas.height = containerRect.height;

    this.ctx = this.canvas.getContext('2d');

    // Create offscreen canvases for each layer (performance optimization)
    this.setupOffscreenLayers(containerRect.width, containerRect.height);
  }

  /**
   * Configure les offscreen canvas layers pour le cache
   * @private
   */
  setupOffscreenLayers(width, height) {
    ['clicks', 'mouseMoves', 'scroll'].forEach(layerName => {
      // Create new offscreen canvas
      const offscreenCanvas = document.createElement('canvas');
      offscreenCanvas.width = width;
      offscreenCanvas.height = height;

      this.offscreenLayers[layerName] = {
        canvas: offscreenCanvas,
        ctx: offscreenCanvas.getContext('2d'),
        cached: false
      };

      // Mark layer as dirty (needs rendering)
      this.dirtyLayers[layerName] = true;
    });
  }

  /**
   * Obtient les dimensions de l'image affichée
   */
  getImageDimensions() {
    const baseImg = document.getElementById('heatmap-base');
    if (!baseImg || !baseImg.complete) {
      return {
        offsetX: 0,
        offsetY: 0,
        displayedWidth: this.canvas.width,
        displayedHeight: this.canvas.height,
        imgNaturalWidth: 1920,
        imgNaturalHeight: 1080
      };
    }

    const imgNaturalWidth = baseImg.naturalWidth;
    const imgNaturalHeight = baseImg.naturalHeight;
    const imgNaturalRatio = imgNaturalWidth / imgNaturalHeight;

    const containerWidth = this.canvas.width;
    const containerHeight = this.canvas.height;
    const containerRatio = containerWidth / containerHeight;

    let displayedWidth, displayedHeight, offsetX, offsetY;

    if (imgNaturalRatio > containerRatio) {
      displayedWidth = containerWidth;
      displayedHeight = containerWidth / imgNaturalRatio;
      offsetX = 0;
      offsetY = (containerHeight - displayedHeight) / 2;
    } else {
      displayedWidth = containerHeight * imgNaturalRatio;
      displayedHeight = containerHeight;
      offsetX = (containerWidth - displayedWidth) / 2;
      offsetY = 0;
    }

    return {
      offsetX,
      offsetY,
      displayedWidth,
      displayedHeight,
      imgNaturalWidth,
      imgNaturalHeight
    };
  }

  /**
   * Génère la heatmap avec cache offscreen
   */
  generateHeatmap() {
    if (!this.canvas || !this.ctx) {
      this.setupCanvas();
      if (!this.canvas || !this.ctx) return;
    }

    const startTime = performance.now();

    // Filtrer les événements pour le screenshot actuel
    const eventsForScreen = this.getEventsForCurrentScreenshot();

    // Render dirty layers to offscreen canvases (cache system)
    this.renderDirtyLayers(eventsForScreen);

    // Composite all enabled layers onto main canvas (fast)
    this.compositeLayers();

    const duration = performance.now() - startTime;
    console.log(`[Heatmap] Generated in ${duration.toFixed(2)}ms (cached layers optimization)`);
  }

  /**
   * Récupère les événements pour le screenshot actuel
   * @private
   * @returns {Array} Événements filtrés
   */
  getEventsForCurrentScreenshot() {
    if (this.session.screenshots && this.session.screenshots.length > 0) {
      const currentScreenshot = this.session.screenshots[this.currentScreenshotIndex];
      const nextScreenshot = this.session.screenshots[this.currentScreenshotIndex + 1];

      const startTime = currentScreenshot.timestamp;
      const endTime = nextScreenshot ? nextScreenshot.timestamp : this.session.metadata.endTime || Infinity;

      const filtered = (this.session.data?.events || []).filter(event => {
        const eventTime = event.timestamp;
        return eventTime >= startTime && eventTime < endTime;
      });

      console.log(`[Heatmap] Screenshot ${this.currentScreenshotIndex + 1}: ${filtered.length} events (${startTime} - ${endTime})`);
      return filtered;
    } else {
      console.warn('[Heatmap] No screenshot timing info, using all events');
      return this.session.events || this.session.data?.events || [];
    }
  }

  /**
   * Render seulement les layers marqués dirty dans les offscreen canvases
   * @private
   */
  renderDirtyLayers(eventsForScreen) {
    // Clicks layer
    if (this.dirtyLayers.clicks && this.offscreenLayers.clicks.ctx) {
      const layerCtx = this.offscreenLayers.clicks.ctx;
      const layerCanvas = this.offscreenLayers.clicks.canvas;

      // Clear offscreen canvas
      layerCtx.clearRect(0, 0, layerCanvas.width, layerCanvas.height);

      // Render clicks to offscreen canvas
      this.drawClicksToLayer(eventsForScreen, layerCtx);

      // Mark as cached
      this.offscreenLayers.clicks.cached = true;
      this.dirtyLayers.clicks = false;
      console.log('[Heatmap] Clicks layer rendered to offscreen canvas');
    }

    // Mouse moves layer
    if (this.dirtyLayers.mouseMoves && this.offscreenLayers.mouseMoves.ctx) {
      const layerCtx = this.offscreenLayers.mouseMoves.ctx;
      const layerCanvas = this.offscreenLayers.mouseMoves.canvas;

      layerCtx.clearRect(0, 0, layerCanvas.width, layerCanvas.height);
      this.drawMouseMovesToLayer(eventsForScreen, layerCtx);

      this.offscreenLayers.mouseMoves.cached = true;
      this.dirtyLayers.mouseMoves = false;
      console.log('[Heatmap] Mouse moves layer rendered to offscreen canvas');
    }

    // Scroll layer
    if (this.dirtyLayers.scroll && this.offscreenLayers.scroll.ctx) {
      const layerCtx = this.offscreenLayers.scroll.ctx;
      const layerCanvas = this.offscreenLayers.scroll.canvas;

      layerCtx.clearRect(0, 0, layerCanvas.width, layerCanvas.height);
      this.drawScrollToLayer(eventsForScreen, layerCtx);

      this.offscreenLayers.scroll.cached = true;
      this.dirtyLayers.scroll = false;
      console.log('[Heatmap] Scroll layer rendered to offscreen canvas');
    }
  }

  /**
   * Composite les layers cached sur le canvas principal
   * @private
   */
  compositeLayers() {
    // Clear main canvas
    this.ctx.clearRect(0, 0, this.canvas.width, this.canvas.height);

    // Composite enabled layers (fast drawImage from cached offscreen)
    if (this.layers.clicks.enabled && this.offscreenLayers.clicks.cached) {
      this.ctx.globalAlpha = 1.0; // Intensity managed in layer rendering
      this.ctx.drawImage(this.offscreenLayers.clicks.canvas, 0, 0);
    }

    if (this.layers.mouseMoves.enabled && this.offscreenLayers.mouseMoves.cached) {
      this.ctx.globalAlpha = 1.0;
      this.ctx.drawImage(this.offscreenLayers.mouseMoves.canvas, 0, 0);
    }

    if (this.layers.scroll.enabled && this.offscreenLayers.scroll.cached) {
      this.ctx.globalAlpha = 1.0;
      this.ctx.drawImage(this.offscreenLayers.scroll.canvas, 0, 0);
    }

    // Reset alpha
    this.ctx.globalAlpha = 1.0;
  }

  /**
   * Dessine les clics dans un layer offscreen
   * @private
   */
  drawClicksToLayer(eventsForScreen, ctx) {
    const clicks = eventsForScreen.filter(e =>
      e.type === 'click' || e.type === 'rage_click'
    );

    const { offsetX, offsetY, displayedWidth, displayedHeight, imgNaturalWidth, imgNaturalHeight } = this.getImageDimensions();

    clicks.forEach(click => {
      const x = offsetX + (click.x / imgNaturalWidth) * displayedWidth;
      const y = offsetY + (click.y / imgNaturalHeight) * displayedHeight;

      // Contraste amélioré - opacité plus forte
      const intensity = click.type === 'rage_click' ? 1 : 0.8;
      const radius = 60 * this.layers.clicks.intensity;

      const gradient = ctx.createRadialGradient(x, y, 0, x, y, radius);
      gradient.addColorStop(0, `rgba(255, 50, 50, ${intensity * this.layers.clicks.intensity})`);
      gradient.addColorStop(0.4, `rgba(255, 0, 0, ${intensity * this.layers.clicks.intensity * 0.7})`);
      gradient.addColorStop(1, 'rgba(255, 0, 0, 0)');

      ctx.fillStyle = gradient;
      ctx.fillRect(x - radius, y - radius, radius * 2, radius * 2);
    });
  }

  /**
   * Dessine les mouvements de souris dans un layer offscreen
   * @private
   */
  drawMouseMovesToLayer(eventsForScreen, ctx) {
    const moves = eventsForScreen.filter(e => e.type === 'mouse_move');
    if (moves.length === 0) return;

    const { offsetX, offsetY, displayedWidth, displayedHeight, imgNaturalWidth, imgNaturalHeight } = this.getImageDimensions();
    const intensity = this.layers.mouseMoves.intensity;

    // Dessiner les trails - contraste amélioré
    ctx.strokeStyle = `rgba(59, 130, 246, ${0.6 * intensity})`;
    ctx.lineWidth = 3 * intensity;
    ctx.lineCap = 'round';
    ctx.lineJoin = 'round';

    ctx.beginPath();
    moves.forEach((move, index) => {
      const x = offsetX + (move.x / imgNaturalWidth) * displayedWidth;
      const y = offsetY + (move.y / imgNaturalHeight) * displayedHeight;

      if (index === 0) {
        ctx.moveTo(x, y);
      } else {
        ctx.lineTo(x, y);
      }
    });
    ctx.stroke();

    // Points de densité - contraste amélioré
    const heatPoints = this.groupMovesByProximity(moves, 20);

    heatPoints.forEach(point => {
      const x = offsetX + (point.x / imgNaturalWidth) * displayedWidth;
      const y = offsetY + (point.y / imgNaturalHeight) * displayedHeight;
      const heatIntensity = Math.min(point.count / 5, 1); // Seuil réduit pour plus de visibilité

      const gradient = ctx.createRadialGradient(x, y, 0, x, y, 30 * intensity);
      gradient.addColorStop(0, `rgba(59, 130, 246, ${0.8 * heatIntensity * intensity})`);
      gradient.addColorStop(0.5, `rgba(59, 130, 246, ${0.5 * heatIntensity * intensity})`);
      gradient.addColorStop(1, 'rgba(59, 130, 246, 0)');

      ctx.fillStyle = gradient;
      ctx.beginPath();
      ctx.arc(x, y, 30 * intensity, 0, Math.PI * 2);
      ctx.fill();
    });
  }

  /**
   * Dessine le scroll dans un layer offscreen - Style Hotjar avec zones de pourcentage
   * @private
   */
  drawScrollToLayer(eventsForScreen, ctx) {
    const scrolls = eventsForScreen.filter(e => e.type === 'scroll');
    if (scrolls.length === 0) return;

    const { offsetX, offsetY, displayedWidth, displayedHeight } = this.getImageDimensions();
    const intensity = this.layers.scroll.intensity;

    // Style Hotjar: diviser en zones et calculer le % d'atteinte
    const numZones = 10; // 10 zones de 10%
    const zoneHeight = displayedHeight / numZones;
    const maxScrollPercentages = [];

    // Trouver le scroll max pour chaque événement
    scrolls.forEach(scroll => {
      const scrollPercentage = scroll.position?.percentage ?? scroll.scrollPercentage;
      if (scrollPercentage !== undefined) {
        maxScrollPercentages.push(scrollPercentage);
      }
    });

    // Si pas de données de pourcentage, utiliser l'ancienne méthode
    if (maxScrollPercentages.length === 0) {
      scrolls.forEach(scroll => {
        const scrollY = scroll.position?.y ?? scroll.y ?? scroll.scrollY;
        if (scrollY !== undefined) {
          // Estimer le pourcentage basé sur scrollY
          const estimatedPercent = Math.min((scrollY / 5000) * 100, 100);
          maxScrollPercentages.push(estimatedPercent);
        }
      });
    }

    // Calculer le max scroll atteint
    const maxReached = Math.max(...maxScrollPercentages, 0);

    // Dessiner les zones avec gradient de couleur (vert→jaune→rouge)
    for (let i = 0; i < numZones; i++) {
      const zoneStart = i * 10;
      const zoneEnd = (i + 1) * 10;
      const zoneY = offsetY + (i * zoneHeight);

      // Calculer combien ont atteint cette zone
      const reachedCount = maxScrollPercentages.filter(p => p >= zoneStart).length;
      const percentage = maxScrollPercentages.length > 0
        ? Math.round((reachedCount / maxScrollPercentages.length) * 100)
        : (zoneStart === 0 ? 100 : 0);

      // Couleur basée sur le pourcentage (vert=100%, jaune=50%, rouge=0%)
      let r, g, b;
      if (percentage >= 50) {
        // Vert → Jaune
        const t = (percentage - 50) / 50;
        r = Math.round(255 * (1 - t));
        g = 200;
        b = 50;
      } else {
        // Jaune → Rouge
        const t = percentage / 50;
        r = 255;
        g = Math.round(200 * t);
        b = 50;
      }

      // Dessiner la zone sur toute la largeur du canvas (pas seulement l'image)
      const alpha = 0.3 * intensity;
      ctx.fillStyle = `rgba(${r}, ${g}, ${b}, ${alpha})`;
      ctx.fillRect(0, zoneY, this.canvas.width, zoneHeight);

      // Ligne de séparation entre zones sur toute la largeur
      ctx.fillStyle = `rgba(${r}, ${g}, ${b}, ${0.6 * intensity})`;
      ctx.fillRect(0, zoneY + zoneHeight - 2, this.canvas.width, 2);

      // Afficher le pourcentage à gauche du canvas
      ctx.font = `bold ${14 * intensity}px sans-serif`;
      ctx.fillStyle = `rgba(255, 255, 255, ${0.9 * intensity})`;
      ctx.textAlign = 'left';
      ctx.textBaseline = 'middle';

      // Background pour le texte
      const text = `${percentage}%`;
      const textWidth = ctx.measureText(text).width;
      const textX = 10; // Toujours à gauche du canvas
      const textY = zoneY + zoneHeight / 2;

      ctx.fillStyle = `rgba(0, 0, 0, ${0.7 * intensity})`;
      ctx.fillRect(textX - 4, textY - 10, textWidth + 12, 20);

      ctx.fillStyle = `rgba(255, 255, 255, ${0.95 * intensity})`;
      ctx.fillText(text, textX, textY);
    }

    // Ligne horizontale indiquant le max scroll atteint sur toute la largeur
    if (maxReached > 0) {
      const maxY = offsetY + (maxReached / 100) * displayedHeight;
      ctx.strokeStyle = `rgba(255, 255, 255, ${0.8 * intensity})`;
      ctx.lineWidth = 3;
      ctx.setLineDash([10, 5]);
      ctx.beginPath();
      ctx.moveTo(0, maxY);
      ctx.lineTo(this.canvas.width, maxY);
      ctx.stroke();
      ctx.setLineDash([]);

      // Label pour le max
      ctx.font = `bold ${12 * intensity}px sans-serif`;
      ctx.fillStyle = `rgba(0, 0, 0, ${0.8 * intensity})`;
      ctx.fillRect(5, maxY - 18, 80, 16);
      ctx.fillStyle = `rgba(255, 255, 255, ${0.95 * intensity})`;
      ctx.textAlign = 'left';
      ctx.fillText(`Max: ${Math.round(maxReached)}%`, 8, maxY - 10);
    }
  }

  /**
   * Dessine la heatmap des clics (legacy - kept for compatibility)
   * @deprecated Use drawClicksToLayer instead
   */
  drawClicksHeatmap(eventsForScreen) {
    const clicks = eventsForScreen.filter(e => 
      e.type === 'click' || e.type === 'rage_click'
    );

    const { offsetX, offsetY, displayedWidth, displayedHeight, imgNaturalWidth, imgNaturalHeight } = this.getImageDimensions();

    clicks.forEach(click => {
      const x = offsetX + (click.x / imgNaturalWidth) * displayedWidth;
      const y = offsetY + (click.y / imgNaturalHeight) * displayedHeight;
      
      const intensity = click.type === 'rage_click' ? 1 : 0.5;
      const radius = 50 * this.layers.clicks.intensity;
      
      const gradient = this.ctx.createRadialGradient(x, y, 0, x, y, radius);
      gradient.addColorStop(0, `rgba(255, 0, 0, ${intensity * this.layers.clicks.intensity})`);
      gradient.addColorStop(0.5, `rgba(255, 0, 0, ${intensity * this.layers.clicks.intensity * 0.5})`);
      gradient.addColorStop(1, 'rgba(255, 0, 0, 0)');
      
      this.ctx.fillStyle = gradient;
      this.ctx.fillRect(x - radius, y - radius, radius * 2, radius * 2);
    });
  }

  /**
   * Dessine la heatmap des mouvements de souris (TRAIL comme Hotjar)
   */
  drawMouseMoveHeatmap(eventsForScreen) {
    const moves = eventsForScreen.filter(e => e.type === 'mouse_move');
    if (moves.length === 0) return;

    const { offsetX, offsetY, displayedWidth, displayedHeight, imgNaturalWidth, imgNaturalHeight } = this.getImageDimensions();
    const intensity = this.layers.mouseMoves.intensity;

    // Dessiner les trails entre chaque mouvement consécutif
    this.ctx.strokeStyle = `rgba(59, 130, 246, ${0.3 * intensity})`;
    this.ctx.lineWidth = 2 * intensity;
    this.ctx.lineCap = 'round';
    this.ctx.lineJoin = 'round';

    this.ctx.beginPath();
    moves.forEach((move, index) => {
      const x = offsetX + (move.x / imgNaturalWidth) * displayedWidth;
      const y = offsetY + (move.y / imgNaturalHeight) * displayedHeight;

      if (index === 0) {
        this.ctx.moveTo(x, y);
      } else {
        this.ctx.lineTo(x, y);
      }
    });
    this.ctx.stroke();

    // Ajouter des points de densité sur les zones où la souris reste
    const heatPoints = this.groupMovesByProximity(moves, 20);
    
    heatPoints.forEach(point => {
      const x = offsetX + (point.x / imgNaturalWidth) * displayedWidth;
      const y = offsetY + (point.y / imgNaturalHeight) * displayedHeight;
      const heatIntensity = Math.min(point.count / 10, 1);

      const gradient = this.ctx.createRadialGradient(x, y, 0, x, y, 20 * intensity);
      gradient.addColorStop(0, `rgba(59, 130, 246, ${0.4 * heatIntensity * intensity})`);
      gradient.addColorStop(0.5, `rgba(59, 130, 246, ${0.2 * heatIntensity * intensity})`);
      gradient.addColorStop(1, 'rgba(59, 130, 246, 0)');

      this.ctx.fillStyle = gradient;
      this.ctx.beginPath();
      this.ctx.arc(x, y, 20 * intensity, 0, Math.PI * 2);
      this.ctx.fill();
    });
  }

  /**
   * Regroupe les mouvements proches pour créer des points de chaleur
   */
  groupMovesByProximity(moves, threshold) {
    const points = [];

    moves.forEach(move => {
      // Chercher un point existant proche
      let found = false;
      for (const point of points) {
        const distance = Math.sqrt(
          Math.pow(move.x - point.x, 2) + 
          Math.pow(move.y - point.y, 2)
        );
        
        if (distance < threshold) {
          point.count++;
          found = true;
          break;
        }
      }

      if (!found) {
        points.push({ x: move.x, y: move.y, count: 1 });
      }
    });

    return points;
  }

  /**
   * Dessine la heatmap du scroll
   */
  drawScrollHeatmap(eventsForScreen) {
    const scrolls = eventsForScreen.filter(e => e.type === 'scroll');
    
    if (scrolls.length === 0) return;
    
    const { offsetX, offsetY, displayedWidth, displayedHeight } = this.getImageDimensions();
    const intensity = this.layers.scroll.intensity;
    
    scrolls.forEach(scroll => {
      const scrollY = scroll.position?.y ?? scroll.y ?? scroll.scrollY;
      const scrollPercentage = scroll.position?.percentage ?? scroll.scrollPercentage;
      
      if (scrollY === undefined && scrollPercentage === undefined) {
        return;
      }
      
      let y;
      if (scrollPercentage !== undefined) {
        y = offsetY + (scrollPercentage / 100) * displayedHeight;
      } else {
        y = offsetY + (scrollY / 10000) * displayedHeight;
      }
      
      const x = offsetX + (displayedWidth / 2);
      
      const gradient = this.ctx.createRadialGradient(x, y, 0, x, y, 40 * intensity);
      gradient.addColorStop(0, `rgba(139, 92, 246, ${0.4 * intensity})`);
      gradient.addColorStop(1, 'rgba(139, 92, 246, 0)');
      
      this.ctx.fillStyle = gradient;
      this.ctx.fillRect(x - 40 * intensity, y - 40 * intensity, 80 * intensity, 80 * intensity);
      
      // Ligne horizontale subtile
      this.ctx.globalAlpha = 0.2 * intensity;
      this.ctx.fillStyle = '#8b5cf6';
      this.ctx.fillRect(offsetX, y, displayedWidth, 3);
    });
    
    this.ctx.globalAlpha = 1;
  }

  /**
   * Exporte la heatmap en PNG
   */
  exportHeatmapPNG() {
    if (!this.canvas) return;

    // Créer un canvas temporaire avec l'image de base + heatmap
    const tempCanvas = document.createElement('canvas');
    tempCanvas.width = this.canvas.width;
    tempCanvas.height = this.canvas.height;
    const tempCtx = tempCanvas.getContext('2d');

    // Dessiner l'image de base
    if (this.baseImage) {
      tempCtx.drawImage(this.baseImage, 0, 0, tempCanvas.width, tempCanvas.height);
    }

    // Dessiner la heatmap par-dessus
    tempCtx.drawImage(this.canvas, 0, 0);

    // Télécharger
    tempCanvas.toBlob((blob) => {
      const url = URL.createObjectURL(blob);
      const a = document.createElement('a');
      a.href = url;
      a.download = `heatmap-${this.session.id}-screenshot-${this.currentScreenshotIndex + 1}.png`;
      a.click();
      URL.revokeObjectURL(url);
    });
  }

  /**
   * Exporte la heatmap en PDF
   */
  async exportHeatmapPDF() {
    try {
      console.log('[Heatmap] Exporting to PDF...');

      const filename = await pdfExport.exportHeatmap(
        this.session,
        this.currentScreenshotIndex
      );

      pdfExport.showNotification(`PDF exporté: ${filename}`, 'success');

    } catch (error) {
      console.error('[Heatmap] Error exporting PDF:', error);
      pdfExport.showNotification('Erreur lors de l\'export PDF', 'error');
    }
  }

  /**
   * Nettoie les ressources (OPTIM #4: free memory)
   */
  cleanup() {
    // OPTIM #4: Libérer la mémoire des screenshots loaded
    if (this.session && this.session.screenshots && this.session.screenshots.length > 0) {
      let freedCount = 0;
      this.session.screenshots.forEach(screenshot => {
        if (screenshot.loaded && screenshot.dataUrl) {
          // Libérer le dataUrl de la mémoire
          screenshot.dataUrl = null;
          screenshot.loaded = false;
          freedCount++;
        }
      });

      if (freedCount > 0) {
        console.log(`[Heatmap] Cleanup: freed ${freedCount} screenshots from memory`);
      }
    }

    // Clear canvas contexts
    if (this.ctx) {
      this.ctx.clearRect(0, 0, this.canvas.width, this.canvas.height);
    }

    // Clear offscreen layers
    ['clicks', 'mouseMoves', 'scroll'].forEach(layerName => {
      const layer = this.offscreenLayers[layerName];
      if (layer && layer.ctx) {
        layer.ctx.clearRect(0, 0, layer.canvas.width, layer.canvas.height);
        layer.cached = false;
      }
    });
  }
}

// Export singleton
export default new HeatmapManager();
