/**
 * SUMO UX Tracker - Funnel UI Controller
 * Gère l'interface de gestion des funnels
 */

import funnelManager from './funnel-manager.js';
import { getAllPresetFunnels, getPresetFunnelsByCategory } from './preset-funnels.js';

class FunnelUIController {
  constructor() {
    this.currentView = 'presets';
    this.editingFunnelId = null;
    this.stepCounter = 0;
  }

  /**
   * Initialise l'UI des funnels
   */
  init() {
    console.log('[FunnelUI] Initializing funnel UI...');

    // Charger les presets dans le FunnelManager
    this.loadPresetFunnels();

    // Attacher les événements
    this.attachEvents();

    // Afficher les presets
    this.renderPresetFunnels();
  }

  /**
   * Charge les funnels prédéfinis dans le FunnelManager
   */
  loadPresetFunnels() {
    const presets = getAllPresetFunnels();
    presets.forEach(preset => {
      // Vérifier si le preset n'est pas déjà chargé
      if (!funnelManager.getFunnel(preset.id)) {
        funnelManager.defineFunnel(preset);
      }
    });
    console.log('[FunnelUI] Loaded', presets.length, 'preset funnels');
  }

  /**
   * Attache les événements UI
   */
  attachEvents() {
    // Bouton "Funnels" dans le header
    const btnManageFunnels = document.getElementById('btn-manage-funnels');
    if (btnManageFunnels) {
      btnManageFunnels.addEventListener('click', () => this.showFunnelsManagement());
    }

    // Bouton retour
    const btnBackFunnels = document.getElementById('btn-back-funnels');
    if (btnBackFunnels) {
      btnBackFunnels.addEventListener('click', () => this.hideFunnelsManagement());
    }

    // Tabs dans la vue gestion
    document.querySelectorAll('.funnel-tab-btn').forEach(btn => {
      btn.addEventListener('click', (e) => {
        const tab = e.target.dataset.funnelTab;
        this.switchFunnelTab(tab);
      });
    });

    // Bouton créer funnel
    const btnCreateFunnel = document.getElementById('btn-create-funnel');
    if (btnCreateFunnel) {
      btnCreateFunnel.addEventListener('click', () => this.showCreateFunnelModal());
    }

    // Modal créer funnel
    const btnCreateFunnelCancel = document.getElementById('btn-create-funnel-cancel');
    if (btnCreateFunnelCancel) {
      btnCreateFunnelCancel.addEventListener('click', () => this.hideCreateFunnelModal());
    }

    const btnCreateFunnelSubmit = document.getElementById('btn-create-funnel-submit');
    if (btnCreateFunnelSubmit) {
      btnCreateFunnelSubmit.addEventListener('click', () => this.handleCreateFunnel());
    }

    // Bouton ajouter étape
    const btnAddStep = document.getElementById('btn-add-step');
    if (btnAddStep) {
      btnAddStep.addEventListener('click', () => this.addStepToForm());
    }
  }

  /**
   * Affiche la vue de gestion des funnels
   */
  showFunnelsManagement() {
    document.getElementById('view-sessions').style.display = 'none';
    document.getElementById('view-analysis').style.display = 'none';
    document.getElementById('view-comparison').style.display = 'none';
    document.getElementById('view-funnels-management').style.display = 'block';

    this.renderPresetFunnels();
    this.renderCustomFunnels();
    this.renderFunnelAnalytics();
  }

  /**
   * Masque la vue de gestion des funnels
   */
  hideFunnelsManagement() {
    document.getElementById('view-funnels-management').style.display = 'none';
    document.getElementById('view-sessions').style.display = 'flex';
  }

  /**
   * Switch entre les tabs de la vue funnels
   */
  switchFunnelTab(tabName) {
    // Mettre à jour les boutons
    document.querySelectorAll('.funnel-tab-btn').forEach(btn => {
      btn.classList.toggle('active', btn.dataset.funnelTab === tabName);
    });

    // Mettre à jour les contenus
    document.querySelectorAll('.funnel-tab-content').forEach(content => {
      content.classList.remove('active');
    });
    document.getElementById(`funnels-tab-${tabName}`).classList.add('active');

    this.currentView = tabName;
  }

  /**
   * Affiche les funnels prédéfinis
   */
  renderPresetFunnels() {
    const presetsList = document.getElementById('presets-list');
    if (!presetsList) return;

    const presets = getAllPresetFunnels();

    // Grouper par catégorie
    const categories = {
      onboarding: [],
      workflow: [],
      training: [],
      support: []
    };

    presets.forEach(preset => {
      if (categories[preset.category]) {
        categories[preset.category].push(preset);
      }
    });

    let html = '';

    Object.keys(categories).forEach(category => {
      const funnels = categories[category];
      if (funnels.length === 0) return;

      const categoryNames = {
        onboarding: 'Onboarding',
        workflow: 'Workflow',
        training: 'Formation',
        support: 'Support'
      };

      html += `
        <div class="funnel-category">
          <h3 class="category-title">${categoryNames[category]}</h3>
          <div class="funnel-cards">
      `;

      funnels.forEach(funnel => {
        html += this.renderFunnelCard(funnel, true);
      });

      html += `
          </div>
        </div>
      `;
    });

    presetsList.innerHTML = html;
  }

  /**
   * Affiche les funnels personnalisés
   */
  renderCustomFunnels() {
    const customList = document.getElementById('custom-funnels-list');
    if (!customList) return;

    const allFunnels = funnelManager.getAllFunnels();
    const customFunnels = allFunnels.filter(f => !f.id.startsWith('preset_'));

    if (customFunnels.length === 0) {
      customList.innerHTML = `
        <div class="empty-state">
          <div class="empty-icon">🎯</div>
          <p>Aucun funnel personnalisé</p>
          <button class="btn-primary" onclick="window.funnelUI.showCreateFunnelModal()">
            + Créer mon premier funnel
          </button>
        </div>
      `;
      return;
    }

    let html = '<div class="funnel-cards">';
    customFunnels.forEach(funnel => {
      html += this.renderFunnelCard(funnel, false);
    });
    html += '</div>';

    customList.innerHTML = html;
  }

  /**
   * Rend une carte funnel
   */
  renderFunnelCard(funnel, isPreset) {
    const categoryIcons = {
      onboarding: '👋',
      workflow: '⚙️',
      training: '📚',
      support: '🆘',
      general: '📊'
    };

    const icon = categoryIcons[funnel.category] || '📊';

    return `
      <div class="funnel-card" data-funnel-id="${funnel.id}">
        <div class="funnel-card-header">
          <div class="funnel-icon">${icon}</div>
          <div class="funnel-info">
            <h4>${funnel.name}</h4>
            <p class="funnel-description">${funnel.description}</p>
          </div>
          ${!isPreset ? `
            <div class="funnel-actions">
              <button class="btn-icon" onclick="window.funnelUI.editFunnel('${funnel.id}')" title="Modifier">✏️</button>
              <button class="btn-icon" onclick="window.funnelUI.deleteFunnel('${funnel.id}')" title="Supprimer">🗑️</button>
            </div>
          ` : ''}
        </div>
        <div class="funnel-card-body">
          <div class="funnel-steps-preview">
            <strong>${funnel.steps.length} étapes</strong>
            <ol class="steps-mini">
              ${funnel.steps.slice(0, 3).map((step, i) => `
                <li>${step.name}</li>
              `).join('')}
              ${funnel.steps.length > 3 ? `<li class="more">+ ${funnel.steps.length - 3} étapes</li>` : ''}
            </ol>
          </div>
        </div>
        <div class="funnel-card-footer">
          <button class="btn-secondary btn-small" onclick="window.funnelUI.viewFunnelDetails('${funnel.id}')">
            Voir détails
          </button>
          ${isPreset ? `<span class="badge badge-preset">Preset</span>` : `<span class="badge badge-custom">Custom</span>`}
        </div>
      </div>
    `;
  }

  /**
   * Affiche les statistiques des funnels
   */
  renderFunnelAnalytics() {
    const analyticsContainer = document.getElementById('funnels-analytics');
    if (!analyticsContainer) return;

    // TODO: Implémenter les statistiques globales sur tous les funnels
    analyticsContainer.innerHTML = `
      <div class="analytics-placeholder">
        <div class="placeholder-icon">📊</div>
        <h3>Statistiques Funnels</h3>
        <p>Les statistiques globales des funnels seront affichées ici après analyse de plusieurs sessions.</p>
      </div>
    `;
  }

  /**
   * Affiche la modal de création de funnel
   */
  showCreateFunnelModal() {
    this.editingFunnelId = null;
    this.stepCounter = 0;

    // Reset form
    document.getElementById('funnel-name-input').value = '';
    document.getElementById('funnel-description-input').value = '';
    document.getElementById('funnel-category-input').value = 'general';
    document.getElementById('funnel-steps-list').innerHTML = '';

    // Ajouter 2 étapes par défaut
    this.addStepToForm();
    this.addStepToForm();

    document.getElementById('modal-create-funnel').classList.add('active');
  }

  /**
   * Masque la modal de création de funnel
   */
  hideCreateFunnelModal() {
    document.getElementById('modal-create-funnel').classList.remove('active');
  }

  /**
   * Ajoute une étape au formulaire
   */
  addStepToForm() {
    const stepsList = document.getElementById('funnel-steps-list');
    this.stepCounter++;

    const stepHtml = `
      <div class="funnel-step-form" data-step-index="${this.stepCounter}">
        <div class="step-number">${this.stepCounter}</div>
        <div class="step-fields">
          <div class="form-group">
            <label>Nom de l'étape</label>
            <input type="text" class="step-name" placeholder="Ex: Accès formulaire" required>
          </div>
          <div class="form-group">
            <label>Type de détection</label>
            <select class="step-detection-type" onchange="window.funnelUI.toggleStepFields(${this.stepCounter})">
              <option value="url">URL Pattern</option>
              <option value="click">Click Selector</option>
            </select>
          </div>
          <div class="form-group step-url-field">
            <label>URL Pattern</label>
            <input type="text" class="step-url-pattern" placeholder="/signup, /dashboard, etc.">
          </div>
          <div class="form-group step-click-field" style="display: none;">
            <label>CSS Selector</label>
            <input type="text" class="step-click-selector" placeholder="#submit-btn, .confirm-button, etc.">
          </div>
        </div>
        <button class="btn-icon btn-remove-step" onclick="window.funnelUI.removeStep(${this.stepCounter})" title="Supprimer">🗑️</button>
      </div>
    `;

    stepsList.insertAdjacentHTML('beforeend', stepHtml);
  }

  /**
   * Toggle des champs selon le type de détection
   */
  toggleStepFields(stepIndex) {
    const stepForm = document.querySelector(`[data-step-index="${stepIndex}"]`);
    const detectionType = stepForm.querySelector('.step-detection-type').value;

    const urlField = stepForm.querySelector('.step-url-field');
    const clickField = stepForm.querySelector('.step-click-field');

    if (detectionType === 'url') {
      urlField.style.display = 'block';
      clickField.style.display = 'none';
    } else {
      urlField.style.display = 'none';
      clickField.style.display = 'block';
    }
  }

  /**
   * Retire une étape
   */
  removeStep(stepIndex) {
    const stepForm = document.querySelector(`[data-step-index="${stepIndex}"]`);
    if (stepForm) {
      stepForm.remove();
      this.renumberSteps();
    }
  }

  /**
   * Renumérote les étapes après suppression
   */
  renumberSteps() {
    const steps = document.querySelectorAll('.funnel-step-form');
    steps.forEach((step, index) => {
      step.querySelector('.step-number').textContent = index + 1;
    });
  }

  /**
   * Crée un nouveau funnel
   */
  handleCreateFunnel() {
    const name = document.getElementById('funnel-name-input').value.trim();
    const description = document.getElementById('funnel-description-input').value.trim();
    const category = document.getElementById('funnel-category-input').value;

    if (!name) {
      alert('Veuillez saisir un nom pour le funnel');
      return;
    }

    // Extraire les étapes
    const stepForms = document.querySelectorAll('.funnel-step-form');
    const steps = [];

    stepForms.forEach(stepForm => {
      const stepName = stepForm.querySelector('.step-name').value.trim();
      const detectionType = stepForm.querySelector('.step-detection-type').value;

      if (!stepName) return;

      const step = {
        name: stepName,
        detection_type: detectionType
      };

      if (detectionType === 'url') {
        step.url_pattern = stepForm.querySelector('.step-url-pattern').value.trim();
      } else if (detectionType === 'click') {
        step.element_selector = stepForm.querySelector('.step-click-selector').value.trim();
      }

      steps.push(step);
    });

    if (steps.length < 2) {
      alert('Un funnel doit avoir au moins 2 étapes');
      return;
    }

    // Créer le funnel
    try {
      const funnel = funnelManager.defineFunnel({
        name,
        description,
        category,
        steps
      });

      console.log('[FunnelUI] Funnel créé:', funnel.id);
      this.hideCreateFunnelModal();
      this.renderCustomFunnels();
      this.switchFunnelTab('custom');
    } catch (error) {
      alert('Erreur lors de la création du funnel: ' + error.message);
    }
  }

  /**
   * Supprime un funnel
   */
  deleteFunnel(funnelId) {
    if (!confirm('Voulez-vous vraiment supprimer ce funnel ?')) return;

    funnelManager.deleteFunnel(funnelId);
    this.renderCustomFunnels();
  }

  /**
   * Édite un funnel
   */
  editFunnel(funnelId) {
    // TODO: Implémenter l'édition
    alert('Édition non implémentée pour le moment');
  }

  /**
   * Affiche les détails d'un funnel
   */
  viewFunnelDetails(funnelId) {
    const funnel = funnelManager.getFunnel(funnelId);
    if (!funnel) return;

    // Créer une modal pour afficher les détails
    const detailsHtml = `
      <div class="funnel-details-modal">
        <h3>🎯 ${funnel.name}</h3>
        <p>${funnel.description}</p>
        <div class="funnel-steps-detailed">
          <h4>Étapes (${funnel.steps.length})</h4>
          <ol>
            ${funnel.steps.map((step, i) => `
              <li>
                <strong>${step.name}</strong>
                <div class="step-detection">
                  ${step.detection_type === 'url' ?
                    `<span class="badge">URL</span> ${step.url_pattern}` :
                    `<span class="badge">Click</span> ${step.element_selector}`
                  }
                </div>
              </li>
            `).join('')}
          </ol>
        </div>
      </div>
    `;

    // Pour l'instant, utiliser alert (TODO: créer une vraie modal)
    alert(`${funnel.name}\n\n${funnel.description}\n\nÉtapes:\n${funnel.steps.map((s, i) => `${i+1}. ${s.name}`).join('\n')}`);
  }

  /**
   * Affiche les funnels détectés pour une session
   */
  renderSessionFunnels(sessionId, funnelResults) {
    const container = document.getElementById('funnels-container');
    if (!container) return;

    if (!funnelResults || funnelResults.length === 0) {
      container.innerHTML = `
        <div class="empty-state">
          <div class="empty-icon">🎯</div>
          <h3>Aucun funnel détecté</h3>
          <p>Cette session ne correspond à aucun funnel défini.</p>
        </div>
      `;
      return;
    }

    let html = '<div class="session-funnels-results">';

    funnelResults.forEach(result => {
      const completionColor = result.completed ? 'success' : (result.completion_rate > 50 ? 'warning' : 'danger');

      html += `
        <div class="funnel-result ${completionColor}">
          <div class="funnel-result-header">
            <h3>${result.funnel_name}</h3>
            <div class="funnel-status">
              ${result.completed ?
                '<span class="badge badge-success">✅ Complété</span>' :
                '<span class="badge badge-abandoned">❌ Abandonné</span>'
              }
            </div>
          </div>

          <div class="funnel-result-body">
            <div class="completion-bar">
              <div class="completion-fill" style="width: ${result.completion_rate}%"></div>
              <span class="completion-text">${Math.round(result.completion_rate)}% complété</span>
            </div>

            <div class="funnel-steps-timeline">
              ${result.steps_completed.map(step => `
                <div class="timeline-step completed">
                  <div class="step-dot">✓</div>
                  <div class="step-info">
                    <strong>${step.step_name}</strong>
                    <span class="step-time">${new Date(step.timestamp).toLocaleTimeString()}</span>
                  </div>
                </div>
              `).join('')}

              ${result.steps_missed.map(step => `
                <div class="timeline-step missed">
                  <div class="step-dot">○</div>
                  <div class="step-info">
                    <strong>${step.step_name}</strong>
                    <span class="step-time">Non complétée</span>
                  </div>
                </div>
              `).join('')}
            </div>

            ${result.time_to_complete ? `
              <div class="funnel-metric">
                <span class="metric-label">⏱️ Temps de complétion</span>
                <span class="metric-value">${this.formatDuration(result.time_to_complete)}</span>
              </div>
            ` : ''}

            ${result.abandonment_point ? `
              <div class="funnel-metric abandonment">
                <span class="metric-label">🚫 Point d'abandon</span>
                <span class="metric-value">${result.abandonment_point.step_name}</span>
              </div>
            ` : ''}
          </div>
        </div>
      `;
    });

    html += '</div>';
    container.innerHTML = html;
  }

  /**
   * Formate une durée en millisecondes
   */
  formatDuration(ms) {
    const seconds = Math.floor(ms / 1000);
    const minutes = Math.floor(seconds / 60);
    const hours = Math.floor(minutes / 60);

    if (hours > 0) {
      return `${hours}h ${minutes % 60}m`;
    } else if (minutes > 0) {
      return `${minutes}m ${seconds % 60}s`;
    } else {
      return `${seconds}s`;
    }
  }
}

// Export singleton
const funnelUI = new FunnelUIController();
window.funnelUI = funnelUI; // Exposer globalement pour onclick
export default funnelUI;
