/**
 * SUMO UX Tracker - Funnel Manager
 * Gère la définition et l'analyse des tunnels de conversion
 */

class FunnelManager {
  constructor() {
    this.funnels = new Map(); // Funnels définis
    this.funnelResults = new Map(); // Résultats d'analyse par session
  }

  /**
   * Définit un nouveau funnel de conversion
   */
  defineFunnel(funnelConfig) {
    const funnel = {
      id: funnelConfig.id || this.generateFunnelId(),
      name: funnelConfig.name,
      description: funnelConfig.description || '',
      steps: funnelConfig.steps, // Array of step definitions
      createdAt: Date.now(),
      category: funnelConfig.category || 'general' // onboarding, workflow, training, etc.
    };

    // Valider la config
    if (!funnel.name || !funnel.steps || funnel.steps.length < 2) {
      throw new Error('Funnel doit avoir un nom et au moins 2 étapes');
    }

    this.funnels.set(funnel.id, funnel);
    console.log('[FunnelManager] Funnel défini:', funnel.id, funnel.name);

    return funnel;
  }

  /**
   * Analyse une session pour détecter les funnels complétés/abandonnés
   */
  analyzeSessionFunnels(session) {
    const results = [];

    this.funnels.forEach((funnel) => {
      const funnelResult = this.analyzeFunnel(session, funnel);
      if (funnelResult.started) {
        results.push(funnelResult);
      }
    });

    if (results.length > 0) {
      this.funnelResults.set(session.id, results);
    }

    return results;
  }

  /**
   * Analyse un funnel spécifique pour une session
   */
  analyzeFunnel(session, funnel) {
    const events = session.data?.events || [];

    const result = {
      funnel_id: funnel.id,
      funnel_name: funnel.name,
      started: false,
      completed: false,
      abandoned: false,
      steps_completed: [],
      steps_missed: [],
      completion_rate: 0,
      time_to_complete: null,
      abandonment_point: null,
      timestamps: {}
    };

    // Détecter les étapes complétées
    funnel.steps.forEach((step, stepIndex) => {
      const stepCompleted = this.detectStep(events, step);

      if (stepCompleted) {
        result.steps_completed.push({
          step_index: stepIndex,
          step_name: step.name,
          timestamp: stepCompleted.timestamp,
          event_index: stepCompleted.eventIndex
        });

        result.timestamps[step.name] = stepCompleted.timestamp;

        if (stepIndex === 0) {
          result.started = true;
        }
      } else {
        result.steps_missed.push({
          step_index: stepIndex,
          step_name: step.name
        });
      }
    });

    // Calculer taux de complétion
    result.completion_rate = (result.steps_completed.length / funnel.steps.length) * 100;

    // Vérifier si complété (toutes les étapes)
    result.completed = result.steps_completed.length === funnel.steps.length;

    // Vérifier si abandonné (commencé mais pas complété)
    result.abandoned = result.started && !result.completed;

    // Point d'abandon
    if (result.abandoned && result.steps_completed.length > 0) {
      const lastCompletedStep = result.steps_completed[result.steps_completed.length - 1];
      result.abandonment_point = {
        step_name: lastCompletedStep.step_name,
        step_index: lastCompletedStep.step_index
      };
    }

    // Temps pour compléter
    if (result.completed && result.steps_completed.length > 0) {
      const firstStep = result.steps_completed[0];
      const lastStep = result.steps_completed[result.steps_completed.length - 1];
      result.time_to_complete = lastStep.timestamp - firstStep.timestamp;
    }

    return result;
  }

  /**
   * Détecte si une étape a été complétée dans les événements
   */
  detectStep(events, stepDefinition) {
    for (let i = 0; i < events.length; i++) {
      const event = events[i];

      // Vérifier selon le type de détection
      if (stepDefinition.detection_type === 'url') {
        // Détection par URL
        if (this.matchUrl(event, stepDefinition.url_pattern)) {
          return {
            timestamp: event.timestamp,
            eventIndex: i
          };
        }
      } else if (stepDefinition.detection_type === 'click') {
        // Détection par clic sur un élément
        if (event.type === 'click' && this.matchElement(event, stepDefinition.element_selector)) {
          return {
            timestamp: event.timestamp,
            eventIndex: i
          };
        }
      } else if (stepDefinition.detection_type === 'custom') {
        // Détection personnalisée via fonction
        if (stepDefinition.detector && stepDefinition.detector(event)) {
          return {
            timestamp: event.timestamp,
            eventIndex: i
          };
        }
      }
    }

    return null;
  }

  /**
   * Vérifie si une URL correspond au pattern
   */
  matchUrl(event, urlPattern) {
    if (event.type === 'navigation' || event.subtype === 'url_change') {
      const url = event.to || event.url;

      if (typeof urlPattern === 'string') {
        return url && url.includes(urlPattern);
      } else if (urlPattern instanceof RegExp) {
        return urlPattern.test(url);
      }
    }

    return false;
  }

  /**
   * Vérifie si un élément correspond au sélecteur
   */
  matchElement(event, selector) {
    if (!selector || !event.target) return false;

    // Simple matching par ID ou classe
    if (selector.startsWith('#')) {
      return event.target.id === selector.substring(1);
    } else if (selector.startsWith('.')) {
      const className = selector.substring(1);
      return event.target.className && event.target.className.includes(className);
    } else {
      // Tag name
      return event.target.tagName && event.target.tagName.toLowerCase() === selector.toLowerCase();
    }
  }

  /**
   * Génère un ID unique pour un funnel
   */
  generateFunnelId() {
    return `funnel_${Date.now()}_${Math.random().toString(36).substr(2, 9)}`;
  }

  /**
   * Récupère tous les funnels définis
   */
  getAllFunnels() {
    return Array.from(this.funnels.values());
  }

  /**
   * Récupère un funnel par ID
   */
  getFunnel(funnelId) {
    return this.funnels.get(funnelId);
  }

  /**
   * Supprime un funnel
   */
  deleteFunnel(funnelId) {
    return this.funnels.delete(funnelId);
  }

  /**
   * Exporte les funnels définis
   */
  exportFunnels() {
    return {
      funnels: Array.from(this.funnels.values()),
      exportedAt: Date.now()
    };
  }

  /**
   * Importe des funnels
   */
  importFunnels(funnelsData) {
    if (!funnelsData || !funnelsData.funnels) {
      throw new Error('Format de données invalide');
    }

    funnelsData.funnels.forEach((funnel) => {
      this.funnels.set(funnel.id, funnel);
    });

    console.log('[FunnelManager] Imported', funnelsData.funnels.length, 'funnels');
  }
}

// Export singleton
const funnelManager = new FunnelManager();
export default funnelManager;
