/**
 * SUMO UX Tracker - Folders Manager
 * Gère la sidebar folders, drag & drop, et modals
 */

import db from './db.js';

class FoldersManager {
  constructor() {
    this.folders = [];
    this.currentFolderId = null;
    this.selectedFolderId = null;
    this.draggedSessionId = null;
    this.elements = {};
  }

  /**
   * Initialise le folders manager
   */
  async init() {
    console.log('[FoldersManager] Initializing...');

    // Cache elements DOM
    this.cacheElements();

    // Charger les dossiers depuis DB
    await this.loadFolders();

    // Attacher les event listeners
    this.attachEventListeners();

    // Afficher les dossiers dans la sidebar
    this.renderFolders();

    console.log('[FoldersManager] Initialized with', this.folders.length, 'folders');
  }

  /**
   * Cache les éléments DOM
   */
  cacheElements() {
    this.elements.sidebar = document.getElementById('folders-sidebar');
    this.elements.foldersList = document.getElementById('folders-list');
    this.elements.btnNewFolder = document.getElementById('btn-new-folder');
    this.elements.btnToggleSidebar = document.getElementById('btn-toggle-sidebar');

    // Modals
    this.elements.modalCreateFolder = document.getElementById('modal-create-folder');
    this.elements.modalRenameSession = document.getElementById('modal-rename-session');
    this.elements.folderNameInput = document.getElementById('folder-name-input');
    this.elements.folderColorInput = document.getElementById('folder-color-input');
    this.elements.sessionTitleInput = document.getElementById('session-title-input');

    console.log('[FoldersManager] Elements cached');
  }

  /**
   * Charge les dossiers depuis la DB
   */
  async loadFolders() {
    try {
      this.folders = await db.getAllFolders();
      console.log('[FoldersManager] Loaded', this.folders.length, 'folders');
    } catch (error) {
      console.error('[FoldersManager] Error loading folders:', error);
      this.folders = [];
    }
  }

  /**
   * Attache les event listeners
   */
  attachEventListeners() {
    // Bouton nouveau dossier
    this.elements.btnNewFolder?.addEventListener('click', () => {
      this.openCreateFolderModal();
    });

    // Bouton toggle sidebar
    this.elements.btnToggleSidebar?.addEventListener('click', () => {
      this.toggleSidebar();
    });

    // Modal create folder - Submit
    const btnCreateFolder = document.getElementById('btn-create-folder-submit');
    btnCreateFolder?.addEventListener('click', () => {
      this.createFolder();
    });

    // Modal create folder - Cancel
    const btnCancelFolder = document.getElementById('btn-create-folder-cancel');
    btnCancelFolder?.addEventListener('click', () => {
      this.closeCreateFolderModal();
    });

    // Modal rename session - Submit
    const btnRenameSubmit = document.getElementById('btn-rename-session-submit');
    btnRenameSubmit?.addEventListener('click', () => {
      this.renameSession();
    });

    // Modal rename session - Cancel
    const btnRenameCancel = document.getElementById('btn-rename-session-cancel');
    btnRenameCancel?.addEventListener('click', () => {
      this.closeRenameSessionModal();
    });

    // Fermer modals avec Escape
    document.addEventListener('keydown', (e) => {
      if (e.key === 'Escape') {
        this.closeCreateFolderModal();
        this.closeRenameSessionModal();
      }
    });

    // Fermer modals en cliquant sur le backdrop
    this.elements.modalCreateFolder?.addEventListener('click', (e) => {
      if (e.target === this.elements.modalCreateFolder) {
        this.closeCreateFolderModal();
      }
    });

    this.elements.modalRenameSession?.addEventListener('click', (e) => {
      if (e.target === this.elements.modalRenameSession) {
        this.closeRenameSessionModal();
      }
    });

    console.log('[FoldersManager] Event listeners attached');
  }

  /**
   * Affiche les dossiers dans la sidebar
   */
  renderFolders() {
    if (!this.elements.foldersList) return;

    // Vider la liste
    this.elements.foldersList.innerHTML = '';

    // Ajouter l'option "Toutes les sessions"
    const allSessionsItem = this.createFolderItem(null, 'Toutes les sessions', null, true);
    this.elements.foldersList.appendChild(allSessionsItem);

    // Ajouter séparateur
    const separator = document.createElement('div');
    separator.className = 'folders-separator';
    this.elements.foldersList.appendChild(separator);

    // Ajouter chaque dossier
    this.folders.forEach(folder => {
      const folderItem = this.createFolderItem(folder.id, folder.name, folder.color, false);
      this.elements.foldersList.appendChild(folderItem);
    });

    console.log('[FoldersManager] Rendered', this.folders.length, 'folders');
  }

  /**
   * Crée un élément de dossier
   */
  createFolderItem(folderId, name, color, isAll = false) {
    const item = document.createElement('div');
    item.className = 'folder-item';
    item.dataset.folderId = folderId || 'all';

    // Active si c'est le dossier sélectionné
    if (folderId === this.selectedFolderId || (isAll && !this.selectedFolderId)) {
      item.classList.add('active');
    }

    // Icône avec couleur
    const icon = document.createElement('div');
    icon.className = 'folder-icon';
    if (color) {
      icon.style.backgroundColor = color;
    }
    icon.textContent = isAll ? '📁' : '📂';

    // Nom
    const nameEl = document.createElement('div');
    nameEl.className = 'folder-name';
    nameEl.textContent = name;

    // Nombre de sessions (à charger)
    const count = document.createElement('div');
    count.className = 'folder-count';
    count.textContent = '0';

    // Boutons d'actions (sauf pour "Toutes les sessions")
    if (!isAll) {
      const actions = document.createElement('div');
      actions.className = 'folder-actions';

      // Bouton renommer
      const btnRename = document.createElement('button');
      btnRename.className = 'folder-action-btn';
      btnRename.innerHTML = '✏️';
      btnRename.title = 'Renommer';
      btnRename.addEventListener('click', (e) => {
        e.stopPropagation();
        this.renameFolderPrompt(folderId);
      });

      // Bouton supprimer
      const btnDelete = document.createElement('button');
      btnDelete.className = 'folder-action-btn';
      btnDelete.innerHTML = '🗑️';
      btnDelete.title = 'Supprimer';
      btnDelete.addEventListener('click', (e) => {
        e.stopPropagation();
        this.deleteFolderPrompt(folderId);
      });

      actions.appendChild(btnRename);
      actions.appendChild(btnDelete);
      item.appendChild(actions);
    }

    item.appendChild(icon);
    item.appendChild(nameEl);
    item.appendChild(count);

    // Click sur le dossier pour filtrer
    item.addEventListener('click', () => {
      this.selectFolder(folderId);
    });

    // Drag & drop sur le dossier
    if (!isAll) {
      item.addEventListener('dragover', (e) => {
        e.preventDefault();
        item.classList.add('drag-over');
      });

      item.addEventListener('dragleave', () => {
        item.classList.remove('drag-over');
      });

      item.addEventListener('drop', (e) => {
        e.preventDefault();
        item.classList.remove('drag-over');
        this.handleDrop(folderId);
      });
    }

    return item;
  }

  /**
   * Sélectionne un dossier et filtre les sessions
   */
  selectFolder(folderId) {
    this.selectedFolderId = folderId;

    // Mettre à jour l'UI
    document.querySelectorAll('.folder-item').forEach(item => {
      item.classList.remove('active');
    });

    const selectedItem = document.querySelector(`[data-folder-id="${folderId || 'all'}"]`);
    if (selectedItem) {
      selectedItem.classList.add('active');
    }

    // Notifier sessions manager de filtrer
    if (window.sumoApp && window.sumoApp.sessionsManager) {
      window.sumoApp.sessionsManager.filterByFolder(folderId);
    }

    console.log('[FoldersManager] Selected folder:', folderId || 'all');
  }

  /**
   * Toggle sidebar
   */
  toggleSidebar() {
    this.elements.sidebar?.classList.toggle('collapsed');

    // Sauvegarder état dans localStorage
    const isCollapsed = this.elements.sidebar?.classList.contains('collapsed');
    localStorage.setItem('sumo-folders-sidebar-collapsed', isCollapsed);
  }

  /**
   * Ouvre le modal de création de dossier
   */
  openCreateFolderModal() {
    this.elements.folderNameInput.value = '';
    this.elements.folderColorInput.value = this.getRandomColor();
    this.elements.modalCreateFolder?.classList.add('active');
    this.elements.folderNameInput?.focus();
  }

  /**
   * Ferme le modal de création de dossier
   */
  closeCreateFolderModal() {
    this.elements.modalCreateFolder?.classList.remove('active');
  }

  /**
   * Crée un nouveau dossier
   */
  async createFolder() {
    const name = this.elements.folderNameInput.value.trim();
    const color = this.elements.folderColorInput.value;

    if (!name) {
      alert('Veuillez entrer un nom de dossier');
      return;
    }

    try {
      const folder = await db.createFolder(name, color);
      console.log('[FoldersManager] Folder created:', folder);

      // Ajouter à la liste
      this.folders.push(folder);

      // Re-render
      this.renderFolders();

      // Sélectionner le nouveau dossier
      this.selectFolder(folder.id);

      // Fermer modal
      this.closeCreateFolderModal();

    } catch (error) {
      console.error('[FoldersManager] Error creating folder:', error);
      alert('Erreur lors de la création du dossier');
    }
  }

  /**
   * Demande confirmation pour renommer un dossier
   */
  async renameFolderPrompt(folderId) {
    const folder = this.folders.find(f => f.id === folderId);
    if (!folder) return;

    const newName = prompt('Nouveau nom du dossier:', folder.name);
    if (!newName || newName === folder.name) return;

    try {
      await db.updateFolder(folderId, { name: newName });
      folder.name = newName;
      this.renderFolders();
      console.log('[FoldersManager] Folder renamed:', folderId);
    } catch (error) {
      console.error('[FoldersManager] Error renaming folder:', error);
      alert('Erreur lors du renommage');
    }
  }

  /**
   * Demande confirmation pour supprimer un dossier
   */
  async deleteFolderPrompt(folderId) {
    const folder = this.folders.find(f => f.id === folderId);
    if (!folder) return;

    const confirm = window.confirm(`Supprimer le dossier "${folder.name}" ?\n\nLes sessions ne seront pas supprimées, seulement déplacées vers "Toutes les sessions".`);
    if (!confirm) return;

    try {
      // Supprimer le dossier
      await db.deleteFolder(folderId);

      // Retirer toutes les sessions de ce dossier
      const allMetadata = await db.getAllSessionsMetadata();
      for (const [sessionId, metadata] of allMetadata.entries()) {
        if (metadata.folderId === folderId) {
          await db.saveSessionMetadata(sessionId, {
            ...metadata,
            folderId: null
          });
        }
      }

      // Retirer de la liste
      this.folders = this.folders.filter(f => f.id !== folderId);

      // Re-render
      this.renderFolders();

      // Sélectionner "Toutes les sessions"
      this.selectFolder(null);

      console.log('[FoldersManager] Folder deleted:', folderId);

    } catch (error) {
      console.error('[FoldersManager] Error deleting folder:', error);
      alert('Erreur lors de la suppression');
    }
  }

  /**
   * Ouvre le modal de renommage de session
   */
  openRenameSessionModal(sessionId, currentTitle) {
    this.currentSessionId = sessionId;
    this.elements.sessionTitleInput.value = currentTitle || '';
    this.elements.modalRenameSession?.classList.add('active');
    this.elements.sessionTitleInput?.focus();
  }

  /**
   * Ferme le modal de renommage
   */
  closeRenameSessionModal() {
    this.elements.modalRenameSession?.classList.remove('active');
    this.currentSessionId = null;
  }

  /**
   * Renomme une session
   */
  async renameSession() {
    const customTitle = this.elements.sessionTitleInput.value.trim();

    if (!customTitle) {
      alert('Veuillez entrer un titre');
      return;
    }

    try {
      // Récupérer metadata existante
      const existingMetadata = await db.getSessionMetadata(this.currentSessionId);

      // Sauvegarder avec nouveau titre
      await db.saveSessionMetadata(this.currentSessionId, {
        ...existingMetadata,
        customTitle: customTitle
      });

      console.log('[FoldersManager] Session renamed:', this.currentSessionId);

      // Fermer modal
      this.closeRenameSessionModal();

      // Recharger les sessions
      if (window.sumoApp && window.sumoApp.sessionsManager) {
        window.sumoApp.sessionsManager.loadSessions();
      }

    } catch (error) {
      console.error('[FoldersManager] Error renaming session:', error);
      alert('Erreur lors du renommage');
    }
  }

  /**
   * Gère le drop d'une session sur un dossier
   */
  async handleDrop(folderId) {
    if (!this.draggedSessionId) return;

    try {
      // Récupérer metadata existante
      const existingMetadata = await db.getSessionMetadata(this.draggedSessionId);

      // Sauvegarder avec nouveau dossier
      await db.saveSessionMetadata(this.draggedSessionId, {
        ...existingMetadata,
        folderId: folderId
      });

      console.log('[FoldersManager] Session moved to folder:', this.draggedSessionId, '->', folderId);

      // Recharger les sessions
      if (window.sumoApp && window.sumoApp.sessionsManager) {
        window.sumoApp.sessionsManager.loadSessions();
      }

      // Mettre à jour le compteur
      this.updateFolderCounts();

    } catch (error) {
      console.error('[FoldersManager] Error moving session:', error);
      alert('Erreur lors du déplacement');
    }

    this.draggedSessionId = null;
  }

  /**
   * Démarre le drag d'une session
   */
  startDragSession(sessionId) {
    this.draggedSessionId = sessionId;
    console.log('[FoldersManager] Drag started:', sessionId);
  }

  /**
   * Met à jour les compteurs de sessions par dossier
   */
  async updateFolderCounts() {
    try {
      const allMetadata = await db.getAllSessionsMetadata();

      // Compter par dossier
      const counts = {};
      let totalCount = 0;

      for (const [sessionId, metadata] of allMetadata.entries()) {
        totalCount++;
        const folderId = metadata.folderId || 'all';
        counts[folderId] = (counts[folderId] || 0) + 1;
      }

      // Mettre à jour l'UI
      document.querySelectorAll('.folder-item').forEach(item => {
        const folderId = item.dataset.folderId;
        const countEl = item.querySelector('.folder-count');

        if (folderId === 'all') {
          countEl.textContent = totalCount;
        } else {
          countEl.textContent = counts[folderId] || 0;
        }
      });

    } catch (error) {
      console.error('[FoldersManager] Error updating counts:', error);
    }
  }

  /**
   * Génère une couleur aléatoire pour un dossier
   */
  getRandomColor() {
    const colors = [
      '#3b82f6', // Blue
      '#10b981', // Green
      '#f59e0b', // Orange
      '#ef4444', // Red
      '#8b5cf6', // Purple
      '#ec4899', // Pink
      '#14b8a6', // Teal
      '#f97316'  // Orange
    ];
    return colors[Math.floor(Math.random() * colors.length)];
  }

  /**
   * Toggle la sidebar (collapse/expand)
   */
  toggleSidebar() {
    if (!this.elements.sidebar) return;

    const isCollapsed = this.elements.sidebar.classList.contains('collapsed');

    if (isCollapsed) {
      // Expand
      this.elements.sidebar.classList.remove('collapsed');
      if (this.elements.btnToggleSidebar) {
        this.elements.btnToggleSidebar.innerHTML = '◀';
        this.elements.btnToggleSidebar.title = 'Masquer';
      }
    } else {
      // Collapse
      this.elements.sidebar.classList.add('collapsed');
      if (this.elements.btnToggleSidebar) {
        this.elements.btnToggleSidebar.innerHTML = '▶';
        this.elements.btnToggleSidebar.title = 'Afficher les dossiers';
      }
    }

    console.log('[FoldersManager] Sidebar toggled:', isCollapsed ? 'expanded' : 'collapsed');
  }

  /**
   * Retourne le dossier sélectionné
   */
  getSelectedFolder() {
    return this.selectedFolderId;
  }

  /**
   * Retourne tous les dossiers
   */
  getFolders() {
    return this.folders;
  }
}

// Export singleton
const foldersManager = new FoldersManager();
export default foldersManager;
