/**
 * SUMO UX Tracker - Feedback Viewer
 * Visualisation des annotations de feedback dans l'application web
 */

export default {
  currentSession: null,
  currentFeedback: null,
  feedbackOverlay: null,
  canvas: null,
  ctx: null,

  /**
   * Initialise le viewer avec la session
   */
  init(session) {
    console.log('🎨 Initialisation Feedback Viewer', session);
    this.currentSession = session;
    this.render();
  },

  /**
   * Affiche la liste des feedbacks
   */
  render() {
    const container = document.getElementById('feedback-container');
    if (!container) {
      console.error('[Feedback Viewer] Container not found');
      return;
    }

    console.log('[Feedback Viewer] Session:', this.currentSession);
    const feedbacks = this.currentSession.feedbacks || [];
    console.log('[Feedback Viewer] Feedbacks found:', feedbacks.length);

    if (feedbacks.length === 0) {
      container.innerHTML = `
        <div class="feedback-empty">
          <div class="empty-icon">📝</div>
          <h3>Aucun feedback disponible</h3>
          <p>Cette session ne contient pas d'annotations de feedback.</p>
          <p style="margin-top: 16px; font-size: 14px; color: #868e96;">
            <strong>Debug:</strong> Session ID: ${this.currentSession.id || 'N/A'}<br>
            Feedbacks array: ${Array.isArray(feedbacks) ? 'Yes' : 'No'}
          </p>
        </div>
      `;
      return;
    }

    // Liste des feedbacks
    const feedbackList = feedbacks.map((feedback, index) => {
      const date = new Date(feedback.timestamp);
      const annotationCount = feedback.annotations?.length || 0;
      const postitCount = feedback.annotations?.filter(a => a.type === 'postit').length || 0;
      const frustrationCount = feedback.annotations?.filter(a => a.isFrustration).length || 0;

      return `
        <div class="feedback-card" data-index="${index}">
          <div class="feedback-card-header">
            <div class="feedback-card-title">
              <span class="feedback-icon">📝</span>
              <span>Feedback #${index + 1}</span>
            </div>
            <span class="feedback-date">${this.formatDate(date)}</span>
          </div>
          <div class="feedback-card-body">
            <div class="feedback-url">
              <i class="fas fa-link"></i>
              ${this.truncateUrl(feedback.url)}
            </div>
            <div class="feedback-stats">
              <span class="stat">
                <i class="fas fa-pen"></i>
                ${annotationCount} annotation${annotationCount > 1 ? 's' : ''}
              </span>
              ${postitCount > 0 ? `
                <span class="stat">
                  <i class="fas fa-sticky-note"></i>
                  ${postitCount} note${postitCount > 1 ? 's' : ''}
                </span>
              ` : ''}
              ${frustrationCount > 0 ? `
                <span class="stat stat-frustration">
                  <i class="fas fa-exclamation-triangle"></i>
                  ${frustrationCount} pain point${frustrationCount > 1 ? 's' : ''}
                </span>
              ` : ''}
            </div>
          </div>
          <div class="feedback-card-actions">
            <button class="btn-view-feedback" onclick="window.feedbackViewer.viewFeedback(${index})">
              <i class="fas fa-eye"></i>
              Visualiser
            </button>
          </div>
        </div>
      `;
    }).join('');

    container.innerHTML = `
      <div class="feedback-list">
        ${feedbackList}
      </div>
    `;
  },

  /**
   * Affiche un feedback spécifique
   */
  viewFeedback(index) {
    const feedbacks = this.currentSession.feedbacks || [];
    if (!feedbacks[index]) return;

    this.currentFeedback = feedbacks[index];
    this.showFeedbackModal();
  },

  /**
   * Affiche la modal de visualisation
   */
  showFeedbackModal() {
    // Créer la modal
    const modal = document.createElement('div');
    modal.id = 'feedback-modal';
    modal.className = 'feedback-modal';

    const annotations = this.currentFeedback.annotations || [];
    const postitList = annotations
      .filter(a => a.type === 'postit')
      .map((postit, index) => `
        <div class="feedback-postit-item ${postit.isFrustration ? 'frustration' : ''}">
          <div class="postit-header">
            <span class="postit-icon">${postit.isFrustration ? '😤' : '📝'}</span>
            <span class="postit-label">${postit.isFrustration ? 'Pain Point' : 'Note'} #${index + 1}</span>
          </div>
          <div class="postit-text">${this.escapeHtml(postit.text)}</div>
        </div>
      `).join('');

    modal.innerHTML = `
      <div class="feedback-modal-overlay" onclick="window.feedbackViewer.closeFeedbackModal()"></div>
      <div class="feedback-modal-content">
        <div class="feedback-modal-header">
          <h3>
            <span class="feedback-icon">📝</span>
            Visualisation du Feedback
          </h3>
          <button class="btn-close-modal" onclick="window.feedbackViewer.closeFeedbackModal()">
            <i class="fas fa-times"></i>
          </button>
        </div>

        <div class="feedback-modal-body">
          <div class="feedback-info">
            <div class="info-item">
              <span class="info-label">URL :</span>
              <span class="info-value">${this.escapeHtml(this.currentFeedback.url)}</span>
            </div>
            <div class="info-item">
              <span class="info-label">Date :</span>
              <span class="info-value">${this.formatDate(new Date(this.currentFeedback.timestamp))}</span>
            </div>
            <div class="info-item">
              <span class="info-label">Viewport :</span>
              <span class="info-value">${this.currentFeedback.viewport.width}x${this.currentFeedback.viewport.height}</span>
            </div>
          </div>

          ${postitList ? `
            <div class="feedback-postits-section">
              <h4>
                <i class="fas fa-sticky-note"></i>
                Notes et Pain Points
              </h4>
              <div class="feedback-postits-list">
                ${postitList}
              </div>
            </div>
          ` : ''}

          ${this.currentFeedback.screenshot ? `
            <div class="feedback-screenshot-section">
              <h4>
                <i class="fas fa-camera"></i>
                Screenshot de la page
              </h4>
              <div class="feedback-screenshot-wrapper">
                <img src="${this.currentFeedback.screenshot}" alt="Screenshot" id="feedback-screenshot">
              </div>
            </div>
          ` : ''}

          <div class="feedback-canvas-section">
            <h4>
              <i class="fas fa-palette"></i>
              Annotations Visuelles
            </h4>
            <div class="feedback-canvas-wrapper">
              <canvas id="feedback-canvas"></canvas>
            </div>
          </div>
        </div>

        <div class="feedback-modal-footer">
          <button class="btn-secondary" onclick="window.feedbackViewer.closeFeedbackModal()">
            Fermer
          </button>
        </div>
      </div>
    `;

    document.body.appendChild(modal);

    // Dessiner les annotations sur le canvas
    this.renderAnnotationsCanvas();
  },

  /**
   * Dessine les annotations sur le canvas
   */
  renderAnnotationsCanvas() {
    const canvas = document.getElementById('feedback-canvas');
    if (!canvas) return;

    const viewport = this.currentFeedback.viewport;
    const annotations = this.currentFeedback.annotations || [];

    // Définir la taille du canvas
    const maxWidth = 800;
    const scale = Math.min(1, maxWidth / viewport.width);

    canvas.width = viewport.width * scale;
    canvas.height = viewport.height * scale;

    const ctx = canvas.getContext('2d');
    ctx.scale(scale, scale);

    // Fond blanc
    ctx.fillStyle = '#FFFFFF';
    ctx.fillRect(0, 0, viewport.width, viewport.height);

    // Dessiner chaque annotation
    annotations.forEach(annotation => {
      this.drawAnnotation(ctx, annotation);
    });
  },

  /**
   * Dessine une annotation
   */
  drawAnnotation(ctx, annotation) {
    ctx.strokeStyle = annotation.color || '#FF0000';
    ctx.lineWidth = annotation.strokeWidth || 3;
    ctx.lineCap = 'round';
    ctx.lineJoin = 'round';

    switch (annotation.type) {
      case 'drawing':
        this.drawPath(ctx, annotation.points);
        break;

      case 'arrow':
        this.drawArrow(ctx, annotation);
        break;

      case 'highlight':
        this.drawHighlight(ctx, annotation);
        break;

      case 'postit':
        this.drawPostit(ctx, annotation);
        break;
    }
  },

  /**
   * Dessine un tracé libre
   */
  drawPath(ctx, points) {
    if (!points || points.length < 2) return;

    ctx.beginPath();
    ctx.moveTo(points[0].x, points[0].y);

    for (let i = 1; i < points.length; i++) {
      ctx.lineTo(points[i].x, points[i].y);
    }

    ctx.stroke();
  },

  /**
   * Dessine une flèche
   */
  drawArrow(ctx, arrow) {
    const {startX, startY, endX, endY} = arrow;

    // Ligne principale
    ctx.beginPath();
    ctx.moveTo(startX, startY);
    ctx.lineTo(endX, endY);
    ctx.stroke();

    // Pointe de la flèche
    const headLength = 20;
    const angle = Math.atan2(endY - startY, endX - startX);

    ctx.beginPath();
    ctx.moveTo(endX, endY);
    ctx.lineTo(
      endX - headLength * Math.cos(angle - Math.PI / 6),
      endY - headLength * Math.sin(angle - Math.PI / 6)
    );
    ctx.moveTo(endX, endY);
    ctx.lineTo(
      endX - headLength * Math.cos(angle + Math.PI / 6),
      endY - headLength * Math.sin(angle + Math.PI / 6)
    );
    ctx.stroke();
  },

  /**
   * Dessine une zone de surbrillance
   */
  drawHighlight(ctx, highlight) {
    const {startX, startY, endX, endY, color} = highlight;

    // Rectangle avec fill semi-transparent
    ctx.fillStyle = color + '30';
    ctx.fillRect(startX, startY, endX - startX, endY - startY);

    // Bordure
    ctx.strokeRect(startX, startY, endX - startX, endY - startY);
  },

  /**
   * Dessine un post-it sur le canvas
   */
  drawPostit(ctx, postit) {
    const {x, y, width, height, text, color, isFrustration} = postit;

    // Fond
    ctx.fillStyle = color;
    ctx.fillRect(x, y, width, height);

    // Bordure
    ctx.strokeStyle = isFrustration ? '#FF6B6B' : '#CCC';
    ctx.lineWidth = 2;
    ctx.strokeRect(x, y, width, height);

    // Icône
    ctx.font = '18px Arial';
    ctx.fillText(isFrustration ? '😤' : '📝', x + 10, y + 25);

    // Texte
    ctx.fillStyle = '#000';
    ctx.font = '14px Arial';
    ctx.textAlign = 'left';
    ctx.textBaseline = 'top';

    // Découpage du texte en lignes
    const words = text.split(' ');
    const lines = [];
    let currentLine = '';

    words.forEach(word => {
      const testLine = currentLine + word + ' ';
      const metrics = ctx.measureText(testLine);

      if (metrics.width > width - 50) {
        lines.push(currentLine);
        currentLine = word + ' ';
      } else {
        currentLine = testLine;
      }
    });
    lines.push(currentLine);

    // Affichage des lignes
    lines.forEach((line, i) => {
      ctx.fillText(line.trim(), x + 40, y + 15 + (i * 20));
    });
  },

  /**
   * Ferme la modal
   */
  closeFeedbackModal() {
    const modal = document.getElementById('feedback-modal');
    if (modal) {
      modal.remove();
    }
  },

  /**
   * Formate une date
   */
  formatDate(date) {
    const options = {
      year: 'numeric',
      month: 'long',
      day: 'numeric',
      hour: '2-digit',
      minute: '2-digit'
    };
    return date.toLocaleDateString('fr-FR', options);
  },

  /**
   * Tronque une URL
   */
  truncateUrl(url) {
    if (url.length <= 60) return url;
    return url.substring(0, 57) + '...';
  },

  /**
   * Échappe le HTML
   */
  escapeHtml(text) {
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
  }
};
