/**
 * SUMO UX Tracker - Extension Share Module (VERSION COMPLÈTE)
 * Partage de l'extension Chrome avec génération réelle du ZIP
 */

class ExtensionShareManager {
  constructor() {
    this.extensionFiles = null;
    this.isGenerating = false;
    this.projectFiles = [
      // Fichiers racine
      'manifest.json',
      'constants.js',
      'export-manager.js',
      'jszip_min.js',
      
      // Background
      'background.js',
      
      // Popup
      'popup.html',
      'popup.js',
      'popup.css',
      
      // Content
      'content-main.js',
      'click-tracker.js',
      'form-tracker.js',
      'mouse-tracker.js',
      'navigation-tracker.js',
      'patterns-analyzer.js',
      'pattern-utils.js',
      'performance-monitor.js',
      'screenshot-handler.js',
      'screenshot-overlay.js',
      'screenshot-tracker.js',
      'scroll-tracker.js',
      'time-tracker.js',
      
      // Assets
      'sumo16.png',
      'sumo32.png',
      'sumo48.png',
      'sumo128.png'
    ];
  }

  /**
   * Affiche le modal de partage
   */
  showShareModal() {
    const modal = this.createShareModal();
    document.body.appendChild(modal);
    
    // Fermer au clic sur overlay
    modal.addEventListener('click', (e) => {
      if (e.target === modal) {
        this.closeModal();
      }
    });
  }

  /**
   * Crée le modal HTML
   */
  createShareModal() {
    const modal = document.createElement('div');
    modal.className = 'share-modal-overlay';
    modal.innerHTML = `
      <div class="share-modal">
        <div class="share-modal-header">
          <h2>📦 Partager l'Extension Chrome</h2>
          <button class="share-modal-close" onclick="extensionShare.closeModal()">✕</button>
        </div>
        
        <div class="share-modal-body">
          <p class="share-description">
            Partagez l'extension SUMO UX Tracker avec votre équipe pour qu'ils puissent capturer leurs propres sessions utilisateur.
          </p>

          <div class="share-options">
            <!-- Option 1 : Télécharger -->
            <div class="share-option">
              <div class="share-option-icon">📥</div>
              <div class="share-option-content">
                <h3>Télécharger le ZIP</h3>
                <p>Téléchargez l'extension packagée (.zip) prête à installer dans Chrome</p>
                <button class="btn-primary" onclick="extensionShare.downloadExtension()">
                  📥 Télécharger l'extension (${this.projectFiles.length} fichiers)
                </button>
              </div>
            </div>

            <!-- Option 2 : Email -->
            <div class="share-option">
              <div class="share-option-icon">📧</div>
              <div class="share-option-content">
                <h3>Partager par email</h3>
                <p>Envoyez les instructions d'installation par email (texte pré-rempli)</p>
                <button class="btn-secondary" onclick="extensionShare.shareByEmail()">
                  📧 Préparer l'email
                </button>
              </div>
            </div>
          </div>

          <div class="share-info">
            <h4>ℹ️ Instructions d'installation</h4>
            <ol>
              <li>Télécharger le fichier <code>sumo-extension.zip</code></li>
              <li>Extraire le contenu dans un dossier</li>
              <li>Ouvrir Chrome → <code>chrome://extensions/</code></li>
              <li>Activer le "Mode développeur" (toggle en haut à droite)</li>
              <li>Cliquer "Charger l'extension non empaquetée"</li>
              <li>Sélectionner le dossier extrait</li>
              <li>L'icône SUMO 🍰 apparaît dans la barre Chrome</li>
            </ol>
          </div>

          <div class="share-note" style="margin-top: var(--space-md); padding: var(--space-md); background: #fef3c7; border: 1px solid #fbbf24; border-radius: var(--radius-md); font-size: 0.875rem;">
            <strong>💡 Note :</strong> Cette extension fonctionne sur tous les sites web et les prototypes Figma. Les données sont stockées localement dans le navigateur.
          </div>
        </div>
      </div>
    `;
    
    return modal;
  }

  /**
   * Ferme le modal
   */
  closeModal() {
    const modal = document.querySelector('.share-modal-overlay');
    if (modal) {
      modal.remove();
    }
  }

  /**
   * Télécharge l'extension en ZIP
   */
  async downloadExtension() {
    try {
      console.log('[ExtensionShare] Downloading extension ZIP...');

      // Créer un lien de téléchargement vers le ZIP
      const link = document.createElement('a');
      link.href = 'downloads/sumo-extension-v4.1.zip';
      link.download = 'sumo-extension-v4.1.zip';
      document.body.appendChild(link);
      link.click();
      document.body.removeChild(link);

      // Afficher message de succès
      this.showSuccessMessage("Téléchargement de l'extension démarré ! 📦");

      // Fermer le modal
      this.closeModal();

    } catch (error) {
      console.error('[ExtensionShare] Error:', error);
      alert('Erreur : ' + error.message);
    }
  }

  /**
   * Ouvre l'email pré-rempli
   */
  shareByEmail() {
    const subject = encodeURIComponent('SUMO UX Tracker - Extension Chrome');
    
    const body = encodeURIComponent(
`Bonjour,

Je te partage l'extension Chrome SUMO UX Tracker pour capturer des sessions utilisateur lors de tes tests UX.

🔧 INSTALLATION

1. Télécharge l'extension (je te l'envoie en pièce jointe ou fournis le lien)
2. Extrais le fichier ZIP dans un dossier
3. Ouvre Chrome et va sur chrome://extensions/
4. Active le "Mode développeur" (toggle en haut à droite)
5. Clique sur "Charger l'extension non empaquetée"
6. Sélectionne le dossier extrait
7. L'icône SUMO 🍰 apparaît dans la barre Chrome

📖 UTILISATION

1. Va sur le site ou prototype Figma à tester
2. Clique sur l'icône SUMO dans Chrome
3. Clique "Démarrer l'enregistrement"
4. Effectue tes tests normalement (clics, scroll, navigation...)
5. Clique "Arrêter"
6. Clique "Exporter la session"
7. Un fichier .uxtrack est téléchargé automatiquement

📊 ANALYSE DES SESSIONS

Une fois le fichier .uxtrack exporté :
- Envoie-le moi pour analyse
- Ou importe-le dans l'app web SUMO UX Tracker pour voir :
  • Replay vidéo de la session
  • Heatmaps (clics, mouvements souris, scroll)
  • Analytics détaillées (temps par page, interactions, rage clicks)
  • Détection automatique des problèmes UX

✨ FONCTIONNALITÉS

✅ Capture automatique des interactions
   • Clics avec position et élément cliqué
   • Mouvements de souris
   • Scroll (position et vitesse)
   • Interactions formulaires

✅ Screenshots contextualisés
   • Capture après chaque clic important
   • Screenshot initial et final
   • Délai intelligent pour pages chargées

✅ Détection intelligente
   • Rage clicks (frustrations utilisateur)
   • Temps d'hésitation
   • Patterns de navigation
   • Zones problématiques

✅ Export optimisé
   • Format .uxtrack (ZIP compressé)
   • Taille : <15MB pour 10min de session
   • Compatible avec tous les navigateurs

✅ Mode de capture flexible
   • Zone visible (rapide)
   • Page entière avec html2canvas (plus lent mais complet)
   • Toggle dans l'interface popup

✅ Heatmaps avancées style Hotjar
   • Scrollmap avec zones colorées et pourcentages
   • Heatmap clics améliorée
   • Trajectoire souris haute visibilité

✅ Compatible partout
   • Sites web classiques
   • Applications web (SPA)
   • Prototypes Figma interactifs
   • Pages avec JavaScript

🛡️ CONFIDENTIALITÉ

• Toutes les données restent en local (pas de serveur)
• Aucun tracking externe
• Export manuel uniquement
• Suppression facile des données

💡 BESOIN D'AIDE ?

N'hésite pas si tu as des questions sur l'installation ou l'utilisation !

À bientôt,
[Ton nom]

---
SUMO UX Tracker v4.1.0
Extension Chrome - Capture de sessions UX
`
    );
    
    // Ouvrir le client email
    window.location.href = `mailto:?subject=${subject}&body=${body}`;
    
    // Afficher message
    this.showSuccessMessage("Email préparé ! Ajoute les destinataires et joins le ZIP de l'extension 📧");
    
    // Fermer le modal
    this.closeModal();
  }

  /**
   * Affiche un message de succès
   */
  showSuccessMessage(message) {
    const toast = document.createElement('div');
    toast.className = 'share-toast success';
    toast.textContent = message;
    document.body.appendChild(toast);
    
    setTimeout(() => {
      toast.classList.add('show');
    }, 100);
    
    setTimeout(() => {
      toast.classList.remove('show');
      setTimeout(() => toast.remove(), 300);
    }, 3000);
  }
}

// Export singleton
const extensionShare = new ExtensionShareManager();
export default extensionShare;
