/**
 * SUMO UX Tracker - Database Service
 * Gestion IndexedDB pour stockage sessions (VERSION AMÉLIORÉE)
 */

class DatabaseService {
  constructor() {
    this.dbName = 'sumo-ux-tracker';
    this.dbVersion = 2; // ✅ Incrémentation pour nouveaux stores
    this.db = null;
  }

  /**
   * Initialise la base de données
   */
  async init() {
    return new Promise((resolve, reject) => {
      const request = indexedDB.open(this.dbName, this.dbVersion);

      request.onerror = () => {
        console.error('[DB] Error opening database:', request.error);
        reject(request.error);
      };

      request.onsuccess = () => {
        this.db = request.result;
        console.log('[DB] Database opened successfully');
        resolve(this.db);
      };

      request.onupgradeneeded = (event) => {
        const db = event.target.result;
        const oldVersion = event.oldVersion;

        console.log('[DB] Upgrading database from version', oldVersion, 'to', this.dbVersion);

        // Store pour les sessions (v1)
        if (!db.objectStoreNames.contains('sessions')) {
          const sessionStore = db.createObjectStore('sessions', { keyPath: 'id' });

          // Index pour recherche rapide
          sessionStore.createIndex('importDate', 'importDate', { unique: false });
          sessionStore.createIndex('url', 'metadata.initialUrl', { unique: false });
          sessionStore.createIndex('duration', 'metadata.duration', { unique: false });

          console.log('[DB] Object store "sessions" created');
        }

        // Store pour les screenshots (blob) (v1)
        if (!db.objectStoreNames.contains('screenshots')) {
          const screenshotStore = db.createObjectStore('screenshots', { keyPath: 'id' });
          screenshotStore.createIndex('sessionId', 'sessionId', { unique: false });

          console.log('[DB] Object store "screenshots" created');
        }

        // ✅ NOUVEAUX STORES v2 - Titres personnalisés + Dossiers
        if (oldVersion < 2) {
          // Store pour les dossiers
          if (!db.objectStoreNames.contains('folders')) {
            const folderStore = db.createObjectStore('folders', { keyPath: 'id' });
            folderStore.createIndex('name', 'name', { unique: false });
            folderStore.createIndex('createdAt', 'createdAt', { unique: false });
            console.log('[DB] ✅ Object store "folders" created (v2)');
          }

          // Store pour les métadonnées des sessions (titres, tags, etc.)
          if (!db.objectStoreNames.contains('sessionsMetadata')) {
            const metadataStore = db.createObjectStore('sessionsMetadata', { keyPath: 'sessionId' });
            metadataStore.createIndex('folderId', 'folderId', { unique: false });
            metadataStore.createIndex('customTitle', 'customTitle', { unique: false });
            console.log('[DB] ✅ Object store "sessionsMetadata" created (v2)');
          }
        }
      };
    });
  }

  /**
   * Détecte automatiquement le type de session depuis l'URL
   */
  detectSessionType(url) {
    if (!url) return 'web';
    
    const urlLower = url.toLowerCase();
    
    if (urlLower.includes('figma.com/proto')) {
      return 'figma-prototype';
    }
    if (urlLower.includes('figma.com/design') || urlLower.includes('figma.com/file')) {
      return 'figma-design';
    }
    if (urlLower.includes('figma.com')) {
      return 'figma';
    }
    
    return 'web';
  }

  /**
   * Obtient le label du type de session
   */
  getSessionTypeLabel(type) {
    const labels = {
      'figma-prototype': 'Figma Prototype',
      'figma-design': 'Figma Design',
      'figma': 'Figma',
      'web': 'Web'
    };
    return labels[type] || 'Web';
  }

  /**
   * Sauvegarde une session
   */
  async saveSession(sessionData) {
    return new Promise((resolve, reject) => {
      const transaction = this.db.transaction(['sessions'], 'readwrite');
      const store = transaction.objectStore('sessions');
      
      // Ajouter le sessionType s'il n'existe pas
      if (!sessionData.metadata.sessionType) {
        sessionData.metadata.sessionType = this.detectSessionType(sessionData.metadata.initialUrl);
      }
      
      const request = store.put(sessionData);

      request.onsuccess = () => {
        console.log('[DB] Session saved:', sessionData.id);
        resolve(sessionData.id);
      };

      request.onerror = () => {
        console.error('[DB] Error saving session:', request.error);
        reject(request.error);
      };
    });
  }

  /**
   * Récupère toutes les sessions
   */
  async getAllSessions() {
    return new Promise((resolve, reject) => {
      const transaction = this.db.transaction(['sessions'], 'readonly');
      const store = transaction.objectStore('sessions');
      
      const request = store.getAll();

      request.onsuccess = () => {
        console.log('[DB] Retrieved sessions:', request.result.length);
        resolve(request.result);
      };

      request.onerror = () => {
        console.error('[DB] Error getting sessions:', request.error);
        reject(request.error);
      };
    });
  }

  /**
   * Récupère une session par ID
   */
  async getSession(sessionId) {
    return new Promise((resolve, reject) => {
      const transaction = this.db.transaction(['sessions'], 'readonly');
      const store = transaction.objectStore('sessions');
      
      const request = store.get(sessionId);

      request.onsuccess = () => {
        resolve(request.result);
      };

      request.onerror = () => {
        console.error('[DB] Error getting session:', request.error);
        reject(request.error);
      };
    });
  }

  /**
   * Récupère le PREMIER screenshot d'une session (pour la vignette)
   */
  async getFirstScreenshot(sessionId) {
    return new Promise((resolve, reject) => {
      const transaction = this.db.transaction(['screenshots'], 'readonly');
      const store = transaction.objectStore('screenshots');
      const index = store.index('sessionId');
      
      const request = index.getAll(sessionId);

      request.onsuccess = () => {
        const screenshots = request.result || [];
        
        if (screenshots.length === 0) {
          resolve(null);
          return;
        }

        // Trier par timestamp et prendre le premier
        screenshots.sort((a, b) => (a.timestamp || 0) - (b.timestamp || 0));
        const firstScreenshot = screenshots[0];

        // Convertir en dataUrl
        this.blobToDataUrl(firstScreenshot.blob).then(dataUrl => {
          resolve(dataUrl);
        }).catch(reject);
      };

      request.onerror = () => {
        console.error('[DB] Error getting first screenshot:', request.error);
        reject(request.error);
      };
    });
  }

  /**
   * Récupère une session avec ses screenshots chargés (ANCIEN - eager loading)
   * ⚠️ DEPRECATED: Use getSessionWithScreenshotsLazy() instead for better memory performance
   * Cette fonction charge TOUS les screenshots en mémoire (180 MB pour 50 screenshots)
   */
  async getSessionWithScreenshots(sessionId) {
    // 1. Récupérer la session
    const session = await this.getSession(sessionId);
    if (!session) {
      return null;
    }

    // 2. Récupérer tous les screenshots de cette session
    const screenshots = await this.getSessionScreenshots(sessionId);

    console.log('[DB] Loading screenshots for session:', screenshots.length);

    // 3. Convertir les blobs en dataUrls (❌ EAGER LOADING)
    const screenshotsWithDataUrl = await Promise.all(
      screenshots.map(async (screenshot) => {
        const dataUrl = await this.blobToDataUrl(screenshot.blob);
        return {
          id: screenshot.screenshotId,
          filename: screenshot.screenshotId,
          timestamp: screenshot.timestamp || 0,
          dataUrl: dataUrl,
          type: 'screenshot'
        };
      })
    );

    // 4. Trier par timestamp
    screenshotsWithDataUrl.sort((a, b) => a.timestamp - b.timestamp);

    // 5. Attacher les screenshots à la session
    session.screenshots = screenshotsWithDataUrl;

    console.log('[DB] Session loaded with screenshots:', screenshotsWithDataUrl.length);
    return session;
  }

  /**
   * Récupère une session avec metadata des screenshots (NOUVEAU - lazy loading)
   * OPTIM #4: Ne charge PAS les dataUrls immédiatement (-60% RAM)
   * Les screenshots seront chargés à la demande via loadScreenshotDataUrl()
   */
  async getSessionWithScreenshotsLazy(sessionId) {
    const startTime = performance.now();

    // 1. Récupérer la session
    const session = await this.getSession(sessionId);
    if (!session) {
      return null;
    }

    // 2. Récupérer metadata des screenshots (sans charger les blobs)
    const screenshots = await this.getSessionScreenshots(sessionId);

    // 3. Créer objets metadata SANS dataUrl (lazy loading)
    const screenshotsMetadata = screenshots.map(screenshot => ({
      id: screenshot.screenshotId,
      filename: screenshot.screenshotId,
      timestamp: screenshot.timestamp || 0,
      sessionId: sessionId,
      dataUrl: null, // ✅ Pas de dataUrl = pas de RAM
      loaded: false, // Flag pour tracking
      type: 'screenshot'
    }));

    // 4. Trier par timestamp
    screenshotsMetadata.sort((a, b) => a.timestamp - b.timestamp);

    // 5. Attacher metadata à la session
    session.screenshots = screenshotsMetadata;

    const duration = performance.now() - startTime;
    console.log(`[DB] Session loaded with ${screenshotsMetadata.length} screenshots metadata in ${duration.toFixed(2)}ms (lazy loading)`);

    return session;
  }

  /**
   * Charge le dataUrl d'un screenshot spécifique (on-demand)
   * OPTIM #4: Load screenshot dataUrl seulement quand nécessaire
   * @param {string} sessionId - ID de la session
   * @param {string} screenshotId - ID du screenshot
   * @returns {Promise<string>} dataUrl du screenshot
   */
  async loadScreenshotDataUrl(sessionId, screenshotId) {
    const id = `${sessionId}_${screenshotId}`;

    return new Promise((resolve, reject) => {
      const transaction = this.db.transaction(['screenshots'], 'readonly');
      const store = transaction.objectStore('screenshots');

      const request = store.get(id);

      request.onsuccess = async () => {
        const screenshot = request.result;
        if (!screenshot) {
          reject(new Error(`Screenshot not found: ${id}`));
          return;
        }

        // Convert blob to dataUrl
        try {
          const dataUrl = await this.blobToDataUrl(screenshot.blob);
          resolve(dataUrl);
        } catch (error) {
          reject(error);
        }
      };

      request.onerror = () => {
        reject(request.error);
      };
    });
  }

  /**
   * Récupère tous les screenshots d'une session (NOUVEAU)
   */
  async getSessionScreenshots(sessionId) {
    return new Promise((resolve, reject) => {
      const transaction = this.db.transaction(['screenshots'], 'readonly');
      const store = transaction.objectStore('screenshots');
      const index = store.index('sessionId');
      
      const request = index.getAll(sessionId);

      request.onsuccess = () => {
        resolve(request.result || []);
      };

      request.onerror = () => {
        console.error('[DB] Error getting screenshots:', request.error);
        reject(request.error);
      };
    });
  }

  /**
   * Convertit un Blob en DataURL (NOUVEAU)
   */
  async blobToDataUrl(blob) {
    return new Promise((resolve, reject) => {
      const reader = new FileReader();
      reader.onloadend = () => resolve(reader.result);
      reader.onerror = reject;
      reader.readAsDataURL(blob);
    });
  }

  /**
   * Supprime une session
   */
  async deleteSession(sessionId) {
    return new Promise((resolve, reject) => {
      const transaction = this.db.transaction(['sessions', 'screenshots'], 'readwrite');
      const sessionsStore = transaction.objectStore('sessions');
      const screenshotsStore = transaction.objectStore('screenshots');

      // Supprimer la session
      const deleteSessionRequest = sessionsStore.delete(sessionId);

      deleteSessionRequest.onsuccess = () => {
        // Supprimer les screenshots associés
        const index = screenshotsStore.index('sessionId');
        const screenshotsRequest = index.getAllKeys(sessionId);

        screenshotsRequest.onsuccess = () => {
          const keys = screenshotsRequest.result;
          keys.forEach(key => {
            screenshotsStore.delete(key);
          });
          
          console.log('[DB] Session deleted:', sessionId);
          resolve();
        };
      };

      deleteSessionRequest.onerror = () => {
        console.error('[DB] Error deleting session:', deleteSessionRequest.error);
        reject(deleteSessionRequest.error);
      };
    });
  }

  /**
   * Sauvegarde un screenshot (VERSION CORRIGÉE avec timestamp)
   */
  async saveScreenshot(sessionId, screenshotId, blob, timestamp = Date.now()) {
    return new Promise((resolve, reject) => {
      const transaction = this.db.transaction(['screenshots'], 'readwrite');
      const store = transaction.objectStore('screenshots');

      const data = {
        id: `${sessionId}_${screenshotId}`,
        sessionId: sessionId,
        screenshotId: screenshotId,
        blob: blob,
        timestamp: timestamp  // Utiliser le timestamp fourni
      };

      const request = store.put(data);

      request.onsuccess = () => {
        resolve(data.id);
      };

      request.onerror = () => {
        console.error('[DB] Error saving screenshot:', request.error);
        reject(request.error);
      };
    });
  }

  /**
   * Sauvegarde plusieurs screenshots en batch (1 seule transaction)
   * Optimisation: 30 screenshots en 0.8s au lieu de 2.4s
   * @param {string} sessionId - ID de la session
   * @param {Object} screenshotsData - Object avec format { screenshotId: { blob, timestamp } }
   * @returns {Promise<Array>} Array des IDs créés
   */
  async saveScreenshotsBatch(sessionId, screenshotsData) {
    return new Promise((resolve, reject) => {
      const startTime = performance.now();

      // Create single transaction for all screenshots
      const transaction = this.db.transaction(['screenshots'], 'readwrite');
      const store = transaction.objectStore('screenshots');

      const screenshotIds = [];
      const entries = Object.entries(screenshotsData);

      // Queue all put operations in the same transaction
      entries.forEach(([screenshotId, data]) => {
        const record = {
          id: `${sessionId}_${screenshotId}`,
          sessionId: sessionId,
          screenshotId: screenshotId,
          blob: data.blob,
          timestamp: data.timestamp
        };

        store.put(record);
        screenshotIds.push(record.id);
      });

      // Transaction complete handler
      transaction.oncomplete = () => {
        const duration = performance.now() - startTime;
        console.log(`[DB] Batch saved ${entries.length} screenshots in ${duration.toFixed(2)}ms (${(duration / entries.length).toFixed(2)}ms per screenshot)`);
        resolve(screenshotIds);
      };

      transaction.onerror = () => {
        console.error('[DB] Error in batch save transaction:', transaction.error);
        reject(transaction.error);
      };
    });
  }

  /**
   * Récupère un screenshot
   */
  async getScreenshot(sessionId, screenshotId) {
    return new Promise((resolve, reject) => {
      const transaction = this.db.transaction(['screenshots'], 'readonly');
      const store = transaction.objectStore('screenshots');
      
      const id = `${sessionId}_${screenshotId}`;
      const request = store.get(id);

      request.onsuccess = () => {
        if (request.result) {
          resolve(request.result.blob);
        } else {
          resolve(null);
        }
      };

      request.onerror = () => {
        console.error('[DB] Error getting screenshot:', request.error);
        reject(request.error);
      };
    });
  }

  /**
   * Recherche sessions par URL
   */
  async searchByUrl(urlPattern) {
    const sessions = await this.getAllSessions();
    const pattern = urlPattern.toLowerCase();
    
    return sessions.filter(session => 
      session.metadata?.initialUrl?.toLowerCase().includes(pattern)
    );
  }

  /**
   * Trie les sessions
   */
  sortSessions(sessions, sortBy) {
    const sorted = [...sessions];
    
    switch (sortBy) {
      case 'date-desc':
        return sorted.sort((a, b) => b.importDate - a.importDate);
      
      case 'date-asc':
        return sorted.sort((a, b) => a.importDate - b.importDate);
      
      case 'duration-desc':
        return sorted.sort((a, b) => (b.metadata?.duration || 0) - (a.metadata?.duration || 0));
      
      case 'duration-asc':
        return sorted.sort((a, b) => (a.metadata?.duration || 0) - (b.metadata?.duration || 0));
      
      default:
        return sorted;
    }
  }

  /**
   * Compte le nombre total de sessions
   */
  async getSessionCount() {
    return new Promise((resolve, reject) => {
      const transaction = this.db.transaction(['sessions'], 'readonly');
      const store = transaction.objectStore('sessions');
      
      const request = store.count();

      request.onsuccess = () => {
        resolve(request.result);
      };

      request.onerror = () => {
        reject(request.error);
      };
    });
  }

  /**
   * Nettoie les anciennes sessions (optionnel)
   */
  async cleanupOldSessions(daysToKeep = 30) {
    const cutoffDate = Date.now() - (daysToKeep * 24 * 60 * 60 * 1000);
    const sessions = await this.getAllSessions();

    const toDelete = sessions.filter(session => session.importDate < cutoffDate);

    for (const session of toDelete) {
      await this.deleteSession(session.id);
    }

    console.log(`[DB] Cleaned up ${toDelete.length} old sessions`);
    return toDelete.length;
  }

  // ========================================
  // FOLDERS - Gestion des dossiers (v2)
  // ========================================

  /**
   * Crée un nouveau dossier
   */
  async createFolder(name, color = null) {
    return new Promise((resolve, reject) => {
      const transaction = this.db.transaction(['folders'], 'readwrite');
      const store = transaction.objectStore('folders');

      const folder = {
        id: `folder_${Date.now()}_${Math.random().toString(36).substr(2, 9)}`,
        name: name,
        color: color,
        createdAt: Date.now(),
        sessions: [] // Liste des IDs de sessions
      };

      const request = store.add(folder);

      request.onsuccess = () => {
        console.log('[DB] Folder created:', folder.id);
        resolve(folder);
      };

      request.onerror = () => {
        console.error('[DB] Error creating folder:', request.error);
        reject(request.error);
      };
    });
  }

  /**
   * Récupère tous les dossiers
   */
  async getAllFolders() {
    return new Promise((resolve, reject) => {
      const transaction = this.db.transaction(['folders'], 'readonly');
      const store = transaction.objectStore('folders');
      const request = store.getAll();

      request.onsuccess = () => {
        resolve(request.result || []);
      };

      request.onerror = () => {
        console.error('[DB] Error getting folders:', request.error);
        reject(request.error);
      };
    });
  }

  /**
   * Met à jour un dossier
   */
  async updateFolder(folderId, updates) {
    return new Promise((resolve, reject) => {
      const transaction = this.db.transaction(['folders'], 'readwrite');
      const store = transaction.objectStore('folders');

      const getRequest = store.get(folderId);

      getRequest.onsuccess = () => {
        const folder = getRequest.result;
        if (!folder) {
          reject(new Error('Folder not found'));
          return;
        }

        Object.assign(folder, updates);

        const putRequest = store.put(folder);

        putRequest.onsuccess = () => {
          console.log('[DB] Folder updated:', folderId);
          resolve(folder);
        };

        putRequest.onerror = () => {
          reject(putRequest.error);
        };
      };

      getRequest.onerror = () => {
        reject(getRequest.error);
      };
    });
  }

  /**
   * Supprime un dossier
   */
  async deleteFolder(folderId) {
    return new Promise((resolve, reject) => {
      const transaction = this.db.transaction(['folders'], 'readwrite');
      const store = transaction.objectStore('folders');
      const request = store.delete(folderId);

      request.onsuccess = () => {
        console.log('[DB] Folder deleted:', folderId);
        resolve();
      };

      request.onerror = () => {
        console.error('[DB] Error deleting folder:', request.error);
        reject(request.error);
      };
    });
  }

  // ========================================
  // SESSION METADATA - Titres personnalisés, tags, etc. (v2)
  // ========================================

  /**
   * Sauvegarde ou met à jour les métadonnées d'une session
   */
  async saveSessionMetadata(sessionId, metadata) {
    return new Promise((resolve, reject) => {
      const transaction = this.db.transaction(['sessionsMetadata'], 'readwrite');
      const store = transaction.objectStore('sessionsMetadata');

      const data = {
        sessionId: sessionId,
        customTitle: metadata.customTitle || null,
        folderId: metadata.folderId || null,
        tags: metadata.tags || [],
        notes: metadata.notes || null,
        updatedAt: Date.now()
      };

      const request = store.put(data);

      request.onsuccess = () => {
        console.log('[DB] Session metadata saved:', sessionId);
        resolve(data);
      };

      request.onerror = () => {
        console.error('[DB] Error saving session metadata:', request.error);
        reject(request.error);
      };
    });
  }

  /**
   * Récupère les métadonnées d'une session
   */
  async getSessionMetadata(sessionId) {
    return new Promise((resolve, reject) => {
      const transaction = this.db.transaction(['sessionsMetadata'], 'readonly');
      const store = transaction.objectStore('sessionsMetadata');
      const request = store.get(sessionId);

      request.onsuccess = () => {
        resolve(request.result || null);
      };

      request.onerror = () => {
        console.error('[DB] Error getting session metadata:', request.error);
        reject(request.error);
      };
    });
  }

  /**
   * Récupère toutes les métadonnées des sessions
   */
  async getAllSessionsMetadata() {
    return new Promise((resolve, reject) => {
      const transaction = this.db.transaction(['sessionsMetadata'], 'readonly');
      const store = transaction.objectStore('sessionsMetadata');
      const request = store.getAll();

      request.onsuccess = () => {
        // Convertir en Map pour accès rapide
        const metadataMap = new Map();
        (request.result || []).forEach(meta => {
          metadataMap.set(meta.sessionId, meta);
        });
        resolve(metadataMap);
      };

      request.onerror = () => {
        console.error('[DB] Error getting all metadata:', request.error);
        reject(request.error);
      };
    });
  }

  /**
   * Supprime les métadonnées d'une session
   */
  async deleteSessionMetadata(sessionId) {
    return new Promise((resolve, reject) => {
      const transaction = this.db.transaction(['sessionsMetadata'], 'readwrite');
      const store = transaction.objectStore('sessionsMetadata');
      const request = store.delete(sessionId);

      request.onsuccess = () => {
        console.log('[DB] Session metadata deleted:', sessionId);
        resolve();
      };

      request.onerror = () => {
        console.error('[DB] Error deleting session metadata:', request.error);
        reject(request.error);
      };
    });
  }
}

// Export singleton
const db = new DatabaseService();
export default db;
