/**
 * SUMO UX Tracker - Data Anonymizer
 * Anonymise les données de session avant envoi à l'API Gemini
 */

class DataAnonymizer {
  constructor() {
    // Domains/URLs detected during session
    this.detectedDomains = new Set();
    this.domainMappings = new Map();
    this.domainCounter = 1;
  }

  /**
   * Anonymise une session complète
   */
  anonymizeSession(session, funnelResults = null, options = {}) {
    console.log("[DataAnonymizer] Anonymizing session...");

    this.detectedDomains.clear();
    this.domainMappings.clear();
    this.domainCounter = 1;

    const includeScreenshots = options.includeScreenshots || false;

    // Extraire le viewport depuis metadata ou depuis le premier screenshot
    let viewport = session.data?.metadata?.viewport;
    if (!viewport && session.data?.screenshots?.length > 0) {
      viewport = session.data.screenshots[0].viewport;
    }
    if (!viewport) {
      viewport = { width: 0, height: 0 };
    }

    const anonymized = {
      id: this.anonymizeId(session.id),
      timestamp: session.timestamp,
      duration: session.duration || 0,
      stats: this.anonymizeStats(session.data?.stats || {}),
      events_summary: this.anonymizeEventsSummary(session.data?.events || []),
      patterns: this.anonymizePatterns(session.patterns || {}),
      screenshots_count: session.data?.screenshots?.length || 0,
      feedbacks: this.anonymizeFeedbacks(session.data?.feedbacks || [], includeScreenshots),
      funnels: this.anonymizeFunnels(funnelResults),
      metadata: {
        browser: session.data?.metadata?.browser || "unknown",
        viewport: viewport,
        device_type: this.getDeviceType(viewport),
      },
    };

    console.log("[DataAnonymizer] Anonymization complete");
    console.log(
      "[DataAnonymizer] Detected domains:",
      Array.from(this.detectedDomains),
    );

    return {
      anonymized_data: anonymized,
      domain_mappings: Object.fromEntries(this.domainMappings),
      domains_detected: Array.from(this.detectedDomains),
    };
  }

  /**
   * Anonymise un ID de session
   */
  anonymizeId(sessionId) {
    // Remplacer par hash SHA-256 des premiers caractères
    const shortId = sessionId.substring(0, 8);
    return `session_${this.hashString(shortId)}`;
  }

  /**
   * Anonymise les statistiques
   */
  anonymizeStats(stats) {
    return {
      total_events: stats.total_events || 0,
      clicks: stats.clicks || 0,
      mouse_moves: stats.mouse_moves || 0,
      scrolls: stats.scrolls || 0,
      page_changes: stats.page_changes || 0,
      errors: stats.errors || 0,
    };
  }

  /**
   * Anonymise le résumé des events (sans données sensibles)
   */
  anonymizeEventsSummary(events) {
    const summary = {
      total: events.length,
      by_type: {},
      timeline: [],
    };

    // Grouper par type
    events.forEach((event) => {
      const type = event.type || "unknown";
      summary.by_type[type] = (summary.by_type[type] || 0) + 1;
    });

    // Timeline anonymisée (seulement types et timestamps relatifs)
    const firstTimestamp = events[0]?.timestamp || Date.now();

    events.forEach((event, index) => {
      const relativeTime = event.timestamp - firstTimestamp;

      const timelineEvent = {
        index: index,
        type: event.type,
        relative_time_ms: relativeTime,
        relative_time_formatted: this.formatDuration(relativeTime),
      };

      // Ajouter infos anonymisées selon le type
      if (event.type === "click") {
        timelineEvent.position = { x: event.x, y: event.y };
        // Extraire le type d'élément depuis target.tagName ou event.element
        timelineEvent.element_type = event.target?.tagName || event.element || "unknown";
      } else if (event.type === "page_change") {
        timelineEvent.url = this.anonymizeUrl(event.url);
      } else if (event.type === "scroll") {
        timelineEvent.scroll_depth = event.scrollY;
      } else if (event.type === "error") {
        timelineEvent.error_type = this.anonymizeErrorMessage(event.message);
      }

      summary.timeline.push(timelineEvent);
    });

    return summary;
  }

  /**
   * Anonymise les patterns détectés
   */
  anonymizePatterns(patterns) {
    const anonymized = {
      total: 0,
      rage_clicks: [],
      dead_clicks: [],
      confusion_zones: [],
      form_abandonments: [],
      repetitive_scrolls: [],
    };

    // Rage clicks
    if (patterns.rage_clicks?.length > 0) {
      patterns.rage_clicks.forEach((rc) => {
        anonymized.rage_clicks.push({
          intensity: rc.intensity,
          click_count: rc.clickCount,
          duration_ms: rc.duration,
          position: rc.position,
        });
      });
      anonymized.total += patterns.rage_clicks.length;
    }

    // Dead clicks
    if (patterns.dead_clicks?.length > 0) {
      patterns.dead_clicks.forEach((dc) => {
        anonymized.dead_clicks.push({
          element_type: dc.element,
          position: dc.position,
          selector_anonymized: this.anonymizeSelector(dc.selector),
        });
      });
      anonymized.total += patterns.dead_clicks.length;
    }

    // Confusion zones
    if (patterns.confusion_zones?.length > 0) {
      patterns.confusion_zones.forEach((cz) => {
        anonymized.confusion_zones.push({
          duration_ms: cz.duration,
          position: cz.position,
        });
      });
      anonymized.total += patterns.confusion_zones.length;
    }

    // Form abandonments
    if (patterns.form_abandonments?.length > 0) {
      patterns.form_abandonments.forEach((fa) => {
        anonymized.form_abandonments.push({
          element_type: fa.element,
          field_type: fa.fieldType,
          characters_lost: fa.charactersLost,
        });
      });
      anonymized.total += patterns.form_abandonments.length;
    }

    // Repetitive scrolls
    if (patterns.repetitive_scrolls?.length > 0) {
      patterns.repetitive_scrolls.forEach((rs) => {
        anonymized.repetitive_scrolls.push({
          direction_changes: rs.changes,
        });
      });
      anonymized.total += patterns.repetitive_scrolls.length;
    }

    return anonymized;
  }

  /**
   * Anonymise les feedbacks utilisateur
   */
  anonymizeFeedbacks(feedbacks, includeScreenshots = false) {
    if (!feedbacks || feedbacks.length === 0) {
      return {
        total: 0,
        frustration_count: 0,
        positive_count: 0,
        comments: [],
        screenshots: []
      };
    }

    const anonymized = {
      total: feedbacks.length,
      frustration_count: 0,
      positive_count: 0,
      comments: [],
      screenshots: []
    };

    feedbacks.forEach((feedback, index) => {
      // Extraire les post-its avec du texte
      if (feedback.annotations) {
        feedback.annotations.forEach((annotation) => {
          if (annotation.type === 'postit' && annotation.text) {
            const isFrustration = annotation.isFrustration || false;

            if (isFrustration) {
              anonymized.frustration_count++;
            } else {
              anonymized.positive_count++;
            }

            anonymized.comments.push({
              index: index,
              type: isFrustration ? 'frustration' : 'positive',
              comment: annotation.text,
              url: this.anonymizeUrl(feedback.url),
              timestamp_relative: feedback.timestamp
            });
          }
        });
      }

      // Inclure les screenshots annotés si demandé (pour analyse visuelle)
      if (includeScreenshots && feedback.screenshot) {
        anonymized.screenshots.push({
          index: index,
          screenshot_base64: feedback.screenshot,
          url: this.anonymizeUrl(feedback.url),
          annotation_types: this.extractAnnotationTypes(feedback.annotations),
          has_frustration: feedback.annotations?.some(a => a.type === 'postit' && a.isFrustration) || false
        });
      }
    });

    return anonymized;
  }

  /**
   * Extrait les types d'annotations pour contexte
   */
  extractAnnotationTypes(annotations) {
    if (!annotations) return [];

    const types = {
      highlight: 0,
      arrow: 0,
      freehand: 0,
      postit: 0,
      postit_frustration: 0
    };

    annotations.forEach(annotation => {
      if (annotation.type === 'postit') {
        types.postit++;
        if (annotation.isFrustration) {
          types.postit_frustration++;
        }
      } else if (types[annotation.type] !== undefined) {
        types[annotation.type]++;
      }
    });

    return types;
  }

  /**
   * Anonymise les résultats de funnels
   */
  anonymizeFunnels(funnelResults) {
    if (!funnelResults || funnelResults.length === 0) {
      return {
        total: 0,
        funnels_analyzed: []
      };
    }

    const anonymized = {
      total: funnelResults.length,
      funnels_analyzed: []
    };

    funnelResults.forEach((result) => {
      anonymized.funnels_analyzed.push({
        funnel_name: result.funnel_name,
        category: result.category || 'unknown',
        started: result.started,
        completed: result.completed,
        abandoned: result.abandoned,
        completion_rate: result.completion_rate,
        time_to_complete_ms: result.time_to_complete,
        steps_completed_count: result.steps_completed.length,
        abandonment_point: result.abandonment_point?.step_name || null
      });
    });

    return anonymized;
  }

  /**
   * Anonymise une URL
   */
  anonymizeUrl(url) {
    if (!url) return "unknown_url";

    try {
      const urlObj = new URL(url);
      const domain = urlObj.hostname;

      // Stocker domain détecté
      this.detectedDomains.add(domain);

      // Mapper domain à un alias
      if (!this.domainMappings.has(domain)) {
        this.domainMappings.set(domain, `domain_${this.domainCounter}`);
        this.domainCounter++;
      }

      const anonymizedDomain = this.domainMappings.get(domain);

      // Anonymiser le path (garder structure mais pas contenu)
      const pathSegments = urlObj.pathname
        .split("/")
        .filter((s) => s.length > 0);
      const anonymizedPath = pathSegments
        .map((_, index) => `segment_${index + 1}`)
        .join("/");

      return `${anonymizedDomain}/${anonymizedPath}`;
    } catch (e) {
      return "invalid_url";
    }
  }

  /**
   * Anonymise un sélecteur CSS
   */
  anonymizeSelector(selector) {
    if (!selector || selector === "unknown") return "unknown_selector";

    // Remplacer IDs, classes, attributs par placeholders
    let anonymized = selector;

    // IDs: #user-profile-123 -> #id_1
    anonymized = anonymized.replace(/#[\w-]+/g, "#id_x");

    // Classes: .user-card -> .class_x
    anonymized = anonymized.replace(/\.[\w-]+/g, ".class_x");

    // Attributs: [data-user="john"] -> [attr_x]
    anonymized = anonymized.replace(
      /\[[\w-]+(?:=["'][^"']*["'])?\]/g,
      "[attr_x]",
    );

    // Garder seulement structure des tags
    return anonymized;
  }

  /**
   * Anonymise un message d'erreur
   */
  anonymizeErrorMessage(message) {
    if (!message) return "unknown_error";

    // Remplacer URLs dans le message
    let anonymized = message.replace(/https?:\/\/[^\s]+/g, "[URL]");

    // Remplacer chemins de fichiers
    anonymized = anonymized.replace(/[a-zA-Z]:\\[\w\\.-]+/g, "[PATH]");
    anonymized = anonymized.replace(/\/[\w\/.-]+/g, "[PATH]");

    // Remplacer numéros (IDs potentiels)
    anonymized = anonymized.replace(/\b\d{3,}\b/g, "[NUMBER]");

    return anonymized;
  }

  /**
   * Détermine le type de device
   */
  getDeviceType(viewport) {
    if (!viewport || !viewport.width) return "unknown";

    const width = viewport.width;

    if (width < 768) return "mobile";
    if (width < 1024) return "tablet";
    return "desktop";
  }

  /**
   * Génère un hash simple d'une chaîne
   */
  hashString(str) {
    let hash = 0;
    for (let i = 0; i < str.length; i++) {
      const char = str.charCodeAt(i);
      hash = (hash << 5) - hash + char;
      hash = hash & hash; // Convert to 32bit integer
    }
    return Math.abs(hash).toString(36).substring(0, 8);
  }

  /**
   * Formate une durée en ms
   */
  formatDuration(ms) {
    const seconds = Math.floor(ms / 1000);
    const minutes = Math.floor(seconds / 60);
    const hours = Math.floor(minutes / 60);

    if (hours > 0) {
      return `${hours}h ${minutes % 60}m ${seconds % 60}s`;
    } else if (minutes > 0) {
      return `${minutes}m ${seconds % 60}s`;
    } else {
      return `${seconds}s`;
    }
  }

  /**
   * Génère un résumé texte des données anonymisées
   */
  generateAnonymizationReport(result) {
    const { anonymized_data, domain_mappings, domains_detected } = result;

    const report = {
      summary: `Session anonymisée avec succès`,
      details: {
        events_anonymized: anonymized_data.events_summary.total,
        patterns_anonymized: anonymized_data.patterns.total,
        domains_detected: domains_detected.length,
        screenshots_removed: anonymized_data.screenshots_count,
      },
      mappings: domain_mappings,
      data_removed: [
        "URLs complètes (remplacées par domaines anonymes)",
        "Sélecteurs CSS (IDs, classes, attributs masqués)",
        "Messages d'erreur (URLs et chemins supprimés)",
        "Screenshots (nombre conservé, images supprimées)",
        "Textes saisis (complètement supprimés)",
        "ID de session (hashé)",
      ],
    };

    return report;
  }

  /**
   * Valide qu'une session anonymisée ne contient pas de données sensibles
   */
  validateAnonymization(anonymizedData) {
    const sensitivePatterns = [
      /https?:\/\//, // URLs
      /@[\w.-]+/, // Emails
      /\b\d{3}-\d{3}-\d{4}\b/, // Phone numbers
      /[a-zA-Z]:\\/, // Windows paths
    ];

    const dataString = JSON.stringify(anonymizedData);

    const warnings = [];
    sensitivePatterns.forEach((pattern, index) => {
      if (pattern.test(dataString)) {
        warnings.push(`Potential sensitive data detected (pattern ${index})`);
      }
    });

    return {
      is_safe: warnings.length === 0,
      warnings: warnings,
    };
  }
}

// Export singleton
const dataAnonymizer = new DataAnonymizer();
export default dataAnonymizer;
