/**
 * SUMO UX Tracker - Comparison Manager
 * FEATURE #2: Comparaison A/B de sessions
 * Compare deux sessions côte à côte avec métriques, insights et diff visuel
 */

import db from './db.js';
import { formatDuration, truncateUrl, percentage } from './utils.js';
import pdfExport from './pdfExport.js';

class ComparisonManager {
  constructor() {
    this.sessionA = null;
    this.sessionB = null;
    this.metricsA = null;
    this.metricsB = null;
  }

  /**
   * Initialise la comparaison avec 2 sessions
   */
  async init(sessionIdA, sessionIdB) {
    console.log('[Comparison] Initializing comparison:', sessionIdA, 'vs', sessionIdB);

    try {
      // Charger les sessions
      this.sessionA = await db.getSessionWithScreenshotsLazy(sessionIdA);
      this.sessionB = await db.getSessionWithScreenshotsLazy(sessionIdB);

      // Calculer les métriques pour les deux sessions
      this.metricsA = this.calculateMetrics(this.sessionA);
      this.metricsB = this.calculateMetrics(this.sessionB);

      // Afficher l'interface
      this.render();

      console.log('[Comparison] Comparison initialized');
    } catch (error) {
      console.error('[Comparison] Error initializing:', error);
      this.showError('Erreur lors du chargement des sessions');
    }
  }

  /**
   * Calcule les métriques pour une session
   */
  calculateMetrics(session) {
    const events = session.data.events || [];
    const meta = session.metadata;

    return {
      // Métriques de base
      duration: meta.duration,
      totalEvents: events.length,
      clicks: meta.clicks || 0,
      rageClicks: meta.rageClicks || 0,
      scrolls: meta.scrolls || 0,
      mouseMoves: meta.mouseMoves || 0,

      // Pages
      pagesViewed: session.data.pages?.length || 1,
      screenshotsCount: session.screenshots?.length || 0,

      // Engagement
      engagementRate: this.calculateEngagementRate(session),
      activeTime: this.calculateActiveTime(events),
      clickRate: this.calculateClickRate(events, meta.duration),
      scrollDepth: this.calculateScrollDepth(events),

      // UX Score
      uxScore: this.calculateUXScore(session, events, meta)
    };
  }

  /**
   * Calcule le taux d'engagement
   */
  calculateEngagementRate(session) {
    const durationMin = session.metadata.duration / 60000;
    const actions = (session.data.events || []).filter(e =>
      ['click', 'scroll', 'form_input'].includes(e.type)
    ).length;

    return durationMin > 0 ? (actions / durationMin).toFixed(1) : 0;
  }

  /**
   * Calcule le temps actif
   */
  calculateActiveTime(events) {
    let activeTime = 0;
    let lastEventTime = 0;
    const inactivityThreshold = 10000;

    events.forEach(event => {
      if (lastEventTime > 0 && event.timestamp - lastEventTime < inactivityThreshold) {
        activeTime += event.timestamp - lastEventTime;
      }
      lastEventTime = event.timestamp;
    });

    return activeTime;
  }

  /**
   * Calcule le taux de clics
   */
  calculateClickRate(events, duration) {
    const clicks = events.filter(e => e.type === 'click').length;
    const durationSec = duration / 1000;
    return durationSec > 0 ? (clicks / durationSec * 60).toFixed(1) : 0;
  }

  /**
   * Calcule la profondeur de scroll
   */
  calculateScrollDepth(events) {
    const scrollEvents = events.filter(e => e.type === 'scroll');
    if (scrollEvents.length === 0) return 0;

    const maxScrollPercentage = Math.max(...scrollEvents.map(e => e.scrollPercentage || 0));
    if (maxScrollPercentage > 0) {
      return Math.min(100, Math.round(maxScrollPercentage));
    }

    const maxScroll = Math.max(...scrollEvents.map(e => e.scrollY || 0));
    const scrollHeight = scrollEvents[0]?.scrollHeight || 1;
    const viewportHeight = scrollEvents[0]?.viewportHeight || 0;
    const scrollableHeight = scrollHeight - viewportHeight;

    if (scrollableHeight <= 0) return 0;
    return Math.min(100, Math.round((maxScroll / scrollableHeight) * 100));
  }

  /**
   * Calcule le UX Score
   */
  calculateUXScore(session, events, meta) {
    const engagementRate = parseFloat(this.calculateEngagementRate(session));
    const engagement = Math.min(100, (engagementRate / 10) * 100);

    const hesitations = events.filter(e => e.type === 'hesitation').length;
    const deadClicks = events.filter(e => e.type === 'dead_click').length;
    const efficiency = Math.max(0, 100 - (hesitations * 5) - (deadClicks * 10));

    const rageClicks = meta.rageClicks || 0;
    const frustration = Math.max(0, 100 - (rageClicks * 20));

    const overall = (engagement * 0.4) + (efficiency * 0.3) + (frustration * 0.3);

    return {
      overall: Math.round(overall),
      engagement: Math.round(engagement),
      efficiency: Math.round(efficiency),
      frustration: Math.round(frustration)
    };
  }

  /**
   * Calcule la différence entre deux valeurs
   */
  calculateDiff(valueA, valueB) {
    const diff = valueB - valueA;
    const percentDiff = valueA > 0 ? ((diff / valueA) * 100).toFixed(1) : 0;

    return {
      absolute: diff,
      percent: parseFloat(percentDiff),
      isPositive: diff > 0,
      isNegative: diff < 0,
      isNeutral: diff === 0
    };
  }

  /**
   * Détermine si une diff est bonne ou mauvaise pour la métrique
   */
  getDiffClass(metricName, diff) {
    // Pour ces métriques, une augmentation est négative
    const inverseMetrics = ['rageClicks', 'duration'];

    if (diff.isNeutral) return 'neutral';

    if (inverseMetrics.includes(metricName)) {
      return diff.isPositive ? 'worse' : 'better';
    }

    return diff.isPositive ? 'better' : 'worse';
  }

  /**
   * Génère l'icône de diff
   */
  getDiffIcon(diffClass) {
    if (diffClass === 'better') return '📈';
    if (diffClass === 'worse') return '📉';
    return '➖';
  }

  /**
   * Formate la diff pour affichage
   */
  formatDiff(diff) {
    if (diff.isNeutral) return '0%';

    const sign = diff.isPositive ? '+' : '';
    return `${sign}${diff.percent}%`;
  }

  /**
   * Affiche l'interface de comparaison
   */
  render() {
    const container = document.getElementById('comparison-container');
    if (!container) return;

    container.innerHTML = `
      <!-- Header avec sélecteur de sessions -->
      <div class="comparison-header">
        <button id="comparison-btn-back" class="btn-secondary">← Retour</button>
        <div class="comparison-header-title">
          <h2>⚖️ Comparaison A/B Sessions</h2>
          <button id="comparison-export-pdf" class="btn-export-pdf" title="Export Comparaison PDF">
            📄 Export PDF
          </button>
        </div>

        <div class="comparison-session-selectors">
          <div class="comparison-session-card session-a">
            <div class="comparison-session-label">Session A</div>
            <div class="comparison-session-info">
              <div class="comparison-session-url">${truncateUrl(this.sessionA.metadata?.initialUrl || this.sessionA.url || 'N/A', 50)}</div>
              <div class="comparison-session-date">${new Date(this.sessionA.metadata?.importDate || Date.now()).toLocaleDateString('fr-FR')}</div>
              <div class="comparison-session-id">ID: ${this.sessionA.id.substring(0, 12)}...</div>
            </div>
          </div>

          <div class="comparison-vs">VS</div>

          <div class="comparison-session-card session-b">
            <div class="comparison-session-label">Session B</div>
            <div class="comparison-session-info">
              <div class="comparison-session-url">${truncateUrl(this.sessionB.metadata?.initialUrl || this.sessionB.url || 'N/A', 50)}</div>
              <div class="comparison-session-date">${new Date(this.sessionB.metadata?.importDate || Date.now()).toLocaleDateString('fr-FR')}</div>
              <div class="comparison-session-id">ID: ${this.sessionB.id.substring(0, 12)}...</div>
            </div>
          </div>
        </div>
      </div>

      <!-- UX Score Comparison -->
      <div class="comparison-section">
        <h3 class="comparison-section-title">🎯 UX Score Global</h3>
        ${this.renderUXScoreComparison()}
      </div>

      <!-- Metrics Grid Comparison -->
      <div class="comparison-section">
        <h3 class="comparison-section-title">📊 Métriques Détaillées</h3>
        ${this.renderMetricsComparison()}
      </div>

      <!-- Insights Comparison -->
      <div class="comparison-section">
        <h3 class="comparison-section-title">💡 Insights & Recommandations</h3>
        ${this.renderInsightsComparison()}
      </div>
    `;

    this.attachListeners();
  }

  /**
   * Affiche la comparaison des UX Scores
   */
  renderUXScoreComparison() {
    const scoreA = this.metricsA.uxScore.overall;
    const scoreB = this.metricsB.uxScore.overall;
    const diff = this.calculateDiff(scoreA, scoreB);
    const diffClass = this.getDiffClass('uxScore', diff);

    return `
      <div class="comparison-ux-scores">
        <div class="comparison-ux-score-item session-a">
          <div class="comparison-ux-score-circle">
            <svg viewBox="0 0 120 120" class="comparison-score-svg">
              <circle cx="60" cy="60" r="50" class="score-bg"></circle>
              <circle cx="60" cy="60" r="50"
                      class="score-progress ${this.getScoreClass(scoreA)}"
                      stroke-dasharray="${(scoreA / 100) * 314} 314"></circle>
            </svg>
            <div class="comparison-score-text">
              <div class="comparison-score-value">${scoreA}</div>
              <div class="comparison-score-label">Session A</div>
            </div>
          </div>
        </div>

        <div class="comparison-diff-indicator ${diffClass}">
          <div class="comparison-diff-icon">${this.getDiffIcon(diffClass)}</div>
          <div class="comparison-diff-value">${this.formatDiff(diff)}</div>
          <div class="comparison-diff-label">${diff.absolute > 0 ? '+' : ''}${diff.absolute} points</div>
        </div>

        <div class="comparison-ux-score-item session-b">
          <div class="comparison-ux-score-circle">
            <svg viewBox="0 0 120 120" class="comparison-score-svg">
              <circle cx="60" cy="60" r="50" class="score-bg"></circle>
              <circle cx="60" cy="60" r="50"
                      class="score-progress ${this.getScoreClass(scoreB)}"
                      stroke-dasharray="${(scoreB / 100) * 314} 314"></circle>
            </svg>
            <div class="comparison-score-text">
              <div class="comparison-score-value">${scoreB}</div>
              <div class="comparison-score-label">Session B</div>
            </div>
          </div>
        </div>
      </div>

      <!-- UX Score Breakdown -->
      <div class="comparison-ux-breakdown">
        ${this.renderScoreBreakdown('engagement', 'Engagement', this.metricsA.uxScore.engagement, this.metricsB.uxScore.engagement)}
        ${this.renderScoreBreakdown('efficiency', 'Efficacité', this.metricsA.uxScore.efficiency, this.metricsB.uxScore.efficiency)}
        ${this.renderScoreBreakdown('frustration', 'Sans frustration', this.metricsA.uxScore.frustration, this.metricsB.uxScore.frustration)}
      </div>
    `;
  }

  /**
   * Affiche le breakdown d'un composant du score
   */
  renderScoreBreakdown(key, label, valueA, valueB) {
    const diff = this.calculateDiff(valueA, valueB);
    const diffClass = this.getDiffClass(key, diff);

    return `
      <div class="comparison-breakdown-item">
        <div class="comparison-breakdown-label">${label}</div>
        <div class="comparison-breakdown-bars">
          <div class="comparison-breakdown-bar session-a">
            <div class="comparison-breakdown-fill" style="width: ${valueA}%"></div>
            <span class="comparison-breakdown-value">${valueA}</span>
          </div>
          <div class="comparison-breakdown-diff ${diffClass}">
            ${this.getDiffIcon(diffClass)} ${this.formatDiff(diff)}
          </div>
          <div class="comparison-breakdown-bar session-b">
            <div class="comparison-breakdown-fill" style="width: ${valueB}%"></div>
            <span class="comparison-breakdown-value">${valueB}</span>
          </div>
        </div>
      </div>
    `;
  }

  /**
   * Affiche la comparaison des métriques
   */
  renderMetricsComparison() {
    const metrics = [
      { key: 'duration', label: 'Durée', format: (v) => formatDuration(v), inverse: true },
      { key: 'clicks', label: 'Clics', format: (v) => v },
      { key: 'rageClicks', label: 'Rage Clicks', format: (v) => v, inverse: true },
      { key: 'scrolls', label: 'Défilements', format: (v) => v },
      { key: 'pagesViewed', label: 'Pages vues', format: (v) => v },
      { key: 'engagementRate', label: 'Engagement (actions/min)', format: (v) => v },
      { key: 'scrollDepth', label: 'Scroll depth', format: (v) => `${v}%` }
    ];

    return `
      <div class="comparison-metrics-grid">
        ${metrics.map(metric => this.renderMetricRow(metric)).join('')}
      </div>
    `;
  }

  /**
   * Affiche une ligne de métrique
   */
  renderMetricRow(metric) {
    const valueA = this.metricsA[metric.key];
    const valueB = this.metricsB[metric.key];
    const diff = this.calculateDiff(
      typeof valueA === 'string' ? parseFloat(valueA) : valueA,
      typeof valueB === 'string' ? parseFloat(valueB) : valueB
    );
    const diffClass = this.getDiffClass(metric.key, diff);

    return `
      <div class="comparison-metric-row">
        <div class="comparison-metric-label">${metric.label}</div>
        <div class="comparison-metric-value session-a">${metric.format(valueA)}</div>
        <div class="comparison-metric-diff ${diffClass}">
          ${this.getDiffIcon(diffClass)} ${this.formatDiff(diff)}
        </div>
        <div class="comparison-metric-value session-b">${metric.format(valueB)}</div>
      </div>
    `;
  }

  /**
   * Affiche la comparaison des insights
   */
  renderInsightsComparison() {
    const insights = this.generateComparisonInsights();

    return `
      <div class="comparison-insights">
        ${insights.map(insight => `
          <div class="comparison-insight ${insight.type}">
            <div class="comparison-insight-icon">${insight.icon}</div>
            <div class="comparison-insight-content">
              <div class="comparison-insight-title">${insight.title}</div>
              <div class="comparison-insight-description">${insight.description}</div>
            </div>
          </div>
        `).join('')}
      </div>
    `;
  }

  /**
   * Génère les insights de comparaison
   */
  generateComparisonInsights() {
    const insights = [];

    // UX Score diff
    const scoreDiff = this.calculateDiff(this.metricsA.uxScore.overall, this.metricsB.uxScore.overall);
    if (Math.abs(scoreDiff.percent) > 10) {
      insights.push({
        type: scoreDiff.isPositive ? 'success' : 'warning',
        icon: scoreDiff.isPositive ? '✅' : '⚠️',
        title: `UX Score ${scoreDiff.isPositive ? 'en hausse' : 'en baisse'}`,
        description: `Session B a un score ${Math.abs(scoreDiff.percent)}% ${scoreDiff.isPositive ? 'supérieur' : 'inférieur'} (${this.metricsB.uxScore.overall} vs ${this.metricsA.uxScore.overall})`
      });
    }

    // Rage clicks
    if (this.metricsB.rageClicks > this.metricsA.rageClicks) {
      insights.push({
        type: 'danger',
        icon: '🔴',
        title: 'Augmentation des frustrations',
        description: `Session B a ${this.metricsB.rageClicks - this.metricsA.rageClicks} rage clicks de plus que Session A`
      });
    } else if (this.metricsB.rageClicks < this.metricsA.rageClicks) {
      insights.push({
        type: 'success',
        icon: '✅',
        title: 'Réduction des frustrations',
        description: `Session B a ${this.metricsA.rageClicks - this.metricsB.rageClicks} rage clicks de moins que Session A`
      });
    }

    // Engagement
    const engagementDiff = this.calculateDiff(
      parseFloat(this.metricsA.engagementRate),
      parseFloat(this.metricsB.engagementRate)
    );
    if (Math.abs(engagementDiff.percent) > 20) {
      insights.push({
        type: engagementDiff.isPositive ? 'success' : 'warning',
        icon: engagementDiff.isPositive ? '📈' : '📉',
        title: `Engagement ${engagementDiff.isPositive ? 'amélioré' : 'réduit'}`,
        description: `Session B a un taux d'engagement ${Math.abs(engagementDiff.percent)}% ${engagementDiff.isPositive ? 'supérieur' : 'inférieur'} (${this.metricsB.engagementRate} vs ${this.metricsA.engagementRate} actions/min)`
      });
    }

    // Scroll depth
    const scrollDiff = this.calculateDiff(this.metricsA.scrollDepth, this.metricsB.scrollDepth);
    if (Math.abs(scrollDiff.absolute) > 20) {
      insights.push({
        type: 'info',
        icon: '📜',
        title: `Profondeur de scroll ${scrollDiff.isPositive ? 'augmentée' : 'réduite'}`,
        description: `Session B a scrollé ${Math.abs(scrollDiff.absolute)}% ${scrollDiff.isPositive ? 'plus' : 'moins'} profond (${this.metricsB.scrollDepth}% vs ${this.metricsA.scrollDepth}%)`
      });
    }

    // Duration
    const durationDiff = this.calculateDiff(this.metricsA.duration, this.metricsB.duration);
    if (Math.abs(durationDiff.percent) > 30) {
      insights.push({
        type: 'info',
        icon: '⏱️',
        title: `Durée ${durationDiff.isPositive ? 'plus longue' : 'plus courte'}`,
        description: `Session B a duré ${formatDuration(Math.abs(durationDiff.absolute))} ${durationDiff.isPositive ? 'de plus' : 'de moins'} (${formatDuration(this.metricsB.duration)} vs ${formatDuration(this.metricsA.duration)})`
      });
    }

    return insights;
  }

  /**
   * Classe CSS selon le score
   */
  getScoreClass(score) {
    if (score >= 80) return 'excellent';
    if (score >= 60) return 'good';
    if (score >= 40) return 'average';
    return 'poor';
  }

  /**
   * Attache les listeners
   */
  attachListeners() {
    // Bouton retour
    document.getElementById('comparison-btn-back')?.addEventListener('click', () => {
      window.sumoApp.showView('sessions');
      window.sumoApp.sessionsManager?.loadSessions();
    });

    // Export PDF
    document.getElementById('comparison-export-pdf')?.addEventListener('click', () => {
      this.exportToPDF();
    });
  }

  /**
   * Export comparaison en PDF
   */
  async exportToPDF() {
    try {
      console.log('[Comparison] Exporting to PDF...');

      // TODO: Implémenter export PDF comparaison
      // Pour l'instant, notification placeholder
      pdfExport.showNotification('Export PDF comparaison bientôt disponible', 'info');

    } catch (error) {
      console.error('[Comparison] Error exporting PDF:', error);
      pdfExport.showNotification('Erreur lors de l\'export PDF', 'error');
    }
  }

  /**
   * Affiche une erreur
   */
  showError(message) {
    const container = document.getElementById('comparison-container');
    if (!container) return;

    container.innerHTML = `
      <div class="comparison-error">
        <div class="comparison-error-icon">⚠️</div>
        <div class="comparison-error-message">${message}</div>
      </div>
    `;
  }

  /**
   * Nettoie les ressources
   */
  cleanup() {
    this.sessionA = null;
    this.sessionB = null;
    this.metricsA = null;
    this.metricsB = null;
  }
}

// Export singleton
export default new ComparisonManager();
