/**
 * SUMO UX Tracker - Analytics Manager
 * Calcul des métriques et insights
 */

import { formatDuration, truncateUrl } from './utils.js';
import pdfExport from './pdfExport.js';
import patternsDetector from './patterns-detector.js';
import aiAnalyzer from './ai-analyzer.js';
import funnelManager from './funnel-manager.js';
import funnelUI from './funnel-ui.js';

class AnalyticsManager {
  constructor() {
    this.session = null;
    this.metrics = null;
    this.patterns = null;
    this.aiAnalysis = null;
    this.aiLoading = false;
    this.funnelResults = null;
  }

  /**
   * Initialise les analytics
   */
  async init(session) {
    console.log('[Analytics] Initializing analytics for session:', session.id);

    this.session = session;

    // Calculer les métriques
    this.calculateMetrics();

    // Analyser les patterns UX
    this.patterns = patternsDetector.analyzeSession(session);
    console.log('[Analytics] Patterns detected:', patternsDetector.getStats());

    // Analyser les funnels
    this.funnelResults = funnelManager.analyzeSessionFunnels(session);
    console.log('[Analytics] Funnel results:', this.funnelResults);

    // Charger les notes personnelles
    this.loadNotes();

    // Afficher l'interface
    this.render();

    // Attacher les event listeners
    this.attachEventListeners();
  }

  /**
   * Charge les notes depuis localStorage
   */
  loadNotes() {
    const storageKey = `sumo_notes_${this.session.id}`;
    const saved = localStorage.getItem(storageKey);
    this.notes = saved || '';
  }

  /**
   * Sauvegarde les notes dans localStorage
   */
  saveNotes(notes) {
    const storageKey = `sumo_notes_${this.session.id}`;
    localStorage.setItem(storageKey, notes);
    this.notes = notes;
    console.log('[Analytics] Notes saved for session:', this.session.id);
  }

  /**
   * Attache les event listeners
   */
  attachEventListeners() {
    // Écouter les changements dans le textarea des notes
    const notesTextarea = document.getElementById('analytics-notes-textarea');
    if (notesTextarea) {
      // Sauvegarder automatiquement après 500ms d'inactivité
      let saveTimeout;
      notesTextarea.addEventListener('input', (e) => {
        clearTimeout(saveTimeout);
        saveTimeout = setTimeout(() => {
          this.saveNotes(e.target.value);
        }, 500);
      });
    }

    // Bouton Export PDF
    const exportBtn = document.getElementById('analytics-export-pdf');
    if (exportBtn) {
      exportBtn.addEventListener('click', () => this.exportToPDF());
    }

    // Boutons AI
    const btnAIAnalyze = document.getElementById('btn-ai-analyze');
    const btnAISettings = document.getElementById('btn-ai-settings');

    if (btnAIAnalyze) {
      btnAIAnalyze.addEventListener('click', () => this.handleAIAnalysis());
    }

    if (btnAISettings) {
      btnAISettings.addEventListener('click', () => this.showAISettingsModal());
    }

    // AI Settings Modal
    this.attachAISettingsListeners();

    // AI Preview Modal
    this.attachAIPreviewListeners();
  }

  /**
   * Calcule toutes les métriques
   */
  calculateMetrics() {
    const events = this.session.data.events || [];
    const meta = this.session.metadata;

    // 1. Calculer les valeurs d'abord
    const engagementRate = this.calculateEngagementRate();
    const activeTime = this.calculateActiveTime(events);
    const clickRate = this.calculateClickRate(events);
    const scrollDepth = this.calculateScrollDepth(events);

    // 2. Créer l'objet metrics SANS uxScore
    this.metrics = {
      // Métriques de base
      duration: meta.duration,
      totalEvents: events.length,
      clicks: meta.clicks,
      rageClicks: meta.rageClicks,
      
      // Engagement
      engagementRate: engagementRate,
      activeTime: activeTime,
      
      // Navigation
      pagesVisited: this.session.data.pages?.length || 0,
      avgTimePerPage: this.calculateAvgTimePerPage(),
      
      // Interactions
      clickRate: clickRate,
      scrollDepth: scrollDepth,
      
      // UX Score sera ajouté après
      uxScore: null
    };

    // 3. Calculer uxScore MAINTENANT que this.metrics existe
    this.metrics.uxScore = this.calculateUXScore(events, meta, engagementRate);

    console.log('[Analytics] Metrics calculated:', this.metrics);
  }

  /**
   * Calcule le taux d'engagement (actions/minute)
   */
  calculateEngagementRate() {
    const durationMin = this.session.metadata.duration / 60000;
    const actions = this.session.data.events.filter(e => 
      ['click', 'scroll', 'form_input'].includes(e.type)
    ).length;
    
    return durationMin > 0 ? (actions / durationMin).toFixed(1) : 0;
  }

  /**
   * Calcule le temps actif
   */
  calculateActiveTime(events) {
    // Approximation : compter les périodes avec interactions
    let activeTime = 0;
    let lastEventTime = 0;
    const inactivityThreshold = 10000; // 10s

    events.forEach(event => {
      if (lastEventTime > 0 && event.timestamp - lastEventTime < inactivityThreshold) {
        activeTime += event.timestamp - lastEventTime;
      }
      lastEventTime = event.timestamp;
    });

    return activeTime;
  }

  /**
   * Temps moyen par page
   */
  calculateAvgTimePerPage() {
    const pages = this.session.data.pages?.length || 1;
    return this.session.metadata.duration / pages;
  }

  /**
   * Taux de clics
   */
  calculateClickRate(events) {
    const clicks = events.filter(e => e.type === 'click').length;
    const durationSec = this.session.metadata.duration / 1000;
    return durationSec > 0 ? (clicks / durationSec * 60).toFixed(1) : 0; // clics/min
  }

  /**
   * Profondeur de scroll
   */
  calculateScrollDepth(events) {
    const scrollEvents = events.filter(e => e.type === 'scroll');
    if (scrollEvents.length === 0) return 0;

    // Utiliser scrollPercentage si disponible, sinon calculer avec scrollY
    const maxScrollPercentage = Math.max(...scrollEvents.map(e => e.scrollPercentage || 0));
    if (maxScrollPercentage > 0) {
      return Math.min(100, Math.round(maxScrollPercentage));
    }

    // Fallback : calculer avec scrollY et scrollHeight
    const maxScroll = Math.max(...scrollEvents.map(e => e.scrollY || 0));
    const scrollHeight = scrollEvents[0]?.scrollHeight || 1;
    const viewportHeight = scrollEvents[0]?.viewportHeight || 0;
    
    const scrollableHeight = scrollHeight - viewportHeight;
    if (scrollableHeight <= 0) return 0;
    
    return Math.min(100, Math.round((maxScroll / scrollableHeight) * 100));
  }

  /**
   * Calcule le UX Score (0-100)
   */
  calculateUXScore(events, meta, engagementRate) {
    // Composantes du score
    const engagement = this.calculateEngagementScore(engagementRate);
    const efficiency = this.calculateEfficiencyScore(events);
    const frustration = this.calculateFrustrationScore(meta);

    // Score global (moyenne pondérée)
    const score = (engagement * 0.4) + (efficiency * 0.3) + (frustration * 0.3);
    
    return {
      overall: Math.round(score),
      engagement: Math.round(engagement),
      efficiency: Math.round(efficiency),
      frustration: Math.round(frustration)
    };
  }

  /**
   * Score d'engagement (0-100)
   */
  calculateEngagementScore(engagementRate) {
    const rate = parseFloat(engagementRate || 0);
    
    // 10+ actions/min = 100, 0 actions = 0
    return Math.min(100, (rate / 10) * 100);
  }

  /**
   * Score d'efficacité (0-100)
   */
  calculateEfficiencyScore(events) {
    // Basé sur hésitations et temps de décision
    const hesitations = events.filter(e => e.type === 'hesitation').length;
    const deadClicks = events.filter(e => e.type === 'dead_click').length;
    
    const penalties = (hesitations * 5) + (deadClicks * 10);
    return Math.max(0, 100 - penalties);
  }

  /**
   * Score de frustration (0-100, inversé)
   */
  calculateFrustrationScore(meta) {
    const rageClicks = meta.rageClicks || 0;
    
    // 0 rage clicks = 100, 5+ = 0
    const score = Math.max(0, 100 - (rageClicks * 20));
    return score;
  }

  /**
   * Exporte les analytics en PDF
   */
  async exportToPDF() {
    try {
      console.log('[Analytics] Exporting to PDF...');

      // Préparer les métriques avec insights
      const metrics = {
        ...this.metrics,
        insights: this.generateInsights()
      };

      // Appeler pdfExport
      const filename = await pdfExport.exportAnalytics(this.session, metrics);

      // Notification de succès
      pdfExport.showNotification(`PDF exporté: ${filename}`, 'success');

    } catch (error) {
      console.error('[Analytics] Error exporting PDF:', error);
      pdfExport.showNotification('Erreur lors de l\'export PDF', 'error');
    }
  }

  /**
   * Génère les insights automatiques
   */
  generateInsights() {
    const insights = [];
    const meta = this.session.metadata;
    const stats = patternsDetector.getStats();

    // Insights patterns UX critiques
    if (stats.rage_clicks > 0) {
      insights.push({
        type: 'danger',
        icon: '🔥',
        title: 'Rage clicks détectés',
        description: `${stats.rage_clicks} rage click${stats.rage_clicks > 1 ? 's' : ''} - l'utilisateur a cliqué rapidement au même endroit`,
        recommendation: 'Vérifier les éléments cliqués pour identifier les problèmes d\'interactivité ou de feedback visuel'
      });
    }

    if (stats.dead_clicks > 0) {
      insights.push({
        type: 'danger',
        icon: '💀',
        title: 'Dead clicks identifiés',
        description: `${stats.dead_clicks} clic${stats.dead_clicks > 1 ? 's' : ''} sur des éléments non-interactifs`,
        recommendation: 'Certains éléments semblent cliquables mais ne le sont pas - ajouter des curseurs ou retirer les affordances'
      });
    }

    if (stats.confusion_zones > 0) {
      insights.push({
        type: 'warning',
        icon: '😕',
        title: 'Zones de confusion détectées',
        description: `${stats.confusion_zones} zone${stats.confusion_zones > 1 ? 's' : ''} où l'utilisateur a hésité longuement`,
        recommendation: 'Améliorer la clarté visuelle, ajouter des labels ou revoir la hiérarchie d\'information'
      });
    }

    if (stats.form_abandonments > 0) {
      insights.push({
        type: 'warning',
        icon: '📝❌',
        title: 'Abandons de formulaire',
        description: `${stats.form_abandonments} champ${stats.form_abandonments > 1 ? 's' : ''} rempli${stats.form_abandonments > 1 ? 's' : ''} puis effacé${stats.form_abandonments > 1 ? 's' : ''}`,
        recommendation: 'Vérifier la validation, les labels et les messages d\'aide des formulaires'
      });
    }

    if (stats.repetitive_scrolls > 0) {
      insights.push({
        type: 'info',
        icon: '🔄',
        title: 'Scrolls répétitifs',
        description: `${stats.repetitive_scrolls} va-et-vient${stats.repetitive_scrolls > 1 ? 's' : ''} - l'utilisateur cherche quelque chose`,
        recommendation: 'Améliorer l\'architecture d\'information ou ajouter une recherche/navigation'
      });
    }

    // Insight engagement
    if (this.metrics.engagementRate > 15) {
      insights.push({
        type: 'success',
        icon: '✅',
        title: 'Engagement élevé',
        description: `Taux d'interaction de ${this.metrics.engagementRate} actions/min`,
        recommendation: 'L\'utilisateur est très engagé avec le contenu'
      });
    } else if (this.metrics.engagementRate < 5) {
      insights.push({
        type: 'warning',
        icon: '💤',
        title: 'Engagement faible',
        description: `Seulement ${this.metrics.engagementRate} actions/min`,
        recommendation: 'Le contenu pourrait manquer d\'interactivité ou d\'intérêt'
      });
    }

    // Insight scroll depth
    if (this.metrics.scrollDepth > 80) {
      insights.push({
        type: 'info',
        icon: '📜',
        title: 'Contenu consulté en profondeur',
        description: `L'utilisateur a scrollé jusqu'à ${this.metrics.scrollDepth}% de la page`,
        recommendation: 'Le contenu capte l\'attention'
      });
    }

    // Insight session courte
    if (meta.duration < 30000) {
      insights.push({
        type: 'warning',
        icon: '⏱️',
        title: 'Session très courte',
        description: `Durée de seulement ${formatDuration(meta.duration)}`,
        recommendation: 'L\'utilisateur n\'a peut-être pas trouvé ce qu\'il cherchait'
      });
    }

    return insights;
  }

  /**
   * Affiche l'interface
   */
  render() {
    const container = document.getElementById('analytics-container');
    if (!container) return;

    const insights = this.generateInsights();

    container.innerHTML = `
      <!-- Overview -->
      <div class="analytics-overview">
        <div class="analytics-metric">
          <div class="analytics-metric-icon">⏱️</div>
          <div class="analytics-metric-label">Durée</div>
          <div class="analytics-metric-value">${formatDuration(this.metrics.duration)}</div>
        </div>
        
        <div class="analytics-metric">
          <div class="analytics-metric-icon">🖱️</div>
          <div class="analytics-metric-label">Interactions</div>
          <div class="analytics-metric-value">${this.metrics.totalEvents}</div>
        </div>
        
        <div class="analytics-metric">
          <div class="analytics-metric-icon">📊</div>
          <div class="analytics-metric-label">Engagement</div>
          <div class="analytics-metric-value">${this.metrics.engagementRate}</div>
          <div class="analytics-metric-trend positive">actions/min</div>
        </div>
        
        <div class="analytics-metric">
          <div class="analytics-metric-icon">📜</div>
          <div class="analytics-metric-label">Scroll</div>
          <div class="analytics-metric-value">${this.metrics.scrollDepth}%</div>
        </div>
      </div>

      <!-- UX Score -->
      <div class="analytics-section">
        <div class="analytics-section-header">
          <div class="analytics-section-title">🎯 UX Score</div>
          <button id="analytics-export-pdf" class="btn-export-pdf" title="Export Analytics en PDF">
            📄 Export PDF
          </button>
        </div>
        
        <div class="ux-score-container">
          <div class="ux-score-gauge">
            <svg viewBox="0 0 200 200" class="ux-score-circle">
              <circle cx="100" cy="100" r="80" class="ux-score-bg"></circle>
              <circle cx="100" cy="100" r="80" 
                      class="ux-score-progress ${this.getScoreClass(this.metrics.uxScore.overall)}"
                      stroke-dasharray="${this.getCircleProgress(this.metrics.uxScore.overall)} 502.4"
                      stroke-dashoffset="0"></circle>
            </svg>
            <div class="ux-score-text">
              <div class="ux-score-value">${this.metrics.uxScore.overall}</div>
              <div class="ux-score-label">Score</div>
            </div>
          </div>
          
          <div class="ux-score-breakdown">
            <div class="ux-score-component">
              <div class="ux-score-component-header">
                <span class="ux-score-component-name">Engagement</span>
                <span class="ux-score-component-value">${this.metrics.uxScore.engagement}</span>
              </div>
              <div class="ux-score-component-bar">
                <div class="ux-score-component-fill" 
                     style="width: ${this.metrics.uxScore.engagement}%; background: #3b82f6;"></div>
              </div>
            </div>
            
            <div class="ux-score-component">
              <div class="ux-score-component-header">
                <span class="ux-score-component-name">Efficacité</span>
                <span class="ux-score-component-value">${this.metrics.uxScore.efficiency}</span>
              </div>
              <div class="ux-score-component-bar">
                <div class="ux-score-component-fill" 
                     style="width: ${this.metrics.uxScore.efficiency}%; background: #10b981;"></div>
              </div>
            </div>
            
            <div class="ux-score-component">
              <div class="ux-score-component-header">
                <span class="ux-score-component-name">Sans frustration</span>
                <span class="ux-score-component-value">${this.metrics.uxScore.frustration}</span>
              </div>
              <div class="ux-score-component-bar">
                <div class="ux-score-component-fill" 
                     style="width: ${this.metrics.uxScore.frustration}%; background: #f59e0b;"></div>
              </div>
            </div>
          </div>
        </div>
      </div>

      <!-- UX Patterns -->
      <div class="analytics-section">
        <div class="analytics-section-header">
          <div class="analytics-section-title">🔍 Patterns UX Détectés</div>
          <div class="analytics-section-subtitle">${patternsDetector.getSummary()}</div>
        </div>

        <div class="patterns-grid">
          <div class="pattern-card ${this.patterns.rage_clicks.length > 0 ? 'has-pattern' : ''}">
            <div class="pattern-card-header">
              <div class="pattern-card-icon danger">🔥</div>
              <div class="pattern-card-info">
                <div class="pattern-card-title">Rage Clicks</div>
                <div class="pattern-card-count">${this.patterns.rage_clicks.length}</div>
              </div>
            </div>
            ${this.patterns.rage_clicks.length > 0 ? `
              <div class="pattern-card-body">
                <div class="pattern-card-description">Clics rapides répétés au même endroit</div>
                <div class="pattern-card-details">
                  ${this.patterns.rage_clicks.slice(0, 3).map(rc => `
                    <div class="pattern-detail">
                      <span>${rc.clickCount} clics</span>
                      <span>Intensité: ${rc.intensity}/10</span>
                    </div>
                  `).join('')}
                </div>
              </div>
            ` : ''}
          </div>

          <div class="pattern-card ${this.patterns.dead_clicks.length > 0 ? 'has-pattern' : ''}">
            <div class="pattern-card-header">
              <div class="pattern-card-icon danger">💀</div>
              <div class="pattern-card-info">
                <div class="pattern-card-title">Dead Clicks</div>
                <div class="pattern-card-count">${this.patterns.dead_clicks.length}</div>
              </div>
            </div>
            ${this.patterns.dead_clicks.length > 0 ? `
              <div class="pattern-card-body">
                <div class="pattern-card-description">Clics sur éléments non-interactifs</div>
                <div class="pattern-card-details">
                  ${this.patterns.dead_clicks.slice(0, 3).map(dc => `
                    <div class="pattern-detail">
                      <span>${dc.element}</span>
                    </div>
                  `).join('')}
                </div>
              </div>
            ` : ''}
          </div>

          <div class="pattern-card ${this.patterns.confusion_zones.length > 0 ? 'has-pattern' : ''}">
            <div class="pattern-card-header">
              <div class="pattern-card-icon warning">😕</div>
              <div class="pattern-card-info">
                <div class="pattern-card-title">Confusion Zones</div>
                <div class="pattern-card-count">${this.patterns.confusion_zones.length}</div>
              </div>
            </div>
            ${this.patterns.confusion_zones.length > 0 ? `
              <div class="pattern-card-body">
                <div class="pattern-card-description">Hovers prolongés sans action</div>
                <div class="pattern-card-details">
                  ${this.patterns.confusion_zones.slice(0, 3).map(cz => `
                    <div class="pattern-detail">
                      <span>${(cz.duration / 1000).toFixed(1)}s d'hésitation</span>
                    </div>
                  `).join('')}
                </div>
              </div>
            ` : ''}
          </div>

          <div class="pattern-card ${this.patterns.form_abandonments.length > 0 ? 'has-pattern' : ''}">
            <div class="pattern-card-header">
              <div class="pattern-card-icon warning">📝❌</div>
              <div class="pattern-card-info">
                <div class="pattern-card-title">Form Abandonment</div>
                <div class="pattern-card-count">${this.patterns.form_abandonments.length}</div>
              </div>
            </div>
            ${this.patterns.form_abandonments.length > 0 ? `
              <div class="pattern-card-body">
                <div class="pattern-card-description">Champs remplis puis effacés</div>
                <div class="pattern-card-details">
                  ${this.patterns.form_abandonments.slice(0, 3).map(fa => `
                    <div class="pattern-detail">
                      <span>${fa.element} (${fa.fieldType})</span>
                    </div>
                  `).join('')}
                </div>
              </div>
            ` : ''}
          </div>

          <div class="pattern-card ${this.patterns.repetitive_scrolls.length > 0 ? 'has-pattern' : ''}">
            <div class="pattern-card-header">
              <div class="pattern-card-icon info">🔄</div>
              <div class="pattern-card-info">
                <div class="pattern-card-title">Repetitive Scrolls</div>
                <div class="pattern-card-count">${this.patterns.repetitive_scrolls.length}</div>
              </div>
            </div>
            ${this.patterns.repetitive_scrolls.length > 0 ? `
              <div class="pattern-card-body">
                <div class="pattern-card-description">Scrolls va-et-vient (recherche)</div>
                <div class="pattern-card-details">
                  ${this.patterns.repetitive_scrolls.slice(0, 3).map(rs => `
                    <div class="pattern-detail">
                      <span>${rs.changes} changements de direction</span>
                    </div>
                  `).join('')}
                </div>
              </div>
            ` : ''}
          </div>
        </div>
      </div>

      <!-- AI Analysis Section -->
      ${this.renderAISection()}

      <!-- Insights -->
      <div class="analytics-section">
        <div class="analytics-section-header">
          <div class="analytics-section-title">💡 Insights automatiques</div>
        </div>
        
        <div class="analytics-insights">
          ${insights.map(insight => `
            <div class="analytics-insight ${insight.type}">
              <div class="analytics-insight-icon">${insight.icon}</div>
              <div class="analytics-insight-content">
                <div class="analytics-insight-title">${insight.title}</div>
                <div class="analytics-insight-description">${insight.description}</div>
                <div class="analytics-insight-recommendation">${insight.recommendation}</div>
              </div>
            </div>
          `).join('')}
        </div>
      </div>

      <!-- Notes personnelles -->
      <div class="analytics-section">
        <div class="analytics-section-header">
          <div class="analytics-section-title">📝 Notes personnelles</div>
          <div class="analytics-section-subtitle">Mémo et observations</div>
        </div>
        
        <div class="analytics-notes">
          <textarea 
            id="analytics-notes-textarea"
            class="analytics-notes-textarea"
            placeholder="Ajoutez vos observations, remarques ou points à retenir sur cette session...
            
Exemples :
• Problème UX identifié sur l'élément X
• L'utilisateur semble chercher Y
• À améliorer : Z"
            rows="8">${this.notes || ''}</textarea>
          <div class="analytics-notes-info">
            <span class="analytics-notes-icon">💾</span>
            Les notes sont sauvegardées automatiquement
          </div>
        </div>
      </div>

      <!-- Timeline -->
      <div class="analytics-section">
        <div class="analytics-section-header">
          <div class="analytics-section-title">📅 Parcours utilisateur</div>
        </div>
        
        <div class="analytics-timeline">
          ${this.renderTimeline()}
        </div>
      </div>
    `;
  }

  /**
   * Affiche la timeline
   */
  renderTimeline() {
    // Essayer d'abord avec session.data.pages
    let pages = this.session.data.pages || [];
    
    // Si pas de pages, créer une entrée avec la session actuelle
    if (pages.length === 0) {
      // Chercher les événements de navigation
      const navEvents = this.session.data.events.filter(e => e.type === 'navigation');
      
      if (navEvents.length === 0) {
        // Créer une entrée unique avec les infos de la session
        const sessionInfo = this.session.data.session || this.session.metadata;
        return `
          <div class="analytics-timeline-item">
            <div class="analytics-timeline-marker"></div>
            <div class="analytics-timeline-content">
              <div class="analytics-timeline-time">0s</div>
              <div class="analytics-timeline-title">Session unique</div>
              <div class="analytics-timeline-description">
                ${truncateUrl(sessionInfo.url || 'Page testée', 60)}<br>
                <small>Durée totale: ${formatDuration(sessionInfo.duration)}</small>
              </div>
            </div>
          </div>
        `;
      }
      
      // Utiliser les événements de navigation
      pages = navEvents.map(event => ({
        url: event.url,
        timestamp: event.timestamp
      }));
    }

    return pages.map((page, index) => {
      const sessionInfo = this.session.data.session || this.session.metadata;
      const time = page.timestamp - sessionInfo.startTime;
      const duration = index < pages.length - 1 
        ? pages[index + 1].timestamp - page.timestamp 
        : sessionInfo.endTime - page.timestamp;

      return `
        <div class="analytics-timeline-item">
          <div class="analytics-timeline-marker"></div>
          <div class="analytics-timeline-content">
            <div class="analytics-timeline-time">${formatDuration(time)}</div>
            <div class="analytics-timeline-title">Page ${index + 1}</div>
            <div class="analytics-timeline-description">
              ${truncateUrl(page.url, 60)}<br>
              <small>Temps passé: ${formatDuration(duration)}</small>
            </div>
          </div>
        </div>
      `;
    }).join('');
  }

  /**
   * Classe CSS selon le score
   */
  getScoreClass(score) {
    if (score >= 80) return 'excellent';
    if (score >= 60) return 'good';
    if (score >= 40) return 'average';
    return 'poor';
  }

  /**
   * Calcule la progress du cercle
   */
  getCircleProgress(score) {
    const circumference = 502.4; // 2 * PI * 80
    return (score / 100) * circumference;
  }

  /**
   * Gère le lancement de l'analyse IA
   */
  async handleAIAnalysis() {
    // Vérifier si l'API est configurée
    if (!aiAnalyzer.checkConfiguration()) {
      this.showAISettingsModal();
      return;
    }

    // Montrer la modal de preview
    this.showAIPreviewModal();
  }

  /**
   * Affiche la modal des paramètres IA
   */
  showAISettingsModal() {
    const modal = document.getElementById('modal-ai-settings');
    const input = document.getElementById('ai-api-key-input');
    const status = document.getElementById('ai-connection-status');

    // Charger la clé actuelle si elle existe
    const currentKey = aiAnalyzer.loadApiKey();
    if (currentKey) {
      input.value = currentKey;
    }

    status.innerHTML = '';
    modal.classList.add('active');
  }

  /**
   * Attache les listeners de la modal AI Settings
   */
  attachAISettingsListeners() {
    const modal = document.getElementById('modal-ai-settings');
    const btnSave = document.getElementById('btn-ai-settings-save');
    const btnCancel = document.getElementById('btn-ai-settings-cancel');
    const btnTest = document.getElementById('btn-ai-settings-test');
    const backdrop = modal?.querySelector('.modal-backdrop');
    const input = document.getElementById('ai-api-key-input');
    const status = document.getElementById('ai-connection-status');

    if (btnSave) {
      btnSave.addEventListener('click', () => {
        const apiKey = input.value.trim();

        if (!apiKey) {
          status.innerHTML = '<div class="ai-status-error">⚠️ Veuillez entrer une clé API</div>';
          return;
        }

        try {
          aiAnalyzer.configure(apiKey);
          status.innerHTML = '<div class="ai-status-success">✅ Configuration enregistrée</div>';

          setTimeout(() => {
            modal.classList.remove('active');
            this.render(); // Re-render pour afficher le bouton AI
          }, 1000);
        } catch (error) {
          status.innerHTML = `<div class="ai-status-error">⚠️ Erreur: ${error.message}</div>`;
        }
      });
    }

    if (btnTest) {
      btnTest.addEventListener('click', async () => {
        const apiKey = input.value.trim();

        if (!apiKey) {
          status.innerHTML = '<div class="ai-status-error">⚠️ Veuillez entrer une clé API</div>';
          return;
        }

        status.innerHTML = '<div class="ai-status-loading">⏳ Test de connexion...</div>';

        try {
          aiAnalyzer.configure(apiKey);
          const result = await aiAnalyzer.testConnection();

          if (result.success) {
            status.innerHTML = '<div class="ai-status-success">✅ Connexion réussie !</div>';
          } else {
            status.innerHTML = `<div class="ai-status-error">❌ ${result.message}</div>`;
          }
        } catch (error) {
          status.innerHTML = `<div class="ai-status-error">❌ Erreur: ${error.message}</div>`;
        }
      });
    }

    if (btnCancel) {
      btnCancel.addEventListener('click', () => {
        modal.classList.remove('active');
      });
    }

    if (backdrop) {
      backdrop.addEventListener('click', () => {
        modal.classList.remove('active');
      });
    }
  }

  /**
   * Affiche la modal de preview des données anonymisées
   */
  showAIPreviewModal() {
    const modal = document.getElementById('modal-ai-preview');

    // Anonymiser les données
    const anonymizationResult = window.dataAnonymizer?.anonymizeSession(this.session) ||
                                 { anonymized_data: {}, domain_mappings: {}, domains_detected: [] };

    const { anonymized_data, domain_mappings, domains_detected } = anonymizationResult;

    // Remplir le résumé
    const summaryContainer = document.getElementById('ai-preview-summary');
    summaryContainer.innerHTML = `
      <div class="ai-preview-stats">
        <div class="ai-preview-stat">
          <div class="ai-preview-stat-label">Événements analysés</div>
          <div class="ai-preview-stat-value">${anonymized_data.events_summary?.total || 0}</div>
        </div>
        <div class="ai-preview-stat">
          <div class="ai-preview-stat-label">Patterns détectés</div>
          <div class="ai-preview-stat-value">${anonymized_data.patterns?.total || 0}</div>
        </div>
        <div class="ai-preview-stat">
          <div class="ai-preview-stat-label">Domaines anonymisés</div>
          <div class="ai-preview-stat-value">${domains_detected.length}</div>
        </div>
        <div class="ai-preview-stat">
          <div class="ai-preview-stat-label">Device</div>
          <div class="ai-preview-stat-value">${anonymized_data.metadata?.device_type || 'unknown'}</div>
        </div>
      </div>
    `;

    // Remplir les données JSON
    const jsonContainer = document.getElementById('ai-preview-json');
    jsonContainer.textContent = JSON.stringify(anonymized_data, null, 2);

    // Remplir le rapport d'anonymisation
    const reportContainer = document.getElementById('ai-preview-report');
    reportContainer.innerHTML = `
      <div class="ai-preview-report">
        <div class="ai-preview-report-section">
          <h4>🔒 Données supprimées</h4>
          <ul>
            <li>URLs complètes (remplacées par domaines anonymes)</li>
            <li>Sélecteurs CSS (IDs, classes, attributs masqués)</li>
            <li>Messages d'erreur (URLs et chemins supprimés)</li>
            <li>Screenshots (images supprimées, nombre conservé)</li>
            <li>Textes saisis (complètement supprimés)</li>
            <li>ID de session (hashé)</li>
          </ul>
        </div>

        <div class="ai-preview-report-section">
          <h4>🔑 Mappings de domaines</h4>
          <div class="ai-preview-mappings">
            ${Object.entries(domain_mappings).map(([original, anonymized]) => `
              <div class="ai-preview-mapping">
                <span class="ai-preview-mapping-original">${original}</span>
                <span class="ai-preview-mapping-arrow">→</span>
                <span class="ai-preview-mapping-anonymized">${anonymized}</span>
              </div>
            `).join('')}
            ${Object.keys(domain_mappings).length === 0 ? '<p class="ai-preview-no-data">Aucun domaine détecté</p>' : ''}
          </div>
        </div>
      </div>
    `;

    // Stocker les données pour confirmation
    this.pendingAIData = anonymized_data;

    modal.classList.add('active');
  }

  /**
   * Attache les listeners de la modal AI Preview
   */
  attachAIPreviewListeners() {
    const modal = document.getElementById('modal-ai-preview');
    const btnConfirm = document.getElementById('btn-ai-preview-confirm');
    const btnCancel = document.getElementById('btn-ai-preview-cancel');
    const backdrop = modal?.querySelector('.modal-backdrop');

    // Tabs
    const tabButtons = document.querySelectorAll('.ai-preview-tab');
    tabButtons.forEach(btn => {
      btn.addEventListener('click', () => {
        const tabName = btn.getAttribute('data-preview-tab');

        // Activer le tab
        tabButtons.forEach(b => b.classList.remove('active'));
        btn.classList.add('active');

        // Afficher le contenu
        document.querySelectorAll('.ai-preview-tab-content').forEach(content => {
          content.classList.remove('active');
        });
        document.getElementById(`ai-preview-${tabName}`).classList.add('active');
      });
    });

    if (btnConfirm) {
      btnConfirm.addEventListener('click', async () => {
        modal.classList.remove('active');
        await this.performAIAnalysis();
      });
    }

    if (btnCancel) {
      btnCancel.addEventListener('click', () => {
        modal.classList.remove('active');
      });
    }

    if (backdrop) {
      backdrop.addEventListener('click', () => {
        modal.classList.remove('active');
      });
    }
  }

  /**
   * Effectue l'analyse IA
   */
  async performAIAnalysis() {
    this.aiLoading = true;
    this.render(); // Re-render pour afficher le loading

    try {
      console.log('[Analytics] Starting AI analysis...');
      this.aiAnalysis = await aiAnalyzer.analyzeSession(this.session);
      console.log('[Analytics] AI analysis complete:', this.aiAnalysis);
    } catch (error) {
      console.error('[Analytics] AI analysis failed:', error);
      this.aiAnalysis = {
        error: true,
        message: error.message
      };
    }

    this.aiLoading = false;
    this.render(); // Re-render pour afficher les résultats
  }

  /**
   * Génère le HTML de la section AI
   */
  renderAISection() {
    const isConfigured = aiAnalyzer.checkConfiguration();

    // Si pas configuré, afficher le bouton de configuration
    if (!isConfigured) {
      return `
        <div class="analytics-section">
          <div class="analytics-section-header">
            <div class="analytics-section-title">🤖 Analyse IA avec Gemini</div>
            <div class="analytics-section-subtitle">Insights automatiques avancés</div>
          </div>
          <div class="ai-not-configured">
            <div class="ai-not-configured-icon">🔧</div>
            <h3>Configuration requise</h3>
            <p>Configurez votre clé API Google Gemini pour accéder à l'analyse IA</p>
            <button id="btn-ai-settings" class="btn-primary">⚙️ Configurer l'API</button>
          </div>
        </div>
      `;
    }

    // Si en cours de chargement
    if (this.aiLoading) {
      return `
        <div class="analytics-section">
          <div class="analytics-section-header">
            <div class="analytics-section-title">🤖 Analyse IA avec Gemini</div>
          </div>
          <div class="ai-loading">
            <div class="ai-loading-spinner"></div>
            <div class="ai-loading-text">Analyse en cours...</div>
            <div class="ai-loading-subtext">Traitement des données anonymisées par Gemini</div>
          </div>
        </div>
      `;
    }

    // Si erreur
    if (this.aiAnalysis?.error) {
      return `
        <div class="analytics-section">
          <div class="analytics-section-header">
            <div class="analytics-section-title">🤖 Analyse IA avec Gemini</div>
            <button id="btn-ai-settings" class="btn-secondary btn-small">⚙️ Paramètres</button>
          </div>
          <div class="ai-error">
            <div class="ai-error-icon">⚠️</div>
            <h3>Erreur d'analyse</h3>
            <p>${this.aiAnalysis.message}</p>
            <button id="btn-ai-analyze" class="btn-primary">🔄 Réessayer</button>
          </div>
        </div>
      `;
    }

    // Si pas encore d'analyse
    if (!this.aiAnalysis) {
      return `
        <div class="analytics-section">
          <div class="analytics-section-header">
            <div class="analytics-section-title">🤖 Analyse IA avec Gemini</div>
            <div class="analytics-section-subtitle">Insights automatiques avancés</div>
            <button id="btn-ai-settings" class="btn-secondary btn-small">⚙️ Paramètres</button>
          </div>
          <div class="ai-ready">
            <div class="ai-ready-icon">✨</div>
            <h3>Prêt pour l'analyse IA</h3>
            <p>Obtenez des insights avancés générés par intelligence artificielle</p>
            <ul class="ai-ready-features">
              <li>📊 Analyse comportementale approfondie</li>
              <li>💡 Recommandations personnalisées</li>
              <li>🎯 Identification des points de friction</li>
              <li>🔒 Données 100% anonymisées</li>
            </ul>
            <button id="btn-ai-analyze" class="btn-primary">🚀 Lancer l'analyse IA</button>
          </div>
        </div>
      `;
    }

    // Afficher les résultats de l'analyse
    const analysis = this.aiAnalysis;

    return `
      <div class="analytics-section">
        <div class="analytics-section-header">
          <div class="analytics-section-title">🤖 Analyse IA avec Gemini</div>
          <div class="analytics-section-actions">
            <button id="btn-ai-analyze" class="btn-secondary btn-small">🔄 Re-analyser</button>
            <button id="btn-ai-settings" class="btn-secondary btn-small">⚙️ Paramètres</button>
          </div>
        </div>

        <!-- UX Score IA -->
        <div class="ai-score-container">
          <div class="ai-score-circle ${this.getScoreClass(analysis.ux_score * 10)}">
            <div class="ai-score-value">${analysis.ux_score}/10</div>
            <div class="ai-score-label">Score UX</div>
          </div>
          <div class="ai-score-explanation">
            <strong>Évaluation IA :</strong> ${analysis.ux_score_explanation}
          </div>
        </div>

        <!-- Résumé -->
        <div class="ai-summary">
          <h4>📝 Résumé</h4>
          <p>${analysis.summary}</p>
        </div>

        <!-- Insights clés -->
        ${analysis.key_insights && analysis.key_insights.length > 0 ? `
          <div class="ai-key-insights">
            <h4>💡 Insights clés</h4>
            <ul>
              ${analysis.key_insights.map(insight => `<li>${insight}</li>`).join('')}
            </ul>
          </div>
        ` : ''}

        <!-- Points de friction -->
        ${analysis.friction_points && analysis.friction_points.length > 0 ? `
          <div class="ai-friction-points">
            <h4>⚠️ Points de friction</h4>
            <div class="ai-friction-grid">
              ${analysis.friction_points.map(friction => `
                <div class="ai-friction-card ${friction.severity}">
                  <div class="ai-friction-header">
                    <span class="ai-friction-title">${friction.title}</span>
                    <span class="ai-friction-severity">${friction.severity}</span>
                  </div>
                  <div class="ai-friction-description">${friction.description}</div>
                  <div class="ai-friction-location">📍 ${friction.location}</div>
                </div>
              `).join('')}
            </div>
          </div>
        ` : ''}

        <!-- Points positifs -->
        ${analysis.positive_points && analysis.positive_points.length > 0 ? `
          <div class="ai-positive-points">
            <h4>✅ Points positifs</h4>
            <div class="ai-positive-grid">
              ${analysis.positive_points.map(point => `
                <div class="ai-positive-card">
                  <div class="ai-positive-title">${point.title}</div>
                  <div class="ai-positive-description">${point.description}</div>
                </div>
              `).join('')}
            </div>
          </div>
        ` : ''}

        <!-- Recommandations -->
        ${analysis.recommendations && analysis.recommendations.length > 0 ? `
          <div class="ai-recommendations">
            <h4>🎯 Recommandations</h4>
            <div class="ai-recommendations-grid">
              ${analysis.recommendations.map(rec => `
                <div class="ai-recommendation-card">
                  <div class="ai-recommendation-header">
                    <span class="ai-recommendation-title">${rec.title}</span>
                    <div class="ai-recommendation-badges">
                      <span class="ai-badge impact-${rec.impact}">Impact: ${rec.impact}</span>
                      <span class="ai-badge effort-${rec.effort}">Effort: ${rec.effort}</span>
                    </div>
                  </div>
                  <div class="ai-recommendation-description">${rec.description}</div>
                  <div class="ai-recommendation-category">🏷️ ${rec.category}</div>
                </div>
              `).join('')}
            </div>
          </div>
        ` : ''}

        <!-- Metadata -->
        <div class="ai-metadata">
          <div class="ai-metadata-item">
            <span class="ai-metadata-label">Analysé le:</span>
            <span class="ai-metadata-value">${new Date(analysis.metadata?.analyzed_at).toLocaleString('fr-FR')}</span>
          </div>
          <div class="ai-metadata-item">
            <span class="ai-metadata-label">Modèle:</span>
            <span class="ai-metadata-value">${analysis.metadata?.model_used || 'Gemini 1.5 Flash'}</span>
          </div>
          <div class="ai-metadata-item">
            <span class="ai-metadata-label">Priorité:</span>
            <span class="ai-metadata-value ai-priority-${analysis.priority}">${analysis.priority}</span>
          </div>
          ${analysis.is_fallback ? '<div class="ai-metadata-warning">⚠️ Analyse générée localement (fallback)</div>' : ''}
        </div>
      </div>
    `;
  }

  // OPTIM #5: Removed duplicated utility functions
  // Now using utils.js: formatDuration, truncateUrl
}

// Export singleton
export default new AnalyticsManager();
