/**
 * SUMO UX Tracker - AI Analyzer
 * Intégration avec Google Gemini pour analyse IA des sessions
 */

import dataAnonymizer from "./data-anonymizer.js";

class AIAnalyzer {
  constructor() {
    this.apiKey = null;
    this.isConfigured = false;
    this.analysisCache = new Map(); // Cache des analyses
    this.lastAnalysis = null;
    this.geminiEndpoint =
      "https://generativelanguage.googleapis.com/v1beta/models/gemini-2.0-flash:generateContent";
  }

  /**
   * Configure l'API key Gemini
   */
  configure(apiKey) {
    if (!apiKey || apiKey.trim().length === 0) {
      throw new Error("API key is required");
    }

    this.apiKey = apiKey.trim();
    this.isConfigured = true;

    // Sauvegarder dans localStorage (chiffré basique)
    this.saveApiKey(apiKey);

    console.log("[AIAnalyzer] API key configured successfully");
  }

  /**
   * Vérifie si l'analyseur est configuré
   */
  checkConfiguration() {
    if (!this.isConfigured) {
      // Tenter de charger depuis localStorage
      const savedKey = this.loadApiKey();
      if (savedKey) {
        this.apiKey = savedKey;
        this.isConfigured = true;
        return true;
      }
      return false;
    }
    return true;
  }

  /**
   * Sauvegarde l'API key (encodée en base64)
   */
  saveApiKey(apiKey) {
    try {
      const encoded = btoa(apiKey);
      localStorage.setItem("sumo_gemini_key", encoded);
    } catch (e) {
      console.error("[AIAnalyzer] Failed to save API key:", e);
    }
  }

  /**
   * Charge l'API key depuis localStorage
   */
  loadApiKey() {
    try {
      const encoded = localStorage.getItem("sumo_gemini_key");
      if (encoded) {
        return atob(encoded);
      }
    } catch (e) {
      console.error("[AIAnalyzer] Failed to load API key:", e);
    }
    return null;
  }

  /**
   * Supprime l'API key
   */
  clearApiKey() {
    this.apiKey = null;
    this.isConfigured = false;
    localStorage.removeItem("sumo_gemini_key");
    console.log("[AIAnalyzer] API key cleared");
  }

  /**
   * Analyse une session avec Gemini
   */
  async analyzeSession(session, options = {}) {
    if (!this.checkConfiguration()) {
      throw new Error(
        "AI Analyzer not configured. Please set your Gemini API key.",
      );
    }

    console.log("[AIAnalyzer] Starting session analysis...");

    // Vérifier cache
    const cacheKey = `analysis_${session.id}`;
    if (this.analysisCache.has(cacheKey) && !options.forceRefresh) {
      console.log("[AIAnalyzer] Returning cached analysis");
      return this.analysisCache.get(cacheKey);
    }

    try {
      // Analyser les funnels si disponibles
      let funnelResults = null;
      if (options.funnelManager) {
        funnelResults = options.funnelManager.analyzeSessionFunnels(session);
      }

      // Étape 1 : Anonymiser les données (avec funnels et screenshots si feedbacks présents)
      const hasFeedbacks = session.data?.feedbacks?.length > 0;
      const anonymizationResult = dataAnonymizer.anonymizeSession(
        session,
        funnelResults,
        { includeScreenshots: hasFeedbacks } // Inclure screenshots seulement si feedbacks présents
      );
      const { anonymized_data, domain_mappings, domains_detected } =
        anonymizationResult;

      // Valider l'anonymisation
      const validation = dataAnonymizer.validateAnonymization(anonymized_data);
      if (!validation.is_safe) {
        console.warn(
          "[AIAnalyzer] Anonymization warnings:",
          validation.warnings,
        );
      }

      // Étape 2 : Construire le prompt pour Gemini (avec images si présentes)
      const hasVisualFeedbacks = anonymized_data.feedbacks?.screenshots?.length > 0;
      const prompt = this.buildAnalysisPrompt(anonymized_data, hasVisualFeedbacks);

      // Étape 3 : Appeler l'API Gemini (avec support vision si feedbacks visuels)
      const geminiResponse = hasVisualFeedbacks
        ? await this.callGeminiVisionAPI(prompt, anonymized_data.feedbacks.screenshots)
        : await this.callGeminiAPI(prompt);

      // Étape 4 : Parser et structurer la réponse
      const analysis = this.parseGeminiResponse(
        geminiResponse,
        anonymized_data,
      );

      // Ajouter metadata
      analysis.metadata = {
        analyzed_at: Date.now(),
        domains_analyzed: domains_detected.length,
        domain_mappings: domain_mappings,
        anonymization_report:
          dataAnonymizer.generateAnonymizationReport(anonymizationResult),
        model_used: "gemini-2.0-flash",
        visual_analysis: hasVisualFeedbacks,
        feedback_screenshots_analyzed: anonymized_data.feedbacks?.screenshots?.length || 0,
      };

      // Cache l'analyse
      this.analysisCache.set(cacheKey, analysis);
      this.lastAnalysis = analysis;

      console.log("[AIAnalyzer] Analysis complete");

      return analysis;
    } catch (error) {
      console.error("[AIAnalyzer] Analysis failed:", error);
      throw new Error(`AI Analysis failed: ${error.message}`);
    }
  }

  /**
   * Construit le prompt pour Gemini
   */
  buildAnalysisPrompt(anonymizedData, hasVisualFeedbacks = false) {
    const visualInstructions = hasVisualFeedbacks ? `

**ANALYSE VISUELLE DES FEEDBACKS** : Des captures d'écran annotées par l'utilisateur sont fournies avec cette session. Ces images montrent :
- Les zones surlignées (highlight) : Éléments que l'utilisateur a voulu mettre en évidence
- Les flèches (arrow) : Relations ou flux pointés par l'utilisateur
- Les dessins libres (freehand) : Annotations spontanées de l'utilisateur
- Les post-its jaunes : Commentaires généraux de l'utilisateur
- Les post-its rouges : Pain points / points de frustration marqués visuellement

EXAMINE ATTENTIVEMENT ces captures d'écran pour identifier :
1. Les zones problématiques marquées visuellement (surlignages, flèches vers des éléments défaillants)
2. Le contexte visuel des commentaires écrits dans les post-its
3. Les patterns visuels de frustration (accumulation d'annotations dans certaines zones)
4. Les éléments d'interface pointés par les flèches ou surlignés

Intègre cette analyse visuelle dans tes recommandations pour être plus précis et contextualisé.` : '';

    const prompt = `Tu es un expert en UX (User Experience) et en analyse comportementale.

Analyse cette session utilisateur anonymisée et fournis une analyse détaillée EN FRANÇAIS en JSON structuré.

**Données de la session :**
${JSON.stringify(anonymizedData, null, 2)}
${visualInstructions}

**Instructions :**
1. Génère un résumé concis de la session EN FRANÇAIS (2-3 phrases)
2. Identifie les points de friction principaux (basé sur patterns, événements${hasVisualFeedbacks ? ', feedbacks textuels ET annotations visuelles' : ' ET feedbacks utilisateur'})
3. Analyse les feedbacks utilisateur (commentaires positifs et frustrations${hasVisualFeedbacks ? ' + annotations visuelles sur les captures d\'écran' : ''}) pour comprendre le ressenti
4. **Analyse les funnels de conversion** : Si des funnels sont détectés, analyse le taux de complétion, les points d'abandon, et recommande des optimisations
5. Évalue la fluidité du parcours utilisateur (score 0-10)
6. Liste des recommandations concrètes et actionnables EN FRANÇAIS (minimum 3, maximum 7)
7. Identifie les points positifs du parcours EN FRANÇAIS
8. Détermine la priorité d'action (low, medium, high, critical)

**ATTENTION AUX FEEDBACKS** : Si la session contient des feedbacks utilisateur (commentaires de frustration ou positifs), ils sont TRÈS IMPORTANTS et doivent être pris en compte en priorité dans l'analyse. Les feedbacks révèlent le ressenti réel de l'utilisateur. Si aucun feedback n'est présent (feedbacks.total = 0), indique "Aucun feedback utilisateur" dans l'analyse.

**ATTENTION AUX FUNNELS** : Si des funnels de conversion sont analysés (funnels.total > 0), identifie :
- Quels funnels ont été complétés vs abandonnés
- À quelle étape les utilisateurs abandonnent (abandonment_point)
- Le temps moyen pour compléter un funnel
- Recommandations spécifiques pour améliorer le taux de conversion

**IMPORTANT : Tous les textes (title, description, etc.) doivent être EN FRANÇAIS.**

**Format de réponse attendu (JSON strict) :**
{
  "summary": "Résumé de la session en 2-3 phrases EN FRANÇAIS",
  "ux_score": 7,
  "ux_score_explanation": "Explication du score EN FRANÇAIS",
  "friction_points": [
    {
      "title": "Titre du point de friction EN FRANÇAIS",
      "description": "Description détaillée EN FRANÇAIS",
      "severity": "high | medium | low",
      "location": "Description de où ça se produit EN FRANÇAIS"
    }
  ],
  "positive_points": [
    {
      "title": "Titre du point positif EN FRANÇAIS",
      "description": "Description EN FRANÇAIS"
    }
  ],
  "recommendations": [
    {
      "title": "Titre de la recommandation EN FRANÇAIS",
      "description": "Description détaillée et actionnable EN FRANÇAIS",
      "impact": "high | medium | low",
      "effort": "high | medium | low",
      "category": "design | technical | content | navigation"
    }
  ],
  "priority": "critical | high | medium | low",
  "key_insights": [
    "Insight 1 EN FRANÇAIS",
    "Insight 2 EN FRANÇAIS",
    "Insight 3 EN FRANÇAIS"
  ],
  "user_feedback_analysis": {
    "sentiment": "positive | mixed | negative",
    "main_frustrations": ["Frustration 1 EN FRANÇAIS", "Frustration 2 EN FRANÇAIS"],
    "main_appreciations": ["Appréciation 1 EN FRANÇAIS", "Appréciation 2 EN FRANÇAIS"],
    "feedback_insights": "Synthèse des feedbacks utilisateur EN FRANÇAIS"
  },
  "funnel_analysis": {
    "funnels_detected": 0,
    "completion_rates": [],
    "main_drop_off_points": ["Étape d'abandon 1 EN FRANÇAIS", "Étape d'abandon 2 EN FRANÇAIS"],
    "optimization_recommendations": ["Recommandation funnel 1 EN FRANÇAIS", "Recommandation funnel 2 EN FRANÇAIS"],
    "conversion_insights": "Synthèse de l'analyse des funnels EN FRANÇAIS"
  }
}

**Réponds UNIQUEMENT avec le JSON, sans texte additionnel avant ou après. TOUS LES TEXTES DOIVENT ÊTRE EN FRANÇAIS.**`;

    return prompt;
  }

  /**
   * Appelle l'API Gemini
   */
  async callGeminiAPI(prompt) {
    const url = this.geminiEndpoint;

    const requestBody = {
      contents: [
        {
          parts: [
            {
              text: prompt,
            },
          ],
        },
      ],
      generationConfig: {
        temperature: 0.7,
        topK: 40,
        topP: 0.95,
        maxOutputTokens: 2048,
      },
    };

    console.log("[AIAnalyzer] Calling Gemini API...");

    const response = await fetch(url, {
      method: "POST",
      headers: {
        "Content-Type": "application/json",
        "X-goog-api-key": this.apiKey,
      },
      body: JSON.stringify(requestBody),
    });

    if (!response.ok) {
      const errorData = await response.json().catch(() => ({}));
      throw new Error(
        `Gemini API error: ${response.status} - ${errorData.error?.message || "Unknown error"}`,
      );
    }

    const data = await response.json();

    if (!data.candidates || data.candidates.length === 0) {
      throw new Error("No response from Gemini API");
    }

    const textResponse = data.candidates[0].content.parts[0].text;
    console.log("[AIAnalyzer] Received response from Gemini");

    return textResponse;
  }

  /**
   * Appelle l'API Gemini avec support vision (pour analyse des screenshots)
   */
  async callGeminiVisionAPI(prompt, screenshots) {
    const url = this.geminiEndpoint;

    console.log(`[AIAnalyzer] Calling Gemini Vision API with ${screenshots.length} screenshot(s)...`);

    // Construire les parts : texte + images
    const parts = [
      {
        text: prompt,
      }
    ];

    // Ajouter chaque screenshot comme image
    screenshots.forEach((screenshot, index) => {
      // Extraire uniquement la partie base64 (sans le prefix data:image/png;base64,)
      const base64Data = screenshot.screenshot_base64.split(',')[1] || screenshot.screenshot_base64;

      parts.push({
        inline_data: {
          mime_type: "image/png",
          data: base64Data
        }
      });

      // Ajouter une description textuelle de l'image pour contexte
      const annotationInfo = Object.entries(screenshot.annotation_types)
        .filter(([_, count]) => count > 0)
        .map(([type, count]) => `${count} ${type}`)
        .join(', ');

      parts.push({
        text: `[Screenshot ${index + 1}/${screenshots.length}] - URL: ${screenshot.url} - Annotations: ${annotationInfo}${screenshot.has_frustration ? ' - CONTIENT DES PAIN POINTS (post-its rouges)' : ''}`
      });
    });

    const requestBody = {
      contents: [
        {
          parts: parts,
        },
      ],
      generationConfig: {
        temperature: 0.7,
        topK: 40,
        topP: 0.95,
        maxOutputTokens: 2048,
      },
    };

    const response = await fetch(url, {
      method: "POST",
      headers: {
        "Content-Type": "application/json",
        "X-goog-api-key": this.apiKey,
      },
      body: JSON.stringify(requestBody),
    });

    if (!response.ok) {
      const errorData = await response.json().catch(() => ({}));
      throw new Error(
        `Gemini Vision API error: ${response.status} - ${errorData.error?.message || "Unknown error"}`,
      );
    }

    const data = await response.json();

    if (!data.candidates || data.candidates.length === 0) {
      throw new Error("No response from Gemini Vision API");
    }

    const textResponse = data.candidates[0].content.parts[0].text;
    console.log("[AIAnalyzer] Received response from Gemini Vision");

    return textResponse;
  }

  /**
   * Parse la réponse de Gemini
   */
  parseGeminiResponse(responseText, anonymizedData) {
    try {
      // Extraire le JSON de la réponse (au cas où il y a du texte avant/après)
      const jsonMatch = responseText.match(/\{[\s\S]*\}/);
      if (!jsonMatch) {
        throw new Error("No JSON found in response");
      }

      const parsed = JSON.parse(jsonMatch[0]);

      // Valider la structure
      if (!parsed.summary || !parsed.recommendations) {
        throw new Error("Invalid response structure");
      }

      // Enrichir avec données de la session
      parsed.session_stats = {
        duration: anonymizedData.duration,
        total_events: anonymizedData.events_summary.total,
        total_patterns: anonymizedData.patterns.total,
        device_type: anonymizedData.metadata.device_type,
      };

      return parsed;
    } catch (error) {
      console.error("[AIAnalyzer] Failed to parse Gemini response:", error);
      console.error("[AIAnalyzer] Raw response:", responseText);

      // Fallback : analyse de base sans IA
      return this.generateFallbackAnalysis(anonymizedData);
    }
  }

  /**
   * Génère une analyse de fallback si l'IA échoue
   */
  generateFallbackAnalysis(anonymizedData) {
    const patterns = anonymizedData.patterns;
    const stats = anonymizedData.events_summary;

    const hasIssues = patterns.total > 0;
    const uxScore = hasIssues ? Math.max(3, 10 - patterns.total) : 8;

    return {
      summary: `Session de ${this.formatDuration(anonymizedData.duration)} avec ${stats.total} événements enregistrés. ${patterns.total} patterns UX problématiques détectés.`,
      ux_score: uxScore,
      ux_score_explanation: hasIssues
        ? "Score impacté par les patterns de friction détectés"
        : "Parcours relativement fluide",
      friction_points: this.generateFallbackFrictionPoints(patterns),
      positive_points: [
        {
          title: "Session complète",
          description:
            "L'utilisateur a complété sa session sans abandon brutal",
        },
      ],
      recommendations: this.generateFallbackRecommendations(patterns),
      priority:
        patterns.total > 5 ? "high" : patterns.total > 2 ? "medium" : "low",
      key_insights: [
        `${stats.total} événements analysés`,
        `${patterns.total} patterns UX détectés`,
        `Durée de session: ${this.formatDuration(anonymizedData.duration)}`,
      ],
      session_stats: {
        duration: anonymizedData.duration,
        total_events: stats.total,
        total_patterns: patterns.total,
        device_type: anonymizedData.metadata.device_type,
      },
      is_fallback: true,
    };
  }

  /**
   * Génère les points de friction pour le fallback
   */
  generateFallbackFrictionPoints(patterns) {
    const frictionPoints = [];

    if (patterns.rage_clicks.length > 0) {
      frictionPoints.push({
        title: "Rage clicks détectés",
        description: `${patterns.rage_clicks.length} rage click(s) indiquant une frustration utilisateur`,
        severity: "high",
        location: "Éléments cliqués de manière répétée",
      });
    }

    if (patterns.dead_clicks.length > 0) {
      frictionPoints.push({
        title: "Clics non-interactifs",
        description: `${patterns.dead_clicks.length} clic(s) sur des éléments non-interactifs`,
        severity: "medium",
        location: "Éléments visuels sans interactivité",
      });
    }

    if (patterns.confusion_zones.length > 0) {
      frictionPoints.push({
        title: "Zones de confusion",
        description: `${patterns.confusion_zones.length} zone(s) où l'utilisateur hésite`,
        severity: "medium",
        location: "Zones de hover prolongé",
      });
    }

    if (patterns.form_abandonments.length > 0) {
      frictionPoints.push({
        title: "Abandons de formulaire",
        description: `${patterns.form_abandonments.length} champ(s) rempli(s) puis vidé(s)`,
        severity: "high",
        location: "Champs de formulaire",
      });
    }

    return frictionPoints;
  }

  /**
   * Génère les recommandations pour le fallback
   */
  generateFallbackRecommendations(patterns) {
    const recommendations = [];

    if (patterns.rage_clicks.length > 0) {
      recommendations.push({
        title: "Améliorer la réactivité des éléments",
        description:
          "Vérifier que tous les éléments cliquables répondent instantanément et fournissent un feedback visuel",
        impact: "high",
        effort: "medium",
        category: "technical",
      });
    }

    if (patterns.dead_clicks.length > 0) {
      recommendations.push({
        title: "Clarifier les éléments interactifs",
        description:
          "Rendre évident quels éléments sont cliquables (curseur, style, affordance)",
        impact: "high",
        effort: "low",
        category: "design",
      });
    }

    if (patterns.confusion_zones.length > 0) {
      recommendations.push({
        title: "Améliorer la navigation",
        description:
          "Simplifier la structure de navigation et ajouter des indicateurs visuels",
        impact: "medium",
        effort: "medium",
        category: "navigation",
      });
    }

    if (patterns.form_abandonments.length > 0) {
      recommendations.push({
        title: "Optimiser les formulaires",
        description:
          "Réduire le nombre de champs, améliorer les labels et la validation",
        impact: "high",
        effort: "medium",
        category: "design",
      });
    }

    if (recommendations.length === 0) {
      recommendations.push({
        title: "Continuer le monitoring",
        description:
          "Continuer à surveiller les sessions pour détecter d'éventuels problèmes",
        impact: "low",
        effort: "low",
        category: "technical",
      });
    }

    return recommendations;
  }

  /**
   * Formate une durée
   */
  formatDuration(ms) {
    const seconds = Math.floor(ms / 1000);
    const minutes = Math.floor(seconds / 60);
    const hours = Math.floor(minutes / 60);

    if (hours > 0) {
      return `${hours}h ${minutes % 60}min`;
    } else if (minutes > 0) {
      return `${minutes}min ${seconds % 60}s`;
    } else {
      return `${seconds}s`;
    }
  }

  /**
   * Teste la connexion à l'API
   */
  async testConnection() {
    if (!this.checkConfiguration()) {
      throw new Error("API key not configured");
    }

    try {
      const testPrompt =
        'Réponds simplement avec "OK" si tu reçois ce message.';
      const response = await this.callGeminiAPI(testPrompt);

      return {
        success: true,
        message: "Connection successful",
        response: response.substring(0, 100),
      };
    } catch (error) {
      return {
        success: false,
        message: error.message,
      };
    }
  }

  /**
   * Obtient les stats du cache
   */
  getCacheStats() {
    return {
      cached_analyses: this.analysisCache.size,
      last_analysis_time: this.lastAnalysis?.metadata?.analyzed_at || null,
    };
  }

  /**
   * Vide le cache
   */
  clearCache() {
    this.analysisCache.clear();
    console.log("[AIAnalyzer] Cache cleared");
  }
}

// Export singleton
const aiAnalyzer = new AIAnalyzer();
export default aiAnalyzer;
