/**
 * SUMO UX Tracker - Export Manager
 * Gestion des exports de sessions en format ZIP
 */

class ExportManager {
  constructor() {
    this.jsZipLoaded = false;
  }

  /**
   * Export une session au format ZIP
   */
  async exportSession(sessionData, format = 'standard') {
    try {
      console.log(`[ExportManager] Starting export for session ${sessionData.id} in ${format} format`);
      
      // VÃ©rifier que la session existe et contient des donnÃ©es
      if (!sessionData) {
        throw new Error('Session non fournie');
      }
      
      if (!sessionData.id) {
        throw new Error('Session invalide (pas d\'ID)');
      }

      // Charger JSZip si nÃ©cessaire
      if (!window.JSZip) {
        console.log('[ExportManager] JSZip not loaded, loading now...');
        await this.loadJSZip();
      }

      // PrÃ©parer les donnÃ©es selon le format
      let exportData;
      switch (format) {
        case 'light':
          exportData = this.prepareUltraLightExport(sessionData);
          break;
        case 'detailed':
          exportData = this.prepareDetailedExport(sessionData);
          break;
        case 'executive':
          exportData = this.prepareExecutiveExport(sessionData);
          break;
        default:
          exportData = this.prepareStandardExport(sessionData);
      }

      // CrÃ©er l'archive ZIP
      const zipBlob = await this.createZipArchive(exportData, sessionData, format);
      
      console.log('[ExportManager] Export completed successfully');
      return zipBlob;
      
    } catch (error) {
      console.error('[ExportManager] Export failed:', error);
      throw error;
    }
  }

  /**
   * PrÃ©pare l'export standard (5-15MB)
   */
  prepareStandardExport(session) {
    return {
      format: 'standard',
      version: '2.1.0',
      exportDate: new Date().toISOString(),
      session: {
        id: session.id,
        startTime: session.startTime,
        endTime: session.endTime,
        duration: session.duration || (session.endTime - session.startTime),
        initialUrl: session.initialUrl,
        initialTitle: session.initialTitle,
        stats: session.stats
      },
      pages: session.pages || [],
      events: this.filterEvents(session.events || [], 'standard'),
      screenshots: this.limitScreenshots(session.screenshots || [], 10),
      insights: this.generateInsights(session)
    };
  }

  /**
   * PrÃ©pare l'export ultra-lÃ©ger (<1MB)
   */
  prepareUltraLightExport(session) {
    return {
      format: 'light',
      version: '2.1.0',
      exportDate: new Date().toISOString(),
      session: {
        id: session.id,
        startTime: session.startTime,
        endTime: session.endTime,
        duration: session.duration || (session.endTime - session.startTime),
        initialUrl: session.initialUrl,
        stats: session.stats
      },
      pages: session.pages || [],
      events: this.filterEvents(session.events || [], 'light'),
      // Pas de screenshots pour la version lÃ©gÃ¨re
      screenshots: [],
      summary: this.generateSummary(session)
    };
  }

  /**
   * PrÃ©pare l'export dÃ©taillÃ© (20-50MB)
   */
  prepareDetailedExport(session) {
    return {
      format: 'detailed',
      version: '2.1.0',
      exportDate: new Date().toISOString(),
      session: {
        id: session.id,
        startTime: session.startTime,
        endTime: session.endTime,
        duration: session.duration || (session.endTime - session.startTime),
        initialUrl: session.initialUrl,
        initialTitle: session.initialTitle,
        tabId: session.tabId,
        stats: session.stats
      },
      pages: session.pages || [],
      events: session.events || [], // Tous les Ã©vÃ©nements
      screenshots: session.screenshots || [], // Tous les screenshots
      insights: this.generateInsights(session),
      debug: {
        eventCount: session.events?.length || 0,
        screenshotCount: session.screenshots?.length || 0,
        totalSize: this.estimateDataSize(session)
      }
    };
  }

  /**
   * PrÃ©pare l'export exÃ©cutif (3-8MB)
   */
  prepareExecutiveExport(session) {
    const insights = this.generateInsights(session);
    const recommendations = this.generateRecommendations(session);
    
    return {
      format: 'executive',
      version: '2.1.0',
      exportDate: new Date().toISOString(),
      summary: {
        sessionId: session.id,
        duration: this.formatDuration(session.duration || (session.endTime - session.startTime)),
        pagesVisited: session.pages?.length || 0,
        totalInteractions: session.stats?.actionCount || 0,
        url: session.initialUrl
      },
      insights: insights,
      recommendations: recommendations,
      keyMetrics: {
        engagementRate: this.calculateEngagementRate(session),
        averageTimePerPage: this.calculateAverageTimePerPage(session),
        interactionHeatmap: this.generateInteractionSummary(session),
        userJourney: this.summarizeUserJourney(session)
      },
      screenshots: this.limitScreenshots(session.screenshots || [], 5) // Seulement 5 screenshots clÃ©s
    };
  }

  /**
   * Filtre les Ã©vÃ©nements selon le format
   */
  filterEvents(events, format) {
    if (format === 'light') {
      // Garder seulement les Ã©vÃ©nements importants
      return events.filter(e => 
        ['page_start', 'page_end', 'rage_click', 'form_submit'].includes(e.type)
      );
    } else if (format === 'standard') {
      // Exclure les Ã©vÃ©nements de mouvement de souris trop frÃ©quents
      return events.filter(e => e.type !== 'mouse_move');
    }
    return events;
  }

  /**
   * Limite le nombre de screenshots
   */
  limitScreenshots(screenshots, maxCount) {
    if (screenshots.length <= maxCount) {
      return screenshots;
    }
    
    // Prendre le premier, le dernier, et des screenshots Ã©quidistants
    const result = [];
    const step = Math.floor(screenshots.length / maxCount);
    
    for (let i = 0; i < maxCount && i * step < screenshots.length; i++) {
      result.push(screenshots[i * step]);
    }
    
    return result;
  }

  /**
   * GÃ©nÃ¨re les insights
   */
  generateInsights(session) {
    const insights = [];
    
    // Analyse des rage clicks
    const rageClicks = (session.events || []).filter(e => e.type === 'rage_click');
    if (rageClicks.length > 0) {
      insights.push({
        type: 'frustration',
        severity: 'high',
        message: `${rageClicks.length} rage clicks dÃ©tectÃ©s`,
        details: rageClicks.map(r => ({
          timestamp: r.timestamp,
          element: r.target
        }))
      });
    }
    
    // Analyse du temps de session
    const duration = session.duration || (session.endTime - session.startTime);
    if (duration < 30000) { // Moins de 30 secondes
      insights.push({
        type: 'engagement',
        severity: 'warning',
        message: 'Session trÃ¨s courte (< 30s)'
      });
    }
    
    // Analyse des pages visitÃ©es
    if (session.pages && session.pages.length > 5) {
      insights.push({
        type: 'navigation',
        severity: 'info',
        message: `Navigation active : ${session.pages.length} pages visitÃ©es`
      });
    }
    
    return insights;
  }

  /**
   * GÃ©nÃ¨re les recommandations
   */
  generateRecommendations(session) {
    const recommendations = [];
    
    const rageClicks = (session.events || []).filter(e => e.type === 'rage_click');
    if (rageClicks.length > 3) {
      recommendations.push({
        priority: 'high',
        category: 'UX',
        suggestion: 'Revoir les Ã©lÃ©ments interactifs causant de la frustration',
        evidence: `${rageClicks.length} rage clicks dÃ©tectÃ©s`
      });
    }
    
    return recommendations;
  }

  /**
   * GÃ©nÃ¨re un rÃ©sumÃ©
   */
  generateSummary(session) {
    return {
      totalDuration: this.formatDuration(session.duration || 0),
      pagesVisited: session.pages?.length || 0,
      totalClicks: (session.events || []).filter(e => e.type === 'click').length,
      screenshotsCaptured: session.screenshots?.length || 0
    };
  }

  /**
   * Calcule le taux d'engagement
   */
  calculateEngagementRate(session) {
    const duration = (session.duration || (session.endTime - session.startTime)) / 1000; // en secondes
    const interactions = session.stats?.actionCount || 0;
    
    if (duration === 0) return 0;
    
    const rate = (interactions / duration) * 60; // interactions par minute
    return Math.round(rate * 100) / 100;
  }

  /**
   * Calcule le temps moyen par page
   */
  calculateAverageTimePerPage(session) {
    const pages = session.pages?.length || 1;
    const duration = session.duration || (session.endTime - session.startTime);
    return this.formatDuration(Math.floor(duration / pages));
  }

  /**
   * GÃ©nÃ¨re un rÃ©sumÃ© des interactions
   */
  generateInteractionSummary(session) {
    const events = session.events || [];
    const summary = {};
    
    events.forEach(event => {
      summary[event.type] = (summary[event.type] || 0) + 1;
    });
    
    return summary;
  }

  /**
   * RÃ©sume le parcours utilisateur
   */
  summarizeUserJourney(session) {
    return (session.pages || []).map((page, index) => ({
      step: index + 1,
      url: page.url,
      timestamp: page.timestamp,
      duration: index < session.pages.length - 1 
        ? session.pages[index + 1].timestamp - page.timestamp
        : session.endTime - page.timestamp
    }));
  }

  /**
   * Estime la taille des donnÃ©es
   */
  estimateDataSize(session) {
    let size = JSON.stringify(session).length;
    return Math.round(size / 1024); // en KB
  }

  /**
   * CrÃ©e l'archive ZIP
   */
  async createZipArchive(exportData, session, format) {
    try {
      // VÃ©rifier que JSZip est disponible
      if (!window.JSZip) {
        throw new Error('JSZip not loaded');
      }

      const zip = new window.JSZip();
      
      // Dossier principal
      const folderName = `sumo-export-${session.id}-${Date.now()}`;
      const folder = zip.folder(folderName);
      
      // Fichier de donnÃ©es principal
      folder.file('session-data.json', JSON.stringify(exportData, null, 2));
      
      // Fichier README
      folder.file('README.txt', this.generateReadme(exportData, session));
      
      // Screenshots sÃ©parÃ©s (si prÃ©sents et format non light)
      if (session.screenshots && session.screenshots.length > 0 && format !== 'light') {
        console.log(`[ExportManager] Processing ${session.screenshots.length} screenshots`);
        const screenshotsFolder = folder.folder('screenshots');
        
        // DÃ©terminer quels screenshots inclure selon le format
        let screenshotsToExport = session.screenshots;
        if (format === 'standard') {
          screenshotsToExport = this.limitScreenshots(session.screenshots, 10);
        } else if (format === 'executive') {
          screenshotsToExport = this.limitScreenshots(session.screenshots, 5);
        }
        
        // Traiter chaque screenshot
        for (let i = 0; i < screenshotsToExport.length; i++) {
          const screenshot = screenshotsToExport[i];
          
          if (screenshot && screenshot.dataUrl) {
            try {
              // Extraire les donnÃ©es base64 du dataUrl
              const base64Data = screenshot.dataUrl.split(',')[1];
              
              if (base64Data) {
                const fileName = `screenshot-${i+1}-${screenshot.timestamp || Date.now()}.jpg`;
                screenshotsFolder.file(fileName, base64Data, { base64: true });
                console.log(`[ExportManager] Added screenshot ${i+1}/${screenshotsToExport.length}`);
              }
            } catch (e) {
              console.warn(`[ExportManager] Could not process screenshot ${i}:`, e);
            }
          } else {
            console.warn(`[ExportManager] Screenshot ${i} missing dataUrl`);
          }
        }
      } else {
        console.log('[ExportManager] No screenshots to export or light format selected');
      }
      
      // GÃ©nÃ©ration du ZIP
      console.log('[ExportManager] Generating ZIP...');
      const zipBlob = await zip.generateAsync({ 
        type: 'blob',
        compression: 'DEFLATE',
        compressionOptions: { level: 6 }
      });
      
      console.log('[ExportManager] ZIP generated successfully, size:', zipBlob.size);
      return zipBlob;
      
    } catch (error) {
      console.error('[ExportManager] ZIP creation failed:', error);
      throw new Error(`Ã‰chec crÃ©ation ZIP: ${error.message}`);
    }
  }

  /**
   * GÃ©nÃ¨re le fichier README
   */
  generateReadme(exportData, session) {
    const now = new Date();
    const dateStr = now.toLocaleDateString('fr-FR');
    const timeStr = now.toLocaleTimeString('fr-FR');
    
    return `SUMO UX Tracker - Export de Session
=====================================

Date d'export: ${dateStr} ${timeStr}
Version: ${exportData.version}
Format: ${exportData.format}

Session ID: ${session.id}
DurÃ©e: ${this.formatDuration(session.duration || 0)}
URL initiale: ${session.initialUrl}

Contenu de l'archive:
- session-data.json : DonnÃ©es complÃ¨tes de la session
- screenshots/ : Captures d'Ã©cran (si disponibles)
- README.txt : Ce fichier

Pour analyser ces donnÃ©es, utilisez l'application SUMO UX Tracker
ou importez le fichier JSON dans votre outil d'analyse.

---
SUMO UX Tracker v2.1.0`;
  }

  /**
   * Formate une durÃ©e
   */
  formatDuration(milliseconds) {
    const seconds = Math.floor(milliseconds / 1000);
    const minutes = Math.floor(seconds / 60);
    const hours = Math.floor(minutes / 60);
    
    if (hours > 0) {
      return `${hours}h ${minutes % 60}m ${seconds % 60}s`;
    } else if (minutes > 0) {
      return `${minutes}m ${seconds % 60}s`;
    }
    return `${seconds}s`;
  }

  /**
   * Charge JSZip
   */
  async loadJSZip() {
    return new Promise((resolve, reject) => {
      // VÃ©rifier si dÃ©jÃ  chargÃ©
      if (window.JSZip) {
        console.log('[ExportManager] JSZip already loaded');
        resolve(window.JSZip);
        return;
      }

      // Si pas chargÃ©, essayer de le charger localement
      console.log('[ExportManager] Loading JSZip locally...');
      
      // Pour l'extension, JSZip devrait dÃ©jÃ  Ãªtre chargÃ© via popup.html
      setTimeout(() => {
        if (window.JSZip) {
          console.log('[ExportManager] JSZip loaded successfully');
          resolve(window.JSZip);
        } else {
          console.error('[ExportManager] JSZip not found');
          reject(new Error('JSZip non trouvÃ©. Assurez-vous que jszip.min.js est inclus.'));
        }
      }, 100);
    });
  }
}