/**
 * SUMO UX Tracker - Debug Logger
 * Système de logging persistant pour diagnostiquer les crashes
 */

class DebugLogger {
  constructor() {
    this.logs = [];
    this.maxLogs = 500; // Garder les 500 derniers logs
    this.sessionId = null;
    this.enabled = true;

    // Charger les logs existants
    this.loadLogs();
  }

  async loadLogs() {
    try {
      const result = await chrome.storage.local.get(['debugLogs']);
      if (result.debugLogs) {
        this.logs = result.debugLogs;
      }
    } catch (error) {
      console.error('[DebugLogger] Error loading logs:', error);
    }
  }

  async saveLogs() {
    try {
      await chrome.storage.local.set({ debugLogs: this.logs });
    } catch (error) {
      console.error('[DebugLogger] Error saving logs:', error);
    }
  }

  setSessionId(sessionId) {
    this.sessionId = sessionId;
    this.log('SESSION', `Session ID set: ${sessionId}`);
  }

  log(category, message, data = null) {
    if (!this.enabled) return;

    const timestamp = Date.now();
    const logEntry = {
      timestamp,
      time: new Date(timestamp).toISOString(),
      category,
      message,
      sessionId: this.sessionId,
      data: data ? JSON.stringify(data) : null
    };

    // Ajouter au tableau
    this.logs.push(logEntry);

    // Limiter la taille
    if (this.logs.length > this.maxLogs) {
      this.logs = this.logs.slice(-this.maxLogs);
    }

    // Sauvegarder de manière asynchrone
    this.saveLogs();

    // Afficher aussi dans la console
    const emoji = this.getCategoryEmoji(category);
    console.log(`${emoji} [${category}] ${message}`, data || '');
  }

  getCategoryEmoji(category) {
    const emojis = {
      'SESSION': '🎬',
      'START': '▶️',
      'STOP': '⏹️',
      'PAUSE': '⏸️',
      'RESUME': '▶️',
      'EVENT': '📝',
      'SCREENSHOT': '📸',
      'FEEDBACK': '💬',
      'NAVIGATION': '🧭',
      'ERROR': '❌',
      'WARNING': '⚠️',
      'CRASH': '💥',
      'STORAGE': '💾',
      'BACKGROUND': '🔧'
    };
    return emojis[category] || '📋';
  }

  async getLogs(filter = null) {
    await this.loadLogs();

    if (!filter) {
      return this.logs;
    }

    return this.logs.filter(log => {
      if (filter.category && log.category !== filter.category) return false;
      if (filter.sessionId && log.sessionId !== filter.sessionId) return false;
      if (filter.since && log.timestamp < filter.since) return false;
      return true;
    });
  }

  async exportLogs() {
    await this.loadLogs();

    const blob = new Blob([JSON.stringify(this.logs, null, 2)], { type: 'application/json' });
    const url = URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.href = url;
    a.download = `sumo-debug-logs-${Date.now()}.json`;
    a.click();
    URL.revokeObjectURL(url);
  }

  async clearLogs() {
    this.logs = [];
    await chrome.storage.local.remove(['debugLogs']);
    console.log('[DebugLogger] Logs cleared');
  }

  async printLogs(filter = null) {
    const logs = await this.getLogs(filter);

    console.group('📋 SUMO Debug Logs');
    logs.forEach(log => {
      const emoji = this.getCategoryEmoji(log.category);
      console.log(
        `${emoji} [${log.time}] [${log.category}] ${log.message}`,
        log.data ? JSON.parse(log.data) : ''
      );
    });
    console.groupEnd();
  }
}

// Instance globale
const debugLogger = new DebugLogger();

// Exposer globalement
if (typeof window !== 'undefined') {
  window.SUMO_DEBUG = {
    log: (category, message, data) => debugLogger.log(category, message, data),
    getLogs: (filter) => debugLogger.getLogs(filter),
    printLogs: (filter) => debugLogger.printLogs(filter),
    exportLogs: () => debugLogger.exportLogs(),
    clearLogs: () => debugLogger.clearLogs(),
    setSessionId: (id) => debugLogger.setSessionId(id)
  };
}

// Export pour modules
if (typeof module !== 'undefined' && module.exports) {
  module.exports = debugLogger;
}