/**
 * SUMO UX Tracker V3 - Popup Script
 */

// État
let isTracking = false;
let sessionData = null;
let updateInterval = null;

// Elements DOM
const startBtn = document.getElementById('startBtn');
const stopBtn = document.getElementById('stopBtn');
const feedbackBtn = document.getElementById('feedbackBtn');
const exportBtn = document.getElementById('exportBtn');
const status = document.getElementById('status');
const statusText = document.getElementById('statusText');
const sessionInfo = document.getElementById('sessionInfo');
const sessionId = document.getElementById('sessionId');
const duration = document.getElementById('duration');
const eventCount = document.getElementById('eventCount');
const screenshotCount = document.getElementById('screenshotCount');

// Init
document.addEventListener('DOMContentLoaded', () => {
  checkStatus();
  setupListeners();
});

function setupListeners() {
  startBtn.addEventListener('click', startTracking);
  stopBtn.addEventListener('click', stopTracking);
  feedbackBtn.addEventListener('click', enableFeedbackMode);
  exportBtn.addEventListener('click', exportSession);
}

async function checkStatus(retryCount = 0) {
  try {
    const response = await chrome.runtime.sendMessage({
      action: 'GET_SESSION_STATUS'
    });

    if (response?.success && response.isActive) {
      isTracking = true;
      sessionData = response.session;
      updateUI();
      startUpdates();
    } else {
      updateUI();
    }
  } catch (error) {
    // Service worker pas encore prêt, retry avec délai
    if (retryCount < 3) {
      console.log(`Background not ready, retrying... (${retryCount + 1}/3)`);
      setTimeout(() => checkStatus(retryCount + 1), 200);
    } else {
      console.error('Error checking status after retries:', error);
      updateUI(); // Afficher UI par défaut
    }
  }
}

async function startTracking() {
  try {
    // Désactiver bouton pendant le démarrage
    startBtn.disabled = true;
    statusText.textContent = 'Starting...';
    
    // Récupérer tab active
    const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
    
    if (!tab) {
      alert('No active tab found');
      startBtn.disabled = false;
      statusText.textContent = 'Ready to track';
      return;
    }
    
    // Vérifier que ce n'est pas une page chrome://
    if (tab.url.startsWith('chrome://') || tab.url.startsWith('chrome-extension://')) {
      alert('Cannot track on Chrome internal pages. Please open a regular website.');
      startBtn.disabled = false;
      statusText.textContent = 'Ready to track';
      return;
    }
    
    statusText.textContent = 'Connecting...';
    
    // Récupérer le mode de capture sélectionné
    const captureMode = document.querySelector('input[name="captureMode"]:checked')?.value || 'viewport';
    console.log('[Popup] Capture mode:', captureMode);

    // Démarrer session
    const response = await chrome.runtime.sendMessage({
      action: 'START_SESSION',
      tabId: tab.id,
      captureMode: captureMode
    });
    
    if (response?.success) {
      isTracking = true;
      sessionData = response.session;
      updateUI();
      startUpdates();
      
      console.log('Tracking started:', sessionData.sessionId);
    } else {
      const errorMsg = response?.error || 'Failed to start tracking';
      alert(errorMsg);
      startBtn.disabled = false;
      statusText.textContent = 'Ready to track';
    }
    
  } catch (error) {
    console.error('Error starting tracking:', error);
    
    // Message d'erreur plus clair
    let errorMsg = error.message;
    if (errorMsg.includes('Content script not ready')) {
      errorMsg = 'Please refresh the page (F5) and try again.';
    }
    
    alert('Error: ' + errorMsg);
    startBtn.disabled = false;
    statusText.textContent = 'Ready to track';
  }
}

async function stopTracking() {
  try {
    const response = await chrome.runtime.sendMessage({
      action: 'STOP_SESSION'
    });
    
    if (response?.success) {
      isTracking = false;
      stopUpdates();
      updateUI();
      
      // Activer bouton export
      exportBtn.disabled = false;
      
      console.log('Tracking stopped');
    }
    
  } catch (error) {
    console.error('Error stopping tracking:', error);
    alert('Error: ' + error.message);
  }
}

async function exportSession() {
  try {
    statusText.textContent = 'Creating ZIP export...';

    const response = await chrome.runtime.sendMessage({
      action: 'EXPORT_SESSION'
    });

    if (!response?.success) {
      throw new Error(response?.error || 'Export failed');
    }

    console.log('[Popup] Export response:', response);

    // Le téléchargement est géré par le background via chrome.downloads API
    statusText.textContent = `Export complete! ${response.filename} (${response.stats.totalEvents} events, ${response.stats.totalScreenshots} screenshots)`;

    setTimeout(() => {
      statusText.textContent = 'Ready to track';
      exportBtn.disabled = true;
      sessionData = null;
      sessionInfo.style.display = 'none';
    }, 3000);

  } catch (error) {
    console.error('Error exporting:', error);
    alert('Export error: ' + error.message);
    statusText.textContent = 'Export failed';
  }
}

function downloadFile(blob, filename) {
  const url = URL.createObjectURL(blob);
  const a = document.createElement('a');
  a.href = url;
  a.download = filename;
  a.click();
  URL.revokeObjectURL(url);
}

function downloadDataUrl(dataUrl, filename) {
  const a = document.createElement('a');
  a.href = dataUrl;
  a.download = filename;
  a.click();
}

function sleep(ms) {
  return new Promise(resolve => setTimeout(resolve, ms));
}

function startUpdates() {
  // Update toutes les secondes
  updateInterval = setInterval(() => {
    updateSessionInfo();
  }, 1000);
  
  updateSessionInfo();
}

function stopUpdates() {
  if (updateInterval) {
    clearInterval(updateInterval);
    updateInterval = null;
  }
}

async function updateSessionInfo() {
  if (!sessionData) return;
  
  try {
    const response = await chrome.runtime.sendMessage({
      action: 'GET_SESSION_STATUS'
    });
    
    if (response?.success && response.session) {
      const session = response.session;
      
      // Update infos
      sessionId.textContent = session.sessionId.substring(0, 20) + '...';
      
      const durationMs = Date.now() - session.startTime;
      const minutes = Math.floor(durationMs / 60000);
      const seconds = Math.floor((durationMs % 60000) / 1000);
      duration.textContent = `${String(minutes).padStart(2, '0')}:${String(seconds).padStart(2, '0')}`;
      
      eventCount.textContent = session.events?.length || 0;
      screenshotCount.textContent = session.screenshots?.length || 0;
    }
  } catch (error) {
    // Ignorer erreurs pendant update
  }
}

function updateUI() {
  if (isTracking) {
    // Mode tracking
    status.className = 'status tracking';
    statusText.textContent = 'Recording...';

    startBtn.disabled = true;
    stopBtn.disabled = false;
    feedbackBtn.disabled = false; // Activer bouton Feedback
    exportBtn.disabled = true;

    sessionInfo.style.display = 'block';

    // Désactiver le toggle et afficher le mode actuel
    const captureModeDiv = document.getElementById('captureMode');
    if (captureModeDiv) {
      captureModeDiv.style.opacity = '0.6';
      captureModeDiv.style.pointerEvents = 'none';
    }

    // Mettre à jour le toggle pour refléter le mode de la session
    if (sessionData?.captureMode) {
      const radio = document.querySelector(`input[name="captureMode"][value="${sessionData.captureMode}"]`);
      if (radio) radio.checked = true;
    }

  } else {
    // Mode idle
    status.className = 'status idle';
    statusText.textContent = sessionData ? 'Session stopped' : 'Ready to track';

    startBtn.disabled = false;
    stopBtn.disabled = true;
    feedbackBtn.disabled = true; // Désactiver Feedback quand pas de tracking
    exportBtn.disabled = !sessionData;

    // Réactiver le toggle
    const captureModeDiv = document.getElementById('captureMode');
    if (captureModeDiv) {
      captureModeDiv.style.opacity = '1';
      captureModeDiv.style.pointerEvents = 'auto';
    }

    if (!sessionData) {
      sessionInfo.style.display = 'none';
    }
  }
}

/**
 * Active le mode Feedback
 */
async function enableFeedbackMode() {
  try {
    feedbackBtn.disabled = true;
    feedbackBtn.textContent = '⏳ Loading...';

    console.log('[Popup] Enabling Feedback Mode...');

    // Récupérer tab active
    const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });

    if (!tab) {
      alert('No active tab found');
      feedbackBtn.disabled = false;
      feedbackBtn.textContent = '📝 Feedback Mode';
      return;
    }

    console.log('[Popup] Sending message to tab:', tab.id);

    // Envoyer message au content script et ATTENDRE la réponse
    const response = await chrome.tabs.sendMessage(tab.id, {
      action: 'enableFeedbackMode'
    });

    console.log('[Popup] Response from content script:', response);

    if (!response || !response.success) {
      throw new Error(response?.error || 'Failed to activate feedback mode');
    }

    console.log('[Popup] Feedback mode activated successfully');

    // Réactiver le bouton
    feedbackBtn.disabled = false;
    feedbackBtn.textContent = '📝 Feedback Mode';

    // Attendre un peu pour s'assurer que tout est chargé avant de fermer
    await new Promise(resolve => setTimeout(resolve, 300));

    // Fermer le popup
    window.close();

  } catch (error) {
    console.error('[Popup] Error enabling feedback mode:', error);
    alert('Error activating feedback mode: ' + error.message + '\n\nPlease check the console for details.');
    feedbackBtn.disabled = false;
    feedbackBtn.textContent = '📝 Feedback Mode';
  }
}
