/**
 * SUMO UX Tracker V3 - Feedback Overlay
 * Système d'annotation visuelle en surimppression de la page
 */

const FeedbackOverlay = {
  isActive: false,
  overlay: null,
  canvas: null,
  ctx: null,
  toolbar: null,
  pageBlocker: null,
  wasTrackingBeforeFeedback: false, // Pour savoir si on doit reprendre le tracking

  // État des annotations
  annotations: [],
  postitElements: [], // Stocker les éléments DOM des post-it
  currentTool: 'draw',
  currentColor: '#FF0000',
  isDrawing: false,
  tempAnnotation: null,
  draggedPostit: null,

  /**
   * Active le mode feedback
   */
  activate() {
    if (this.isActive) return;

    console.log('🎨 Activation du mode Feedback');

    // IMPORTANT: Mettre le tracking en pause pour ne pas polluer la session
    if (window.sumoContent && window.sumoContent.isTracking) {
      console.log('🎨 Mise en pause du tracking pendant le feedback');
      this.wasTrackingBeforeFeedback = true;
      window.sumoContent.pauseTracking();
    }

    // Créer l'overlay
    this.createOverlay();

    // Geler la page
    this.freezePage();

    // Créer la toolbar
    this.createToolbar();

    // Initialiser les événements
    this.initEvents();

    this.isActive = true;

    // Notifier le background
    chrome.runtime.sendMessage({
      action: 'feedbackModeActivated',
      timestamp: Date.now()
    });
  },

  /**
   * Désactive le mode feedback
   */
  deactivate() {
    if (!this.isActive) return;

    console.log('🎨 Désactivation du mode Feedback');

    // Retirer les post-it DOM
    this.postitElements.forEach(({element}) => {
      if (element && element.parentNode) {
        element.remove();
      }
    });

    // Retirer les éléments
    if (this.overlay) this.overlay.remove();
    if (this.toolbar) this.toolbar.remove();
    if (this.pageBlocker) this.pageBlocker.remove();

    // IMPORTANT: Reprendre le tracking si il était actif avant
    if (this.wasTrackingBeforeFeedback && window.sumoContent) {
      console.log('🎨 Reprise du tracking après le feedback');
      window.sumoContent.resumeTracking();
      this.wasTrackingBeforeFeedback = false;
    }

    // Réinitialiser les propriétés
    this.overlay = null;
    this.canvas = null;
    this.ctx = null;
    this.toolbar = null;
    this.pageBlocker = null;
    this.annotations = [];
    this.postitElements = [];
    this.tempAnnotation = null;
    this.draggedPostit = null;

    this.isActive = false;
  },

  /**
   * Crée l'overlay canvas transparent
   */
  createOverlay() {
    // Conteneur principal
    this.overlay = document.createElement('div');
    this.overlay.id = 'ux-tracker-feedback-overlay';
    this.overlay.style.cssText = `
      position: fixed !important;
      top: 0 !important;
      left: 0 !important;
      width: 100vw !important;
      height: 100vh !important;
      z-index: 2147483647 !important;
      background: transparent !important;
      pointer-events: auto !important;
    `;

    // Canvas de dessin
    this.canvas = document.createElement('canvas');
    this.canvas.width = window.innerWidth * window.devicePixelRatio;
    this.canvas.height = window.innerHeight * window.devicePixelRatio;
    this.canvas.style.cssText = `
      width: 100% !important;
      height: 100% !important;
      display: block !important;
    `;

    // Context 2D
    this.ctx = this.canvas.getContext('2d');
    this.ctx.scale(window.devicePixelRatio, window.devicePixelRatio);

    this.overlay.appendChild(this.canvas);
    document.body.appendChild(this.overlay);
  },

  /**
   * Gèle les interactions avec la page
   */
  freezePage() {
    this.pageBlocker = document.createElement('div');
    this.pageBlocker.id = 'ux-tracker-page-blocker';
    this.pageBlocker.style.cssText = `
      position: fixed !important;
      top: 0 !important;
      left: 0 !important;
      width: 100vw !important;
      height: 100vh !important;
      z-index: 2147483646 !important;
      background: rgba(0, 0, 0, 0.03) !important;
      pointer-events: auto !important;
      backdrop-filter: blur(0.5px) !important;
    `;
    document.body.appendChild(this.pageBlocker);
  },

  /**
   * Crée la toolbar flottante
   */
  createToolbar() {
    console.log('🎨 Création de la toolbar...');

    this.toolbar = document.createElement('div');
    this.toolbar.id = 'ux-tracker-feedback-toolbar';

    // Ajouter des styles inline de fallback CRITIQUES pour s'assurer que la toolbar est visible
    this.toolbar.style.cssText = `
      position: fixed !important;
      bottom: 20px !important;
      left: 50% !important;
      transform: translateX(-50%) !important;
      z-index: 2147483648 !important;
      display: flex !important;
      align-items: center !important;
      gap: 16px !important;
      background: linear-gradient(135deg, #9800F3 0%, #B84EFF 100%) !important;
      padding: 16px 24px !important;
      border-radius: 20px !important;
      box-shadow: 0 8px 32px rgba(152, 0, 243, 0.4), 0 2px 8px rgba(0, 0, 0, 0.1) !important;
      font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', sans-serif !important;
      user-select: none !important;
      pointer-events: auto !important;
    `;

    this.toolbar.innerHTML = `
      <div class="feedback-tools">
        <button class="tool-btn active" data-tool="draw" title="Dessin libre">
          ✏️
        </button>
        <button class="tool-btn" data-tool="arrow" title="Flèche">
          ➡️
        </button>
        <button class="tool-btn" data-tool="postit" title="Note">
          📝
        </button>
        <button class="tool-btn" data-tool="frustration" title="Pain Point / Frustration">
          😤
        </button>
        <button class="tool-btn" data-tool="highlight" title="Zone de surbrillance">
          🔴
        </button>
      </div>

      <div class="feedback-colors">
        <button class="color-btn active" data-color="#FF0000" style="background: #FF0000" title="Rouge"></button>
        <button class="color-btn" data-color="#00FF00" style="background: #00FF00" title="Vert"></button>
        <button class="color-btn" data-color="#0000FF" style="background: #0000FF" title="Bleu"></button>
        <button class="color-btn" data-color="#9800F3" style="background: #9800F3" title="Violet SUMO"></button>
        <button class="color-btn" data-color="#FFF59D" style="background: #FFF59D" title="Jaune"></button>
      </div>

      <div class="feedback-actions">
        <button id="feedback-undo" class="action-btn" title="Annuler (Ctrl+Z)">
          ↶ Annuler
        </button>
        <button id="feedback-clear" class="action-btn danger" title="Tout effacer">
          🗑️ Effacer
        </button>
      </div>

      <div class="feedback-validate">
        <button id="feedback-cancel" class="validate-btn cancel" title="Annuler le feedback">
          ❌ Annuler
        </button>
        <button id="feedback-save" class="validate-btn save" title="Sauvegarder et continuer">
          ✅ Valider
        </button>
      </div>
    `;

    document.body.appendChild(this.toolbar);

    console.log('🎨 Toolbar ajoutée au DOM:', this.toolbar);
    console.log('🎨 Toolbar visible:', this.toolbar.offsetParent !== null);

    // Ajouter des styles inline de fallback pour tous les boutons
    this.applyFallbackStyles();

    // Attacher les événements
    this.attachToolbarEvents();
  },

  /**
   * Applique des styles inline de fallback aux boutons
   */
  applyFallbackStyles() {
    // Groupes
    const groups = this.toolbar.querySelectorAll('.feedback-tools, .feedback-colors, .feedback-actions, .feedback-validate');
    groups.forEach(group => {
      group.style.cssText = `
        display: flex !important;
        align-items: center !important;
        gap: 8px !important;
      `;
    });

    // Separators
    const separators = this.toolbar.querySelectorAll('.feedback-tools, .feedback-colors, .feedback-actions');
    separators.forEach(sep => {
      sep.style.paddingRight = '16px';
      sep.style.borderRight = '1px solid rgba(255, 255, 255, 0.3)';
    });

    // Tool buttons
    const toolBtns = this.toolbar.querySelectorAll('.tool-btn');
    toolBtns.forEach(btn => {
      btn.style.cssText = `
        width: 48px !important;
        height: 48px !important;
        display: flex !important;
        align-items: center !important;
        justify-content: center !important;
        background: rgba(255, 255, 255, 0.2) !important;
        border: 1px solid rgba(255, 255, 255, 0.3) !important;
        border-radius: 12px !important;
        font-size: 20px !important;
        cursor: pointer !important;
        color: white !important;
        padding: 0 !important;
        margin: 0 !important;
      `;

      if (btn.classList.contains('active')) {
        btn.style.background = 'white !important';
        btn.style.color = '#9800F3 !important';
        btn.style.borderColor = 'white !important';
      }
    });

    // Color buttons
    const colorBtns = this.toolbar.querySelectorAll('.color-btn');
    colorBtns.forEach(btn => {
      const baseStyles = `
        width: 32px !important;
        height: 32px !important;
        border-radius: 50% !important;
        border: 3px solid rgba(255, 255, 255, 0.5) !important;
        cursor: pointer !important;
        padding: 0 !important;
        margin: 0 !important;
      `;
      btn.style.cssText = baseStyles + btn.getAttribute('style');

      if (btn.classList.contains('active')) {
        btn.style.borderColor = 'white !important';
        btn.style.borderWidth = '4px !important';
      }
    });

    // Action buttons
    const actionBtns = this.toolbar.querySelectorAll('.action-btn');
    actionBtns.forEach(btn => {
      btn.style.cssText = `
        display: flex !important;
        align-items: center !important;
        gap: 6px !important;
        padding: 10px 16px !important;
        background: rgba(255, 255, 255, 0.2) !important;
        border: 1px solid rgba(255, 255, 255, 0.3) !important;
        border-radius: 10px !important;
        color: white !important;
        font-size: 13px !important;
        font-weight: 500 !important;
        cursor: pointer !important;
        margin: 0 !important;
      `;
    });

    // Validate buttons
    const cancelBtn = this.toolbar.querySelector('.validate-btn.cancel');
    if (cancelBtn) {
      cancelBtn.style.cssText = `
        display: flex !important;
        align-items: center !important;
        gap: 6px !important;
        padding: 12px 20px !important;
        border-radius: 12px !important;
        font-size: 14px !important;
        font-weight: 600 !important;
        cursor: pointer !important;
        background: rgba(255, 255, 255, 0.2) !important;
        border: 2px solid rgba(255, 255, 255, 0.5) !important;
        color: white !important;
        margin: 0 !important;
      `;
    }

    const saveBtn = this.toolbar.querySelector('.validate-btn.save');
    if (saveBtn) {
      saveBtn.style.cssText = `
        display: flex !important;
        align-items: center !important;
        gap: 6px !important;
        padding: 12px 20px !important;
        border-radius: 12px !important;
        font-size: 14px !important;
        font-weight: 600 !important;
        cursor: pointer !important;
        background: white !important;
        border: 2px solid white !important;
        color: #9800F3 !important;
        margin: 0 !important;
      `;
    }

    console.log('🎨 Styles de fallback appliqués');
  },

  /**
   * Attache les événements de la toolbar
   */
  attachToolbarEvents() {
    // Sélection outil
    this.toolbar.querySelectorAll('.tool-btn').forEach(btn => {
      btn.addEventListener('click', (e) => {
        this.toolbar.querySelectorAll('.tool-btn').forEach(b => b.classList.remove('active'));
        btn.classList.add('active');
        this.currentTool = btn.dataset.tool;

        // Changer le curseur avec classe CSS
        if (this.currentTool === 'postit' || this.currentTool === 'frustration') {
          this.overlay.classList.add('cursor-text');
        } else {
          this.overlay.classList.remove('cursor-text');
        }
      });
    });

    // Sélection couleur
    this.toolbar.querySelectorAll('.color-btn').forEach(btn => {
      btn.addEventListener('click', () => {
        this.toolbar.querySelectorAll('.color-btn').forEach(b => b.classList.remove('active'));
        btn.classList.add('active');
        this.currentColor = btn.dataset.color;
      });
    });

    // Actions
    document.getElementById('feedback-undo').addEventListener('click', () => this.undo());
    document.getElementById('feedback-clear').addEventListener('click', () => this.clear());
    document.getElementById('feedback-cancel').addEventListener('click', () => this.cancel());
    document.getElementById('feedback-save').addEventListener('click', () => this.save());
  },

  /**
   * Initialise les événements de dessin
   */
  initEvents() {
    this.canvas.addEventListener('mousedown', (e) => this.handleMouseDown(e));
    this.canvas.addEventListener('mousemove', (e) => this.handleMouseMove(e));
    this.canvas.addEventListener('mouseup', (e) => this.handleMouseUp(e));

    // Raccourcis clavier
    document.addEventListener('keydown', (e) => {
      if (!this.isActive) return;

      if (e.ctrlKey && e.key === 'z') {
        e.preventDefault();
        this.undo();
      }
      if (e.key === 'Escape') {
        e.preventDefault();
        this.cancel();
      }
    });
  },

  /**
   * Gère le début du dessin
   */
  handleMouseDown(e) {
    this.isDrawing = true;
    const point = this.getCanvasPoint(e);

    switch (this.currentTool) {
      case 'draw':
        this.tempAnnotation = {
          type: 'drawing',
          color: this.currentColor,
          strokeWidth: 3,
          points: [point]
        };
        break;

      case 'arrow':
        this.tempAnnotation = {
          type: 'arrow',
          color: this.currentColor,
          strokeWidth: 4,
          startX: point.x,
          startY: point.y,
          endX: point.x,
          endY: point.y
        };
        break;

      case 'postit':
        this.createPostit(point, false);
        break;

      case 'frustration':
        this.createPostit(point, true);
        break;

      case 'highlight':
        this.tempAnnotation = {
          type: 'highlight',
          color: this.currentColor,
          startX: point.x,
          startY: point.y,
          endX: point.x,
          endY: point.y
        };
        break;
    }
  },

  /**
   * Gère le mouvement de la souris
   */
  handleMouseMove(e) {
    if (!this.isDrawing || !this.tempAnnotation) return;

    const point = this.getCanvasPoint(e);

    switch (this.currentTool) {
      case 'draw':
        this.tempAnnotation.points.push(point);
        this.redrawCanvas();
        this.drawAnnotation(this.tempAnnotation);
        break;

      case 'arrow':
        this.tempAnnotation.endX = point.x;
        this.tempAnnotation.endY = point.y;
        this.redrawCanvas();
        this.drawAnnotation(this.tempAnnotation);
        break;

      case 'highlight':
        this.tempAnnotation.endX = point.x;
        this.tempAnnotation.endY = point.y;
        this.redrawCanvas();
        this.drawAnnotation(this.tempAnnotation);
        break;
    }
  },

  /**
   * Gère la fin du dessin
   */
  handleMouseUp(e) {
    if (!this.isDrawing) return;

    this.isDrawing = false;

    if (this.tempAnnotation && this.currentTool !== 'postit' && this.currentTool !== 'frustration') {
      this.annotations.push(this.tempAnnotation);
      this.tempAnnotation = null;
      this.redrawCanvas();
    }
  },

  /**
   * Convertit les coordonnées écran en coordonnées canvas
   */
  getCanvasPoint(e) {
    const rect = this.canvas.getBoundingClientRect();
    return {
      x: e.clientX - rect.left,
      y: e.clientY - rect.top
    };
  },

  /**
   * Redessine tout le canvas
   */
  redrawCanvas() {
    this.ctx.clearRect(0, 0, window.innerWidth, window.innerHeight);
    this.annotations.forEach(annotation => this.drawAnnotation(annotation));
  },

  /**
   * Dessine une annotation
   */
  drawAnnotation(annotation) {
    this.ctx.strokeStyle = annotation.color;
    this.ctx.lineWidth = annotation.strokeWidth || 3;
    this.ctx.lineCap = 'round';
    this.ctx.lineJoin = 'round';

    switch (annotation.type) {
      case 'drawing':
        this.drawPath(annotation.points);
        break;

      case 'arrow':
        this.drawArrow(annotation);
        break;

      case 'highlight':
        this.drawHighlight(annotation);
        break;
    }
  },

  /**
   * Dessine un tracé libre
   */
  drawPath(points) {
    if (points.length < 2) return;

    this.ctx.beginPath();
    this.ctx.moveTo(points[0].x, points[0].y);

    for (let i = 1; i < points.length; i++) {
      this.ctx.lineTo(points[i].x, points[i].y);
    }

    this.ctx.stroke();
  },

  /**
   * Dessine une flèche
   */
  drawArrow(arrow) {
    const {startX, startY, endX, endY} = arrow;

    // Ligne principale
    this.ctx.beginPath();
    this.ctx.moveTo(startX, startY);
    this.ctx.lineTo(endX, endY);
    this.ctx.stroke();

    // Pointe de la flèche
    const headLength = 20;
    const angle = Math.atan2(endY - startY, endX - startX);

    this.ctx.beginPath();
    this.ctx.moveTo(endX, endY);
    this.ctx.lineTo(
      endX - headLength * Math.cos(angle - Math.PI / 6),
      endY - headLength * Math.sin(angle - Math.PI / 6)
    );
    this.ctx.moveTo(endX, endY);
    this.ctx.lineTo(
      endX - headLength * Math.cos(angle + Math.PI / 6),
      endY - headLength * Math.sin(angle + Math.PI / 6)
    );
    this.ctx.stroke();
  },

  /**
   * Dessine une zone de surbrillance
   */
  drawHighlight(highlight) {
    const {startX, startY, endX, endY, color} = highlight;

    // Rectangle avec fill semi-transparent
    this.ctx.fillStyle = color + '30'; // Ajoute alpha 30%
    this.ctx.fillRect(startX, startY, endX - startX, endY - startY);

    // Bordure
    this.ctx.strokeRect(startX, startY, endX - startX, endY - startY);
  },

  /**
   * Crée un post-it (comme élément DOM)
   */
  createPostit(point, isFrustration = false) {
    const text = prompt(isFrustration ? 'Décrivez le pain point :' : 'Entrez votre note :');
    if (!text) return;

    const postitId = `postit-${Date.now()}`;
    const color = isFrustration ? '#FFB3BA' : (this.currentColor === '#FFF59D' ? this.currentColor : '#FFF59D');

    const annotation = {
      type: 'postit',
      id: postitId,
      x: point.x,
      y: point.y,
      width: 200,
      height: 150,
      text: text,
      color: color,
      isFrustration: isFrustration
    };

    this.annotations.push(annotation);
    this.createPostitElement(annotation);
  },

  /**
   * Crée un élément DOM pour le post-it
   */
  createPostitElement(postit) {
    const {id, x, y, width, height, text, color, isFrustration} = postit;

    const element = document.createElement('div');
    element.id = id;
    element.className = 'ux-tracker-postit';
    element.style.cssText = `
      position: absolute !important;
      left: ${x}px !important;
      top: ${y}px !important;
      width: ${width}px !important;
      min-height: ${height}px !important;
      background: ${color} !important;
      border: 2px solid ${isFrustration ? '#FF6B6B' : '#CCC'} !important;
      border-radius: 8px !important;
      padding: 12px !important;
      box-shadow: 0 4px 12px rgba(0, 0, 0, 0.2) !important;
      cursor: move !important;
      z-index: 2147483647 !important;
      font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', sans-serif !important;
      font-size: 14px !important;
      color: #000 !important;
      word-wrap: break-word !important;
      user-select: none !important;
      pointer-events: auto !important;
    `;

    // Ajouter l'icône et le texte
    element.innerHTML = `
      <div style="display: flex; align-items: flex-start; gap: 8px;">
        <span style="font-size: 18px; flex-shrink: 0;">${isFrustration ? '😤' : '📝'}</span>
        <p style="margin: 0; flex: 1; line-height: 1.5;">${this.escapeHtml(text)}</p>
      </div>
    `;

    // Ajouter les événements de drag & drop
    element.addEventListener('mousedown', (e) => this.startDragPostit(e, postit));

    // Ajouter double-clic pour éditer
    element.addEventListener('dblclick', (e) => {
      e.stopPropagation();
      this.editPostit(postit, element);
    });

    this.overlay.appendChild(element);
    this.postitElements.push({id, element});
  },

  /**
   * Éditer le texte d'un post-it
   */
  editPostit(postit, element) {
    const newText = prompt('Modifier la note :', postit.text);
    if (newText === null) return; // Annulé
    if (newText.trim() === '') {
      alert('Le texte ne peut pas être vide');
      return;
    }

    // Mettre à jour le texte dans l'annotation
    postit.text = newText;

    // Mettre à jour l'affichage
    element.innerHTML = `
      <div style="display: flex; align-items: flex-start; gap: 8px;">
        <span style="font-size: 18px; flex-shrink: 0;">${postit.isFrustration ? '😤' : '📝'}</span>
        <p style="margin: 0; flex: 1; line-height: 1.5;">${this.escapeHtml(newText)}</p>
      </div>
    `;
  },

  /**
   * Commence le drag d'un post-it
   */
  startDragPostit(e, postit) {
    e.stopPropagation();

    const startX = e.clientX;
    const startY = e.clientY;
    let hasMoved = false;

    this.draggedPostit = {
      postit: postit,
      offsetX: e.clientX - postit.x,
      offsetY: e.clientY - postit.y
    };

    // Événements temporaires pour le drag
    const moveHandler = (e) => {
      // Considérer comme mouvement si déplacement > 3px
      if (!hasMoved && (Math.abs(e.clientX - startX) > 3 || Math.abs(e.clientY - startY) > 3)) {
        hasMoved = true;
      }
      if (hasMoved) {
        this.dragPostit(e);
      }
    };

    const upHandler = () => {
      this.stopDragPostit(moveHandler, upHandler);
      // Si pas de mouvement, c'était juste un clic
      if (!hasMoved) {
        this.draggedPostit = null;
      }
    };

    document.addEventListener('mousemove', moveHandler);
    document.addEventListener('mouseup', upHandler);
  },

  /**
   * Déplace le post-it pendant le drag
   */
  dragPostit(e) {
    if (!this.draggedPostit) return;

    const newX = e.clientX - this.draggedPostit.offsetX;
    const newY = e.clientY - this.draggedPostit.offsetY;

    // Mettre à jour la position dans l'annotation
    this.draggedPostit.postit.x = newX;
    this.draggedPostit.postit.y = newY;

    // Mettre à jour l'élément DOM
    const postitData = this.postitElements.find(p => p.id === this.draggedPostit.postit.id);
    if (postitData) {
      postitData.element.style.left = `${newX}px`;
      postitData.element.style.top = `${newY}px`;
    }
  },

  /**
   * Termine le drag du post-it
   */
  stopDragPostit(moveHandler, upHandler) {
    document.removeEventListener('mousemove', moveHandler);
    document.removeEventListener('mouseup', upHandler);
    this.draggedPostit = null;
  },

  /**
   * Échapper le HTML pour éviter les injections
   */
  escapeHtml(text) {
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
  },

  /**
   * Annuler la dernière action
   */
  undo() {
    if (this.annotations.length === 0) return;

    const lastAnnotation = this.annotations.pop();

    // Si c'est un post-it, retirer aussi l'élément DOM
    if (lastAnnotation.type === 'postit') {
      const postitIndex = this.postitElements.findIndex(p => p.id === lastAnnotation.id);
      if (postitIndex !== -1) {
        const {element} = this.postitElements[postitIndex];
        if (element && element.parentNode) {
          element.remove();
        }
        this.postitElements.splice(postitIndex, 1);
      }
    }

    this.redrawCanvas();
  },

  /**
   * Tout effacer
   */
  clear() {
    if (!confirm('Voulez-vous vraiment tout effacer ?')) return;

    // Retirer tous les post-it DOM
    this.postitElements.forEach(({element}) => {
      if (element && element.parentNode) {
        element.remove();
      }
    });

    this.annotations = [];
    this.postitElements = [];
    this.redrawCanvas();
  },

  /**
   * Annuler le feedback
   */
  cancel() {
    if (this.annotations.length > 0) {
      if (!confirm('Annuler les annotations ? Elles seront perdues.')) return;
    }
    this.deactivate();
  },

  /**
   * Sauvegarder les annotations
   */
  async save() {
    console.log('💾 Sauvegarde des annotations...', this.annotations);

    // IMPORTANT: Capturer un screenshot AVANT de fermer l'overlay
    let screenshotDataUrl = null;
    try {
      console.log('📸 Capture du screenshot du feedback...');
      const response = await chrome.runtime.sendMessage({
        action: 'CAPTURE_SCREENSHOT',
        data: {
          trigger: 'feedback_validation',
          url: window.location.href,
          viewport: {
            width: window.innerWidth,
            height: window.innerHeight,
            scrollX: window.scrollX,
            scrollY: window.scrollY
          }
        }
      });

      if (response && response.success) {
        screenshotDataUrl = response.dataUrl;
        console.log('📸 Screenshot capturé pour le feedback');
      }
    } catch (error) {
      console.error('Erreur capture screenshot feedback:', error);
    }

    const feedbackData = {
      feedbackId: `fb-${Date.now()}`,
      timestamp: new Date().toISOString(),
      url: window.location.href,
      viewport: {
        width: window.innerWidth,
        height: window.innerHeight,
        scrollX: window.scrollX,
        scrollY: window.scrollY
      },
      annotations: this.annotations,
      screenshot: screenshotDataUrl // Ajouter le screenshot
    };

    // Envoyer au background script pour sauvegarde
    chrome.runtime.sendMessage({
      action: 'saveFeedback',
      feedback: feedbackData
    });

    // Désactiver le mode feedback
    this.deactivate();

    // Afficher confirmation
    alert('✅ Feedback sauvegardé !');
  }
};

// Exposer FeedbackOverlay globalement pour que content-main.js puisse y accéder
window.FeedbackOverlay = FeedbackOverlay;

console.log('✅ Feedback Overlay chargé et exposé globalement');
