/**
 * SUMO UX Tracker V3 - Recording Indicator
 * Affiche un indicateur visuel REC en haut à droite de la page
 */

class RecordingIndicator {
  constructor() {
    this.indicator = null;
    this.isVisible = false;
  }

  /**
   * Crée l'élément DOM de l'indicateur
   */
  create() {
    if (this.indicator) {
      return; // Déjà créé
    }

    // Créer conteneur principal
    this.indicator = document.createElement('div');
    this.indicator.id = 'sumo-rec-indicator';
    this.indicator.className = 'sumo-rec-indicator';

    // Structure HTML
    this.indicator.innerHTML = `
      <div class="sumo-rec-dot"></div>
      <span class="sumo-rec-text">REC</span>
    `;

    // Ajouter styles inline (pour éviter conflits avec page)
    this.injectStyles();

    // Ne pas ajouter au DOM tout de suite
    console.log('[RecordingIndicator] Created');
  }

  /**
   * Injecte les styles CSS dans la page
   */
  injectStyles() {
    // Vérifier si styles déjà injectés
    if (document.getElementById('sumo-rec-styles')) {
      return;
    }

    // Vérifier que document.head existe
    if (!document.head) {
      console.warn('[RecordingIndicator] document.head not ready, retrying...');
      setTimeout(() => this.injectStyles(), 100);
      return;
    }

    const style = document.createElement('style');
    style.id = 'sumo-rec-styles';
    style.textContent = `
      /* Indicateur REC */
      .sumo-rec-indicator {
        position: fixed;
        top: 20px;
        right: 20px;
        z-index: 2147483647; /* Max z-index */
        display: flex;
        align-items: center;
        gap: 8px;
        background: rgba(0, 0, 0, 0.85);
        backdrop-filter: blur(8px);
        padding: 10px 16px;
        border-radius: 24px;
        box-shadow: 0 4px 12px rgba(239, 68, 68, 0.3);
        font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, 'Helvetica Neue', Arial, sans-serif;
        user-select: none;
        pointer-events: none; /* Ne bloque pas les clics */
        animation: sumo-rec-fade-in 0.3s ease;
      }

      /* Dot rouge animé */
      .sumo-rec-dot {
        width: 12px;
        height: 12px;
        background: #EF4444;
        border-radius: 50%;
        animation: sumo-rec-pulse 1.5s ease-in-out infinite;
        box-shadow: 0 0 8px rgba(239, 68, 68, 0.6);
      }

      /* Texte REC */
      .sumo-rec-text {
        color: #FFFFFF;
        font-size: 14px;
        font-weight: 700;
        letter-spacing: 0.1em;
        text-transform: uppercase;
      }

      /* Animation fade in */
      @keyframes sumo-rec-fade-in {
        from {
          opacity: 0;
          transform: translateY(-10px);
        }
        to {
          opacity: 1;
          transform: translateY(0);
        }
      }

      /* Animation pulsation du dot */
      @keyframes sumo-rec-pulse {
        0%, 100% {
          transform: scale(1);
          opacity: 1;
        }
        50% {
          transform: scale(1.2);
          opacity: 0.7;
        }
      }

      /* Responsive mobile */
      @media (max-width: 768px) {
        .sumo-rec-indicator {
          top: 10px;
          right: 10px;
          padding: 8px 12px;
        }

        .sumo-rec-dot {
          width: 10px;
          height: 10px;
        }

        .sumo-rec-text {
          font-size: 12px;
        }
      }
    `;

    document.head.appendChild(style);
    console.log('[RecordingIndicator] Styles injected');
  }

  /**
   * Affiche l'indicateur
   */
  show() {
    if (this.isVisible) {
      return;
    }

    if (!this.indicator) {
      this.create();
    }

    // Vérifier que document.body existe
    if (!document.body) {
      console.warn('[RecordingIndicator] document.body not ready, retrying...');
      setTimeout(() => this.show(), 100);
      return;
    }

    // Ajouter au DOM
    document.body.appendChild(this.indicator);
    this.isVisible = true;

    console.log('[RecordingIndicator] Shown');
  }

  /**
   * Masque l'indicateur
   */
  hide() {
    if (!this.isVisible || !this.indicator) {
      return;
    }

    // Retirer du DOM
    if (this.indicator.parentNode) {
      this.indicator.parentNode.removeChild(this.indicator);
    }

    this.isVisible = false;

    console.log('[RecordingIndicator] Hidden');
  }

  /**
   * Détruit l'indicateur
   */
  destroy() {
    this.hide();

    // Retirer styles
    const styles = document.getElementById('sumo-rec-styles');
    if (styles && styles.parentNode) {
      styles.parentNode.removeChild(styles);
    }

    this.indicator = null;
    this.isVisible = false;

    console.log('[RecordingIndicator] Destroyed');
  }
}

// Export pour utilisation dans content-main.js
window.RecordingIndicator = RecordingIndicator;
