/**
 * SUMO UX Tracker V3 - Scroll Tracker
 * Capture scroll position (throttled)
 */

class ScrollTracker {
  constructor(callback) {
    this.callback = callback;
    this.enabled = false;
    this.lastSampleTime = 0;
    this.sampleRate = 200; // 1 sample tous les 200ms
    this.scrollCount = 0;
    
    this.handleScroll = this.handleScroll.bind(this);
  }
  
  start() {
    if (this.enabled) return;
    this.enabled = true;
    window.addEventListener('scroll', this.handleScroll, { passive: true });
    console.log('[ScrollTracker] Started');
  }
  
  stop() {
    this.enabled = false;
    window.removeEventListener('scroll', this.handleScroll);
    console.log('[ScrollTracker] Stopped');
  }
  
  handleScroll() {
    if (!this.enabled) return;
    
    const now = Date.now();
    
    // Throttle: 1 sample tous les 200ms
    if (now - this.lastSampleTime < this.sampleRate) return;
    
    this.lastSampleTime = now;
    this.scrollCount++;
    
    this.callback({
      type: 'scroll',
      timestamp: now,
      scrollX: window.scrollX,
      scrollY: window.scrollY,
      scrollHeight: document.documentElement.scrollHeight,
      viewportHeight: window.innerHeight,
      scrollPercentage: Math.round(
        (window.scrollY / (document.documentElement.scrollHeight - window.innerHeight)) * 100
      )
    });
  }
  
  getStats() {
    return {
      totalScrolls: this.scrollCount
    };
  }
}

// Exposer globalement
window.ScrollTracker = ScrollTracker;
