/**
 * SUMO UX Tracker V3 - Screenshot Handler
 * Gère capture de screenshots intelligemment
 */

class ScreenshotHandler {
  constructor(callback) {
    this.callback = callback;
    this.enabled = false;
    this.screenshotCount = 0;
    this.pendingCaptures = [];
  }
  
  start() {
    if (this.enabled) return;
    this.enabled = true;
    
    // Capturer screenshot initial
    this.captureScreenshot(
      window.SUMO_CONSTANTS.SCREENSHOT_TRIGGERS.INITIAL
    );
    
    console.log('[ScreenshotHandler] Started');
  }
  
  stop() {
    this.enabled = false;
    console.log('[ScreenshotHandler] Stopped');
  }
  
  /**
   * Demande capture screenshot
   */
  requestCapture(trigger, metadata = {}) {
    if (!this.enabled) return;
    
    // Éviter captures trop rapprochées
    const minDelay = 500; // 500ms entre captures
    const now = Date.now();
    
    if (this.lastCaptureTime && (now - this.lastCaptureTime) < minDelay) {
      console.log('[ScreenshotHandler] Capture throttled');
      return;
    }
    
    this.captureScreenshot(trigger, metadata);
  }
  
  /**
   * Capture screenshot via background
   */
  async captureScreenshot(trigger, metadata = {}) {
    this.screenshotCount++;
    this.lastCaptureTime = Date.now();
    
    const screenshotData = {
      id: this.screenshotCount,
      trigger: trigger,
      timestamp: Date.now(),
      url: window.location.href,
      viewport: {
        width: window.innerWidth,
        height: window.innerHeight,
        scrollX: window.scrollX,
        scrollY: window.scrollY
      },
      ...metadata
    };
    
    console.log(`[ScreenshotHandler] Capturing screenshot #${this.screenshotCount} (${trigger})`);
    
    try {
      // Demander au background de capturer
      const response = await chrome.runtime.sendMessage({
        action: 'CAPTURE_SCREENSHOT',
        data: screenshotData
      });
      
      if (response?.success) {
        // Envoyer événement avec dataUrl du screenshot
        this.callback({
          type: 'screenshot_captured',
          ...screenshotData,
          dataUrl: response.dataUrl
        });
        
        console.log(`[ScreenshotHandler] Screenshot #${this.screenshotCount} captured successfully`);
      } else {
        console.error('[ScreenshotHandler] Failed to capture:', response?.error);
      }
      
    } catch (error) {
      console.error('[ScreenshotHandler] Error capturing screenshot:', error);
    }
  }
  
  getStats() {
    return {
      totalScreenshots: this.screenshotCount
    };
  }
}

// Exposer globalement
window.ScreenshotHandler = ScreenshotHandler;
