/**
 * SUMO UX Tracker V3 - Patterns Analyzer
 * Détecte rage clicks et autres patterns
 */

class PatternsAnalyzer {
  constructor(callback) {
    this.callback = callback;
    this.enabled = false;
    this.clickHistory = [];
    this.hoverHistory = [];
    this.scrollHistory = [];
    this.formInteractions = new Map(); // trackage des champs de formulaire
    this.deadClickElements = new Set(); // éléments ayant eu des dead clicks
  }

  start() {
    if (this.enabled) return;
    this.enabled = true;

    // Attacher listeners pour détection avancée
    this.attachListeners();

    console.log('[PatternsAnalyzer] Started with advanced pattern detection');
  }

  stop() {
    this.enabled = false;
    this.removeListeners();
    console.log('[PatternsAnalyzer] Stopped');
  }

  /**
   * Attache les listeners pour détection de patterns
   */
  attachListeners() {
    // Listener pour hovers prolongés
    this.hoverListener = (e) => this.trackHover(e);
    document.addEventListener('mousemove', this.hoverListener);

    // Listener pour scroll répétitif
    this.scrollListener = () => this.trackScroll();
    window.addEventListener('scroll', this.scrollListener);

    // Listener pour interactions formulaires
    this.inputListener = (e) => this.trackFormInput(e);
    document.addEventListener('input', this.inputListener, true);

    this.focusListener = (e) => this.trackFormFocus(e);
    document.addEventListener('focus', this.focusListener, true);
  }

  /**
   * Retire les listeners
   */
  removeListeners() {
    if (this.hoverListener) {
      document.removeEventListener('mousemove', this.hoverListener);
    }
    if (this.scrollListener) {
      window.removeEventListener('scroll', this.scrollListener);
    }
    if (this.inputListener) {
      document.removeEventListener('input', this.inputListener, true);
    }
    if (this.focusListener) {
      document.removeEventListener('focus', this.focusListener, true);
    }
  }
  
  /**
   * Analyse un clic pour détecter patterns
   */
  analyzeClick(clickData) {
    if (!this.enabled) return;

    this.clickHistory.push({
      x: clickData.x,
      y: clickData.y,
      timestamp: clickData.timestamp,
      element: clickData.element
    });

    // Nettoyer historique (garder 3 dernières secondes)
    this.cleanHistory();

    // Détecter rage click
    const rageClick = this.detectRageClick();
    if (rageClick) {
      console.log('[PatternsAnalyzer] 🔥 RAGE CLICK detected!');

      this.callback({
        type: 'pattern_detected',
        pattern: 'rage_click',
        timestamp: Date.now(),
        intensity: rageClick.intensity,
        clickCount: rageClick.count,
        duration: rageClick.duration,
        position: rageClick.center
      });

      // Demander screenshot du rage click
      this.callback({
        type: 'screenshot_request',
        trigger: window.SUMO_CONSTANTS.SCREENSHOT_TRIGGERS.PATTERN_RAGE,
        pattern: 'rage_click',
        intensity: rageClick.intensity
      });
    }

    // Détecter dead click (clic sur élément non-interactif)
    const deadClick = this.detectDeadClick(clickData);
    if (deadClick) {
      console.log('[PatternsAnalyzer] 💀 DEAD CLICK detected!');

      this.callback({
        type: 'pattern_detected',
        pattern: 'dead_click',
        timestamp: Date.now(),
        element: deadClick.element,
        position: { x: clickData.x, y: clickData.y }
      });
    }
  }
  
  /**
   * Détecte rage click
   */
  detectRageClick() {
    const cfg = window.SUMO_CONSTANTS.PATTERNS;
    
    if (this.clickHistory.length < cfg.RAGE_CLICK_COUNT) {
      return null;
    }
    
    const now = Date.now();
    const recentClicks = this.clickHistory.filter(
      click => (now - click.timestamp) < cfg.RAGE_CLICK_TIME
    );
    
    if (recentClicks.length < cfg.RAGE_CLICK_COUNT) {
      return null;
    }
    
    // Vérifier si clics sont dans même zone
    const firstClick = recentClicks[0];
    const allInRadius = recentClicks.every(click => {
      const distance = Math.sqrt(
        Math.pow(click.x - firstClick.x, 2) + 
        Math.pow(click.y - firstClick.y, 2)
      );
      return distance < cfg.RAGE_CLICK_RADIUS;
    });
    
    if (!allInRadius) {
      return null;
    }
    
    // Calculer centre et intensité
    const center = {
      x: Math.round(recentClicks.reduce((sum, c) => sum + c.x, 0) / recentClicks.length),
      y: Math.round(recentClicks.reduce((sum, c) => sum + c.y, 0) / recentClicks.length)
    };
    
    const duration = now - recentClicks[0].timestamp;
    const intensity = Math.min(10, Math.round(recentClicks.length / 2));
    
    return {
      count: recentClicks.length,
      duration: duration,
      center: center,
      intensity: intensity
    };
  }
  
  /**
   * Détecte dead click (clic sur élément non-interactif)
   */
  detectDeadClick(clickData) {
    const element = clickData.element;
    if (!element) return null;

    // Liste des tags interactifs
    const interactiveTags = ['A', 'BUTTON', 'INPUT', 'SELECT', 'TEXTAREA', 'LABEL'];

    // Liste des rôles interactifs
    const interactiveRoles = ['button', 'link', 'menuitem', 'tab', 'checkbox', 'radio'];

    // Vérifier si l'élément est interactif
    const tagName = element.tagName;
    const role = element.getAttribute('role');
    const isClickable = element.onclick || element.style.cursor === 'pointer';
    const hasClickEvent = element.hasAttribute('data-action') || element.classList.contains('clickable');

    const isInteractive =
      interactiveTags.includes(tagName) ||
      (role && interactiveRoles.includes(role)) ||
      isClickable ||
      hasClickEvent;

    // Si non-interactif, c'est un dead click
    if (!isInteractive) {
      const elementKey = this.getElementKey(element);

      // Éviter de détecter plusieurs fois le même élément
      if (this.deadClickElements.has(elementKey)) {
        return null;
      }

      this.deadClickElements.add(elementKey);

      return {
        element: tagName,
        selector: this.getElementSelector(element)
      };
    }

    return null;
  }

  /**
   * Track hovers pour détecter confusion zones
   */
  trackHover(e) {
    if (!this.enabled) return;

    const now = Date.now();

    this.hoverHistory.push({
      x: e.clientX,
      y: e.clientY,
      timestamp: now,
      element: e.target
    });

    // Garder seulement 5 dernières secondes
    this.hoverHistory = this.hoverHistory.filter(
      h => (now - h.timestamp) < 5000
    );

    // Détecter confusion (hover prolongé dans petite zone sans clic)
    this.detectConfusionZone();
  }

  /**
   * Détecte zone de confusion (hover prolongé sans action)
   */
  detectConfusionZone() {
    const now = Date.now();
    const threshold = 3000; // 3 secondes de hover

    if (this.hoverHistory.length < 10) return;

    // Prendre les 15 derniers hovers
    const recentHovers = this.hoverHistory.slice(-15);
    const firstHover = recentHovers[0];
    const duration = now - firstHover.timestamp;

    if (duration < threshold) return;

    // Vérifier si tous les hovers sont dans une petite zone (100px de rayon)
    const radius = 100;
    const allInRadius = recentHovers.every(hover => {
      const distance = Math.sqrt(
        Math.pow(hover.x - firstHover.x, 2) +
        Math.pow(hover.y - firstHover.y, 2)
      );
      return distance < radius;
    });

    if (allInRadius) {
      console.log('[PatternsAnalyzer] 😕 CONFUSION ZONE detected!');

      // Clear history pour éviter détections multiples
      this.hoverHistory = [];

      this.callback({
        type: 'pattern_detected',
        pattern: 'confusion_zone',
        timestamp: now,
        duration: duration,
        position: {
          x: Math.round(recentHovers.reduce((sum, h) => sum + h.x, 0) / recentHovers.length),
          y: Math.round(recentHovers.reduce((sum, h) => sum + h.y, 0) / recentHovers.length)
        }
      });
    }
  }

  /**
   * Track scroll pour détecter scroll répétitif
   */
  trackScroll() {
    if (!this.enabled) return;

    const now = Date.now();
    const scrollY = window.scrollY;

    this.scrollHistory.push({
      position: scrollY,
      timestamp: now
    });

    // Garder seulement 10 dernières secondes
    this.scrollHistory = this.scrollHistory.filter(
      s => (now - s.timestamp) < 10000
    );

    // Détecter scroll répétitif
    this.detectRepetitiveScroll();
  }

  /**
   * Détecte scroll répétitif (va-et-vient = recherche)
   */
  detectRepetitiveScroll() {
    if (this.scrollHistory.length < 8) return;

    const recentScrolls = this.scrollHistory.slice(-8);
    let changes = 0;

    // Compter changements de direction
    for (let i = 1; i < recentScrolls.length; i++) {
      const prev = recentScrolls[i - 1].position;
      const curr = recentScrolls[i].position;
      const diff = Math.abs(curr - prev);

      if (diff > 100) { // Scroll significatif
        if (i > 1) {
          const prevPrev = recentScrolls[i - 2].position;
          const direction1 = curr > prev;
          const direction2 = prev > prevPrev;

          if (direction1 !== direction2) {
            changes++;
          }
        }
      }
    }

    // Si 3+ changements de direction = scroll répétitif
    if (changes >= 3) {
      console.log('[PatternsAnalyzer] 🔄 REPETITIVE SCROLL detected!');

      // Clear pour éviter détections multiples
      this.scrollHistory = [];

      this.callback({
        type: 'pattern_detected',
        pattern: 'repetitive_scroll',
        timestamp: Date.now(),
        changes: changes
      });
    }
  }

  /**
   * Track input de formulaire
   */
  trackFormInput(e) {
    if (!this.enabled) return;

    const input = e.target;
    if (!['INPUT', 'TEXTAREA'].includes(input.tagName)) return;

    const inputKey = this.getElementKey(input);
    const value = input.value;

    if (!this.formInteractions.has(inputKey)) {
      this.formInteractions.set(inputKey, {
        element: input,
        values: [],
        cleared: false
      });
    }

    const interaction = this.formInteractions.get(inputKey);
    interaction.values.push({
      value: value,
      timestamp: Date.now()
    });

    // Détecter form abandonment (champ rempli puis vidé)
    if (interaction.values.length >= 2) {
      const lastValue = interaction.values[interaction.values.length - 1].value;
      const prevValue = interaction.values[interaction.values.length - 2].value;

      if (prevValue.length > 3 && lastValue.length === 0 && !interaction.cleared) {
        console.log('[PatternsAnalyzer] 📝❌ FORM ABANDONMENT detected!');

        interaction.cleared = true;

        this.callback({
          type: 'pattern_detected',
          pattern: 'form_abandonment',
          timestamp: Date.now(),
          element: input.tagName,
          fieldType: input.type || 'text',
          charactersLost: prevValue.length
        });
      }
    }
  }

  /**
   * Track focus de formulaire
   */
  trackFormFocus(e) {
    if (!this.enabled) return;

    const input = e.target;
    if (!['INPUT', 'TEXTAREA'].includes(input.tagName)) return;

    const inputKey = this.getElementKey(input);

    if (!this.formInteractions.has(inputKey)) {
      this.formInteractions.set(inputKey, {
        element: input,
        values: [],
        cleared: false
      });
    }
  }

  /**
   * Génère une clé unique pour un élément
   */
  getElementKey(element) {
    if (element.id) return `#${element.id}`;
    if (element.name) return `[name="${element.name}"]`;
    return this.getElementSelector(element);
  }

  /**
   * Génère un sélecteur CSS pour un élément
   */
  getElementSelector(element) {
    if (element.id) return `#${element.id}`;

    let path = [];
    let current = element;

    while (current && current !== document.body) {
      let selector = current.tagName.toLowerCase();

      if (current.className && typeof current.className === 'string') {
        selector += '.' + current.className.trim().split(/\s+/).join('.');
      }

      path.unshift(selector);
      current = current.parentElement;

      if (path.length > 3) break; // Limiter profondeur
    }

    return path.join(' > ');
  }

  /**
   * Nettoie historique vieux clics
   */
  cleanHistory() {
    const now = Date.now();
    const maxAge = 3000; // 3 secondes

    this.clickHistory = this.clickHistory.filter(
      click => (now - click.timestamp) < maxAge
    );
  }

  getStats() {
    return {
      clickHistory: this.clickHistory.length,
      hoverHistory: this.hoverHistory.length,
      scrollHistory: this.scrollHistory.length,
      formInteractions: this.formInteractions.size,
      deadClickElements: this.deadClickElements.size
    };
  }
}

// Exposer globalement
window.PatternsAnalyzer = PatternsAnalyzer;
