/**
 * SUMO UX Tracker V3 - Navigation Tracker
 * Détecte navigation web classique + Figma SPA
 */

class NavigationTracker {
  constructor(callback) {
    this.callback = callback;
    this.enabled = false;
    this.currentUrl = window.location.href;
    this.urlCheckInterval = null;
    this.navigationCount = 0;
    
    this.handlePopState = this.handlePopState.bind(this);
  }
  
  start() {
    if (this.enabled) return;
    this.enabled = true;
    
    // Écouter navigation classique (back/forward)
    window.addEventListener('popstate', this.handlePopState);
    
    // Vérifier URL change périodiquement (pour Figma et SPAs)
    this.urlCheckInterval = setInterval(() => {
      this.checkUrlChange();
    }, window.SUMO_CONSTANTS.DELAYS.URL_CHECK_INTERVAL);
    
    console.log('[NavigationTracker] Started - Monitoring web + Figma navigation');
  }
  
  stop() {
    this.enabled = false;
    window.removeEventListener('popstate', this.handlePopState);
    
    if (this.urlCheckInterval) {
      clearInterval(this.urlCheckInterval);
      this.urlCheckInterval = null;
    }
    
    console.log('[NavigationTracker] Stopped');
  }
  
  handlePopState(event) {
    if (!this.enabled) return;
    
    this.navigationCount++;
    
    this.callback({
      type: 'navigation',
      subtype: 'popstate',
      timestamp: Date.now(),
      from: this.currentUrl,
      to: window.location.href,
      navigationId: this.navigationCount
    });
    
    // Déclencher screenshot de la nouvelle page
    this.callback({
      type: 'screenshot_request',
      trigger: window.SUMO_CONSTANTS.SCREENSHOT_TRIGGERS.NAVIGATION,
      timestamp: Date.now()
    });
    
    this.currentUrl = window.location.href;
    
    console.log(`[NavigationTracker] Navigation #${this.navigationCount} via popstate`);
  }
  
  checkUrlChange() {
    if (!this.enabled) return;
    
    const newUrl = window.location.href;
    
    if (newUrl !== this.currentUrl) {
      this.navigationCount++;
      
      // Détecter type de changement
      const oldHash = this.currentUrl.split('#')[1] || '';
      const newHash = newUrl.split('#')[1] || '';
      const hashChanged = oldHash !== newHash;
      
      const oldPath = this.currentUrl.split('#')[0];
      const newPath = newUrl.split('#')[0];
      const pathChanged = oldPath !== newPath;
      
      this.callback({
        type: 'navigation',
        subtype: hashChanged ? 'hash_change' : 'url_change',
        timestamp: Date.now(),
        from: this.currentUrl,
        to: newUrl,
        navigationId: this.navigationCount,
        isFigma: this.isFigmaUrl(newUrl)
      });
      
      // Déclencher screenshot (attendre 2 sec que Figma charge)
      setTimeout(() => {
        this.callback({
          type: 'screenshot_request',
          trigger: window.SUMO_CONSTANTS.SCREENSHOT_TRIGGERS.NAVIGATION,
          timestamp: Date.now()
        });
      }, 2000);
      
      this.currentUrl = newUrl;
      
      console.log(`[NavigationTracker] Navigation #${this.navigationCount} - URL changed`);
      if (this.isFigmaUrl(newUrl)) {
        console.log('[NavigationTracker] Figma navigation detected');
      }
    }
  }
  
  isFigmaUrl(url) {
    return url.includes('figma.com') || url.includes('figma.prototype');
  }
  
  getStats() {
    return {
      totalNavigations: this.navigationCount,
      currentUrl: this.currentUrl,
      isFigma: this.isFigmaUrl(this.currentUrl)
    };
  }
}

// Exposer globalement
window.NavigationTracker = NavigationTracker;
