/**
 * SUMO UX Tracker V3 - Click Tracker
 * Capture clicks et déclenche screenshot IMMÉDIATEMENT sur mousedown
 */

class ClickTracker {
  constructor(callback) {
    this.callback = callback;
    this.clickCount = 0;
    this.recentClicks = [];
    this.enabled = false;
    
    this.handleClick = this.handleClick.bind(this);
    this.handleMouseDown = this.handleMouseDown.bind(this);
  }
  
  start() {
    if (this.enabled) return;
    this.enabled = true;
    
    // Écouter click (pour enregistrer l'événement)
    document.addEventListener('click', this.handleClick, true);
    
    // Écouter mousedown (pour déclencher screenshot immédiatement)
    document.addEventListener('mousedown', this.handleMouseDown, true);
    
    console.log('[ClickTracker] Started');
  }
  
  stop() {
    this.enabled = false;
    document.removeEventListener('click', this.handleClick, true);
    document.removeEventListener('mousedown', this.handleMouseDown, true);
    console.log('[ClickTracker] Stopped');
  }
  
  /**
   * Déclenche screenshot immédiatement sur mousedown (press)
   */
  handleMouseDown(event) {
    if (!this.enabled) return;
    
    // Attribuer un ID temporaire basé sur le timestamp
    const tempClickId = Date.now();
    
    // Déclencher screenshot IMMÉDIATEMENT
    this.callback({
      type: 'screenshot_request',
      trigger: window.SUMO_CONSTANTS.SCREENSHOT_TRIGGERS.CLICK_AFTER,
      relatedEventId: tempClickId,
      timestamp: Date.now(),
      position: {
        x: event.clientX,
        y: event.clientY
      }
    });
    
    console.log(`[ClickTracker] MouseDown detected at (${event.clientX}, ${event.clientY}), screenshot triggered immediately`);
  }
  
  /**
   * Enregistre le click complet (après mouseup)
   */
  handleClick(event) {
    if (!this.enabled) return;
    
    this.clickCount++;
    const clickId = this.clickCount;
    
    // Données du clic
    const clickData = {
      type: 'click',
      id: clickId,
      timestamp: Date.now(),
      x: event.clientX,
      y: event.clientY,
      pageX: event.pageX,
      pageY: event.pageY,
      target: this.getElementInfo(event.target),
      button: event.button,
      url: window.location.href
    };
    
    // Envoyer l'événement immédiatement
    this.callback(clickData);
    
    // Ajouter aux clics récents pour pattern detection
    this.recentClicks.push({
      x: event.clientX,
      y: event.clientY,
      timestamp: Date.now()
    });
    
    // Nettoyer les vieux clics (> 3 secondes)
    this.cleanOldClicks();
    
    console.log(`[ClickTracker] Click #${clickId} recorded at (${event.clientX}, ${event.clientY})`);
  }
  
  getElementInfo(element) {
    if (!element) return null;
    
    return {
      tagName: element.tagName,
      id: element.id || null,
      className: element.className || null,
      text: element.textContent?.substring(0, 50) || null,
      href: element.href || null
    };
  }
  
  cleanOldClicks() {
    const now = Date.now();
    const maxAge = 3000; // 3 secondes
    
    this.recentClicks = this.recentClicks.filter(
      click => (now - click.timestamp) < maxAge
    );
  }
  
  getRecentClicks() {
    return this.recentClicks;
  }
  
  getStats() {
    return {
      totalClicks: this.clickCount,
      recentClicks: this.recentClicks.length
    };
  }
}

// Exposer globalement
window.ClickTracker = ClickTracker;
