/**
 * SUMO UX Tracker V3 - Content Main
 * Orchestrateur principal - CLEAN & SIMPLE
 */

if (!window.sumoContent) {
  class SUMOContent {
    constructor() {
      this.isTracking = false;
      this.isPaused = false;  // NOUVEAU: état de pause distinct
      this.sessionId = null;
      this.trackers = {};
      this.events = [];
      this.screenshots = [];
      this.recordingIndicator = null;

      this.init();
    }
    
    async init() {
      console.log('[SUMO V3] Initializing on:', window.location.href);

      // Initialiser les trackers
      this.initTrackers();

      // Écouter messages du background
      chrome.runtime.onMessage.addListener((msg, sender, sendResponse) => {
        this.handleMessage(msg, sender, sendResponse);
        return true;
      });

      // Notifier background que content script est prêt
      this.notifyReady();

      // IMPORTANT: Vérifier si une session est active (cas de rechargement de page pendant l'enregistrement)
      await this.checkActiveSession();

      console.log('[SUMO V3] ✅ Ready with', Object.keys(this.trackers).length, 'trackers');
    }
    
    initTrackers() {
      const callback = (event) => this.handleEvent(event);
      
      // Initialiser tous les trackers
      this.trackers.click = new ClickTracker(callback);
      this.trackers.mouse = new MouseTracker(callback);
      this.trackers.scroll = new ScrollTracker(callback);
      this.trackers.time = new TimeTracker(callback);
      this.trackers.navigation = new NavigationTracker(callback);
      this.trackers.screenshot = new ScreenshotHandler(callback);
      this.trackers.patterns = new PatternsAnalyzer(callback);
      
      console.log('[SUMO V3] Trackers initialized:', Object.keys(this.trackers));
    }
    
    handleMessage(msg, sender, sendResponse) {
      console.log('[SUMO V3] Message:', msg.action);
      
      switch (msg.action) {
        case 'PING':
          sendResponse({
            success: true,
            url: window.location.href,
            trackers: Object.keys(this.trackers)
          });
          break;

        case 'START_TRACKING':
          this.startTracking(msg.sessionId);
          sendResponse({ success: true });
          break;

        case 'STOP_TRACKING':
          this.stopTracking();
          sendResponse({ success: true });
          break;

        case 'GET_DATA':
          sendResponse({
            success: true,
            events: this.events,
            screenshots: this.screenshots
          });
          break;

        case 'GET_PAGE_INFO':
          // Infos pour capture page entière
          sendResponse({
            success: true,
            scrollHeight: Math.max(
              document.body.scrollHeight,
              document.documentElement.scrollHeight
            ),
            scrollWidth: Math.max(
              document.body.scrollWidth,
              document.documentElement.scrollWidth
            ),
            viewportHeight: window.innerHeight,
            viewportWidth: window.innerWidth,
            originalScrollY: window.scrollY
          });
          break;

        case 'SCROLL_TO':
          // Scroll pour capture page entière
          window.scrollTo(0, msg.scrollY);
          sendResponse({ success: true });
          break;

        case 'CAPTURE_FULL_PAGE_DOM':
          // Capture page entière via html2canvas
          this.captureFullPageDOM().then(result => {
            sendResponse(result);
          }).catch(error => {
            sendResponse({ success: false, error: error.message });
          });
          return true; // Indique réponse asynchrone

        case 'enableFeedbackMode':
          console.log('[SUMO V3] 🎨 Received enableFeedbackMode message');
          console.log('[SUMO V3] window.FeedbackOverlay exists?', !!window.FeedbackOverlay);

          if (window.FeedbackOverlay) {
            console.log('[SUMO V3] Calling FeedbackOverlay.activate()...');
            try {
              window.FeedbackOverlay.activate();
              console.log('[SUMO V3] ✅ FeedbackOverlay activated successfully');
              sendResponse({ success: true });
            } catch (error) {
              console.error('[SUMO V3] ❌ Error activating FeedbackOverlay:', error);
              sendResponse({ success: false, error: error.message });
            }
          } else {
            console.error('[SUMO V3] ❌ FeedbackOverlay not found in window object!');
            console.log('[SUMO V3] Available window properties:', Object.keys(window).filter(k => k.includes('Feedback') || k.includes('sumo')));
            sendResponse({ success: false, error: 'FeedbackOverlay not loaded' });
          }
          break;

        default:
          sendResponse({ success: false, error: 'Unknown action' });
      }
    }
    
    startTracking(sessionId) {
      if (this.isTracking) {
        console.log('[SUMO V3] Already tracking');
        return;
      }

      this.isTracking = true;
      this.sessionId = sessionId;
      this.events = [];
      this.screenshots = [];

      // Logger
      if (window.SUMO_DEBUG) {
        window.SUMO_DEBUG.setSessionId(sessionId);
        window.SUMO_DEBUG.log('START', 'Tracking started', { sessionId, url: window.location.href });
      }

      // Afficher l'indicateur REC
      this.showRecordingIndicator();

      // Démarrer tous les trackers
      Object.values(this.trackers).forEach(tracker => {
        if (tracker && tracker.start) {
          tracker.start();
        }
      });

      console.log('[SUMO V3] 🎬 Tracking started -', sessionId);
    }
    
    stopTracking() {
      if (!this.isTracking) {
        console.log('[SUMO V3] Not tracking');
        return;
      }

      // Masquer l'indicateur REC
      this.hideRecordingIndicator();

      // Arrêter tous les trackers
      Object.values(this.trackers).forEach(tracker => {
        if (tracker && tracker.stop) {
          tracker.stop();
        }
      });

      // IMPORTANT: Envoyer les events finaux au background avant de s'arrêter
      if (this.events.length > 0) {
        console.log('[SUMO V3] Sending final events to background...');
        this.sendEventsToBackground();
      }

      this.isTracking = false;
      this.isPaused = false;  // Réinitialiser aussi l'état de pause

      console.log('[SUMO V3] ⏹️ Tracking stopped');
      console.log('[SUMO V3] Captured:', this.events.length, 'events,', this.screenshots.length, 'screenshots');
    }

    /**
     * Met le tracking en pause (pour mode feedback)
     */
    pauseTracking() {
      console.log('[SUMO V3] 🔧 pauseTracking() called, isTracking:', this.isTracking, 'isPaused:', this.isPaused);

      if (!this.isTracking) {
        console.log('[SUMO V3] ⚠️ Cannot pause: not tracking');
        return;
      }

      if (this.isPaused) {
        console.log('[SUMO V3] ⚠️ Already paused');
        return;
      }

      try {
        console.log('[SUMO V3] ⏸️ Tracking paused (for feedback mode)');

        // Logger
        if (window.SUMO_DEBUG) {
          window.SUMO_DEBUG.log('PAUSE', 'Tracking paused for feedback', {
            isTracking: this.isTracking,
            isPaused: this.isPaused,
            events: this.events.length,
            screenshots: this.screenshots.length
          });
        }

        this.isPaused = true;  // IMPORTANT: marquer comme en pause

        // Arrêter temporairement tous les trackers
        Object.values(this.trackers).forEach(tracker => {
          if (tracker && tracker.stop) {
            try {
              console.log('[SUMO V3] Stopping tracker:', tracker.constructor.name);
              tracker.stop();
            } catch (e) {
              console.error('[SUMO V3] Error stopping tracker:', e);
              if (window.SUMO_DEBUG) {
                window.SUMO_DEBUG.log('ERROR', `Error stopping tracker: ${tracker.constructor.name}`, { error: e.message });
              }
            }
          }
        });

        // Masquer l'indicateur REC temporairement
        if (this.recordingIndicator) {
          this.recordingIndicator.hide();
        }

        console.log('[SUMO V3] ✅ Pause complete');
        if (window.SUMO_DEBUG) {
          window.SUMO_DEBUG.log('PAUSE', 'Pause complete', { isPaused: this.isPaused });
        }
      } catch (error) {
        console.error('[SUMO V3] ❌ Error in pauseTracking:', error);
        if (window.SUMO_DEBUG) {
          window.SUMO_DEBUG.log('CRASH', 'CRASH in pauseTracking', { error: error.message, stack: error.stack });
        }
      }
    }

    /**
     * Reprend le tracking (après mode feedback)
     */
    resumeTracking() {
      console.log('[SUMO V3] 🔧 resumeTracking() called, isTracking:', this.isTracking, 'isPaused:', this.isPaused);

      if (!this.isTracking) {
        console.log('[SUMO V3] ⚠️ Cannot resume: session is not active');
        return;
      }

      if (!this.isPaused) {
        console.log('[SUMO V3] ⚠️ Not paused, nothing to resume');
        return;
      }

      try {
        console.log('[SUMO V3] ▶️ Tracking resumed (after feedback mode)');

        // Logger
        if (window.SUMO_DEBUG) {
          window.SUMO_DEBUG.log('RESUME', 'Tracking resumed after feedback', {
            isTracking: this.isTracking,
            isPaused: this.isPaused,
            events: this.events.length
          });
        }

        this.isPaused = false;  // IMPORTANT: désactiver l'état de pause

        // Redémarrer tous les trackers
        Object.values(this.trackers).forEach(tracker => {
          if (tracker && tracker.start) {
            try {
              console.log('[SUMO V3] Starting tracker:', tracker.constructor.name);
              tracker.start();
            } catch (e) {
              console.error('[SUMO V3] Error starting tracker:', e);
              if (window.SUMO_DEBUG) {
                window.SUMO_DEBUG.log('ERROR', `Error starting tracker: ${tracker.constructor.name}`, { error: e.message });
              }
            }
          }
        });

        // Réafficher l'indicateur REC
        if (this.recordingIndicator) {
          this.recordingIndicator.show();
        }

        console.log('[SUMO V3] ✅ Resume complete');
        if (window.SUMO_DEBUG) {
          window.SUMO_DEBUG.log('RESUME', 'Resume complete', { isPaused: this.isPaused, isTracking: this.isTracking });
        }
      } catch (error) {
        console.error('[SUMO V3] ❌ Error in resumeTracking:', error);
        if (window.SUMO_DEBUG) {
          window.SUMO_DEBUG.log('CRASH', 'CRASH in resumeTracking', { error: error.message, stack: error.stack });
        }
      }
    }
    
    handleEvent(event) {
      if (!this.isTracking) {
        console.log('[SUMO V3] ⚠️ Event ignored: not tracking', event.type);
        return;
      }
      if (this.isPaused) {
        console.log('[SUMO V3] ⚠️ Event ignored: paused', event.type);
        return;
      }

      // Gérer types spéciaux
      if (event.type === 'screenshot_request') {
        // Demander screenshot au handler
        this.trackers.screenshot.requestCapture(event.trigger, {
          relatedEventId: event.relatedEventId,
          pattern: event.pattern,
          intensity: event.intensity
        });
        return;
      }
      
      if (event.type === 'screenshot_captured') {
        // Stocker screenshot
        this.screenshots.push(event);
        
        // Envoyer au background pour stockage
        chrome.runtime.sendMessage({
          action: 'STORE_SCREENSHOT',
          screenshot: event
        });
        
        return;
      }
      
      if (event.type === 'click') {
        // Analyser patterns
        this.trackers.patterns.analyzeClick(event);
      }
      
      // Stocker event
      this.events.push(event);
      console.log(`[SUMO V3] 📝 Event captured: ${event.type}, total: ${this.events.length}`);

      // Envoyer au background périodiquement
      if (this.events.length % 10 === 0) {
        console.log(`[SUMO V3] 📤 Sending ${this.events.length} events to background`);
        this.sendEventsToBackground();
      }
    }
    
    sendEventsToBackground() {
      if (this.events.length === 0) return;
      
      chrome.runtime.sendMessage({
        action: 'STORE_EVENTS',
        events: [...this.events]
      });
      
      // Garder en mémoire aussi (ne pas vider)
    }
    
    async notifyReady() {
      try {
        await chrome.runtime.sendMessage({
          action: 'CONTENT_READY',
          url: window.location.href
        });
      } catch (error) {
        // Background pas encore prêt, retry
        setTimeout(() => this.notifyReady(), 1000);
      }
    }

    /**
     * Capture page entière via html2canvas
     */
    async captureFullPageDOM() {
      console.log('[SUMO V3] 📸 Starting html2canvas capture...');

      if (typeof html2canvas === 'undefined') {
        throw new Error('html2canvas not loaded');
      }

      const scrollHeight = Math.max(
        document.body.scrollHeight,
        document.documentElement.scrollHeight
      );
      const scrollWidth = Math.max(
        document.body.scrollWidth,
        document.documentElement.scrollWidth
      );

      console.log(`[SUMO V3] Capturing page: ${scrollWidth}x${scrollHeight}`);

      // Capturer la page entière - Options ultra-optimisées pour performance
      const canvas = await html2canvas(document.body, {
        scrollY: -window.scrollY,
        scrollX: -window.scrollX,
        windowWidth: scrollWidth,
        windowHeight: scrollHeight,
        width: scrollWidth,
        height: scrollHeight,
        useCORS: true,
        allowTaint: true,
        logging: false,
        scale: 0.35, // Résolution très réduite pour max performance
        imageTimeout: 3000, // Timeout court pour images externes
        removeContainer: true,
        foreignObjectRendering: false,
        ignoreElements: (element) => {
          // Ignorer les éléments lourds non essentiels
          return element.tagName === 'IFRAME' ||
                 element.tagName === 'VIDEO' ||
                 element.classList?.contains('sumo-ignore');
        }
      });

      // Convertir en dataUrl - qualité réduite pour légèreté
      const dataUrl = canvas.toDataURL('image/jpeg', 0.55);

      console.log('[SUMO V3] ✅ html2canvas capture complete');

      return {
        success: true,
        dataUrl,
        dimensions: {
          width: scrollWidth,
          height: scrollHeight
        }
      };
    }

    /**
     * Vérifie si une session est active au démarrage (cas de navigation pendant enregistrement)
     */
    async checkActiveSession() {
      try {
        console.log('[SUMO V3] 🔍 Vérification session active...');

        const response = await chrome.runtime.sendMessage({
          action: 'GET_SESSION_STATUS'
        });

        console.log('[SUMO V3] Response session status:', response);

        if (response?.isActive && response?.session) {
          console.log('[SUMO V3] 🔄 Session active détectée, reprise du tracking...');
          console.log('[SUMO V3] Session ID:', response.session.sessionId);

          this.sessionId = response.session.sessionId;
          this.isTracking = true;
          this.isPaused = false;

          // Démarrer les trackers
          Object.values(this.trackers).forEach(tracker => {
            if (tracker && tracker.start) {
              console.log('[SUMO V3] Starting tracker:', tracker.constructor.name);
              tracker.start();
            }
          });

          // Afficher indicateur REC
          this.showRecordingIndicator();

          // Capturer screenshot de la nouvelle page
          if (this.trackers.screenshot && window.SUMO_CONSTANTS) {
            this.trackers.screenshot.captureScreenshot(
              window.SUMO_CONSTANTS.SCREENSHOT_TRIGGERS.NAVIGATION
            );
          }

          console.log('[SUMO V3] ✅ Tracking repris sur nouvelle page');
        } else {
          console.log('[SUMO V3] Aucune session active détectée');
        }
      } catch (error) {
        console.error('[SUMO V3] ❌ Erreur vérification session active:', error);
      }
    }

    /**
     * Affiche l'indicateur REC sur la page
     */
    showRecordingIndicator() {
      if (!this.recordingIndicator) {
        this.recordingIndicator = new RecordingIndicator();
      }
      this.recordingIndicator.show();
    }

    /**
     * Masque l'indicateur REC
     */
    hideRecordingIndicator() {
      if (this.recordingIndicator) {
        this.recordingIndicator.hide();
      }
    }
  }
  
  // Créer instance globale
  window.sumoContent = new SUMOContent();
  
  // Méthodes debug
  window.sumoContent.debug = {
    getStats: () => {
      const stats = {
        isTracking: window.sumoContent.isTracking,
        sessionId: window.sumoContent.sessionId,
        events: window.sumoContent.events.length,
        screenshots: window.sumoContent.screenshots.length,
        trackers: {}
      };
      
      Object.entries(window.sumoContent.trackers).forEach(([name, tracker]) => {
        if (tracker?.getStats) {
          stats.trackers[name] = tracker.getStats();
        }
      });
      
      return stats;
    },
    
    getEvents: () => window.sumoContent.events,
    getScreenshots: () => window.sumoContent.screenshots
  };
  
  console.log('[SUMO V3] 🚀 Extension loaded - Use window.sumoContent.debug.getStats()');
}
