/**
 * SUMO UX Tracker V4.1 - Background Service Worker
 * Gère sessions, screenshots et export
 *
 * V4.1 Améliorations:
 * - IndexedDB pour screenshots (pas de perte RAM sur sessions longues)
 * - Queue d'écriture pour éviter les problèmes de concurrence
 * - Protection contre le démarrage multiple de sessions
 */

// Import JSZip pour création de fichiers .uxtrack
importScripts('../libs/jszip.min.js');

class SUMOBackground {
  constructor() {
    this.currentSession = null;
    this.isRecording = false;
    this.sessions = new Map();
    this.db = null;
    this.dbName = 'SUMO_Screenshots_V4';
    this.dbVersion = 1;

    // Queue pour les écritures IndexedDB
    this.writeQueue = [];
    this.isProcessingQueue = false;

    this.init();
  }

  // ===== QUEUE D'ÉCRITURE POUR INDEXEDDB =====

  async processWriteQueue() {
    if (this.isProcessingQueue || this.writeQueue.length === 0) return;

    this.isProcessingQueue = true;

    while (this.writeQueue.length > 0) {
      const { screenshot, sessionId, resolve, reject } = this.writeQueue.shift();

      try {
        const db = await this.openDB();

        await new Promise((res, rej) => {
          const transaction = db.transaction(['screenshots'], 'readwrite');
          const store = transaction.objectStore('screenshots');

          const record = {
            sessionId,
            timestamp: screenshot.timestamp,
            trigger: screenshot.trigger,
            url: screenshot.url,
            viewport: screenshot.viewport,
            relatedEventId: screenshot.relatedEventId,
            dataUrl: screenshot.dataUrl
          };

          const request = store.add(record);
          request.onsuccess = () => res(request.result);
          request.onerror = () => rej(request.error);
        });

        resolve();
      } catch (error) {
        console.error('[Background] Queue write error:', error);
        reject(error);
      }
    }

    this.isProcessingQueue = false;
  }

  queueScreenshotWrite(sessionId, screenshot) {
    return new Promise((resolve, reject) => {
      this.writeQueue.push({ screenshot, sessionId, resolve, reject });
      this.processWriteQueue();
    });
  }

  // ===== INDEXEDDB POUR SCREENSHOTS (évite la saturation RAM) =====

  async openDB() {
    if (this.db) return this.db;

    return new Promise((resolve, reject) => {
      const request = indexedDB.open(this.dbName, this.dbVersion);

      request.onerror = () => {
        console.error('[Background] IndexedDB error:', request.error);
        reject(request.error);
      };

      request.onsuccess = () => {
        this.db = request.result;
        console.log('[Background] 📦 IndexedDB opened');
        resolve(this.db);
      };

      request.onupgradeneeded = (event) => {
        const db = event.target.result;

        // Store pour les screenshots
        if (!db.objectStoreNames.contains('screenshots')) {
          const store = db.createObjectStore('screenshots', { keyPath: 'id', autoIncrement: true });
          store.createIndex('sessionId', 'sessionId', { unique: false });
          store.createIndex('timestamp', 'timestamp', { unique: false });
          console.log('[Background] 📦 IndexedDB store created');
        }
      };
    });
  }

  async saveScreenshotToDB(sessionId, screenshot) {
    try {
      const db = await this.openDB();

      return new Promise((resolve, reject) => {
        const transaction = db.transaction(['screenshots'], 'readwrite');
        const store = transaction.objectStore('screenshots');

        const record = {
          sessionId,
          timestamp: screenshot.timestamp,
          trigger: screenshot.trigger,
          url: screenshot.url,
          viewport: screenshot.viewport,
          relatedEventId: screenshot.relatedEventId,
          dataUrl: screenshot.dataUrl  // L'image complète
        };

        const request = store.add(record);

        request.onsuccess = () => {
          resolve(request.result);
        };

        request.onerror = () => {
          console.error('[Background] Error saving screenshot to DB:', request.error);
          reject(request.error);
        };
      });
    } catch (error) {
      console.error('[Background] IndexedDB save error:', error);
      throw error;
    }
  }

  async getScreenshotsFromDB(sessionId) {
    try {
      const db = await this.openDB();

      return new Promise((resolve, reject) => {
        const transaction = db.transaction(['screenshots'], 'readonly');
        const store = transaction.objectStore('screenshots');
        const index = store.index('sessionId');
        const request = index.getAll(sessionId);

        request.onsuccess = () => {
          const screenshots = request.result || [];
          // Trier par timestamp
          screenshots.sort((a, b) => a.timestamp - b.timestamp);
          console.log(`[Background] 📦 Retrieved ${screenshots.length} screenshots from IndexedDB`);
          resolve(screenshots);
        };

        request.onerror = () => {
          console.error('[Background] Error reading screenshots from DB:', request.error);
          reject(request.error);
        };
      });
    } catch (error) {
      console.error('[Background] IndexedDB read error:', error);
      return [];
    }
  }

  async clearScreenshotsFromDB(sessionId = null) {
    try {
      const db = await this.openDB();

      return new Promise((resolve, reject) => {
        const transaction = db.transaction(['screenshots'], 'readwrite');
        const store = transaction.objectStore('screenshots');

        if (sessionId) {
          // Supprimer uniquement les screenshots de cette session
          const index = store.index('sessionId');
          const request = index.openCursor(IDBKeyRange.only(sessionId));

          request.onsuccess = (event) => {
            const cursor = event.target.result;
            if (cursor) {
              cursor.delete();
              cursor.continue();
            }
          };

          transaction.oncomplete = () => {
            console.log(`[Background] 📦 Cleared screenshots for session ${sessionId}`);
            resolve();
          };
        } else {
          // Supprimer tous les screenshots
          const request = store.clear();
          request.onsuccess = () => {
            console.log('[Background] 📦 Cleared all screenshots from IndexedDB');
            resolve();
          };
          request.onerror = () => reject(request.error);
        }
      });
    } catch (error) {
      console.error('[Background] IndexedDB clear error:', error);
    }
  }

  async countScreenshotsInDB(sessionId) {
    try {
      const db = await this.openDB();

      return new Promise((resolve, reject) => {
        const transaction = db.transaction(['screenshots'], 'readonly');
        const store = transaction.objectStore('screenshots');
        const index = store.index('sessionId');
        const request = index.count(sessionId);

        request.onsuccess = () => resolve(request.result);
        request.onerror = () => reject(request.error);
      });
    } catch (error) {
      return 0;
    }
  }
  
  async init() {
    console.log('[SUMO Background V4.1] Starting...');

    // IMPORTANT: Restaurer l'état depuis chrome.storage (service worker peut se réveiller après veille)
    await this.restoreState();

    // Écouter messages
    chrome.runtime.onMessage.addListener((msg, sender, sendResponse) => {
      this.handleMessage(msg, sender, sendResponse);
      return true;
    });

    // Écouter changements de tabs
    chrome.tabs.onActivated.addListener((activeInfo) => {
      this.handleTabActivated(activeInfo);
    });

    console.log('[SUMO Background V4.1] Ready');
  }

  /**
   * Restaure l'état depuis chrome.storage (au cas où le service worker se réveille après veille)
   */
  async restoreState() {
    try {
      const result = await chrome.storage.local.get(['currentSession', 'isRecording']);

      if (result.currentSession) {
        this.currentSession = result.currentSession;
        console.log('[Background] 🔄 Session restaurée depuis storage:', this.currentSession.sessionId);
      }

      if (result.isRecording !== undefined) {
        this.isRecording = result.isRecording;
        console.log('[Background] 🔄 isRecording restauré:', this.isRecording);
      }

      // Restaurer le badge si on est en cours d'enregistrement
      if (this.isRecording && this.currentSession) {
        this.updateBadge(true);
        console.log('[Background] 🔴 Badge REC restauré');
      }
    } catch (error) {
      console.error('[Background] Erreur restauration état:', error);
    }
  }

  /**
   * Sauvegarde l'état dans chrome.storage (pour survivre au réveil du service worker)
   */
  async saveState() {
    try {
      await chrome.storage.local.set({
        currentSession: this.currentSession,
        isRecording: this.isRecording
      });
      console.log('[Background] 💾 État sauvegardé: isRecording =', this.isRecording);
    } catch (error) {
      console.error('[Background] Erreur sauvegarde état:', error);
    }
  }
  
  /**
   * S'assure que le content script est chargé sur le tab
   */
  async ensureContentScript(tabId) {
    try {
      // Tester si content script répond
      await chrome.tabs.sendMessage(tabId, { action: 'PING' });
      return true;
    } catch (error) {
      // Content script pas prêt
      console.log('[Background] Content script not responding, waiting...');
      
      // Attendre 2 secondes et réessayer
      await new Promise(resolve => setTimeout(resolve, 2000));
      
      try {
        await chrome.tabs.sendMessage(tabId, { action: 'PING' });
        return true;
      } catch (retryError) {
        console.error('[Background] Content script not available after retry');
        return false;
      }
    }
  }
  
  async handleMessage(msg, sender, sendResponse) {
    try {
      switch (msg.action) {
        case 'CONTENT_READY':
          console.log('[Background] Content ready on:', msg.url);
          sendResponse({ success: true });
          break;
          
        case 'START_SESSION':
          // Récupérer tab active si pas fournie
          let tab = sender.tab;
          if (!tab && msg.tabId) {
            tab = await chrome.tabs.get(msg.tabId);
          }
          if (!tab) {
            // Fallback: tab active
            const tabs = await chrome.tabs.query({ active: true, currentWindow: true });
            tab = tabs[0];
          }

          // Récupérer le mode de capture (viewport ou fullpage)
          const captureMode = msg.captureMode || 'viewport';
          console.log('[Background] Capture mode:', captureMode);

          const session = await this.startSession(tab, captureMode);
          sendResponse({ success: true, session });
          break;
          
        case 'STOP_SESSION':
          console.log('[Background] 🛑 STOP_SESSION requested');
          await this.stopSession();
          sendResponse({ success: true });
          break;
          
        case 'GET_SESSION_STATUS':
          console.log('[Background] 📊 GET_SESSION_STATUS - isRecording:', this.isRecording, 'sessionId:', this.currentSession?.sessionId);
          sendResponse({
            success: true,
            isActive: this.isRecording,  // Utiliser le flag isRecording au lieu de !!this.currentSession
            session: this.currentSession
          });
          break;
          
        case 'STORE_EVENTS':
          this.storeEvents(msg.events);
          sendResponse({ success: true });
          break;

        case 'STORE_SCREENSHOT':
          this.storeScreenshot(msg.screenshot);
          sendResponse({ success: true });
          break;

        case 'saveFeedback':
          this.storeFeedback(msg.feedback);
          sendResponse({ success: true });
          break;
          
        case 'CAPTURE_SCREENSHOT':
          const result = await this.captureScreenshot(sender.tab.id, msg.data);
          sendResponse(result);
          break;
          
        case 'EXPORT_SESSION':
          const exportData = await this.exportSession();
          sendResponse(exportData);
          break;
          
        default:
          sendResponse({ success: false, error: 'Unknown action' });
      }
    } catch (error) {
      console.error('[Background] Error:', error);
      sendResponse({ success: false, error: error.message });
    }
  }
  
  async startSession(tab, captureMode = 'viewport') {
    if (!tab) {
      throw new Error('No tab provided');
    }

    // PROTECTION: Ne pas démarrer une nouvelle session si une est déjà en cours
    if (this.isRecording && this.currentSession) {
      console.warn('[Background] ⚠️ Session already active! Ignoring START_SESSION request.');
      console.log('[Background] Current session:', this.currentSession.sessionId);
      return this.currentSession;
    }

    const sessionId = `sumo_${Date.now()}_${Math.random().toString(36).substr(2, 9)}`;

    // Nettoyer les anciens screenshots dans IndexedDB avant de commencer
    await this.clearScreenshotsFromDB();

    this.currentSession = {
      sessionId,
      startTime: Date.now(),
      url: tab.url || 'unknown',
      title: tab.title || 'Unknown Page',
      tabId: tab.id,
      captureMode: captureMode, // 'viewport' ou 'fullpage'
      events: [],
      screenshots: []
    };

    console.log('[Background] Session capture mode:', captureMode);

    this.isRecording = true;  // IMPORTANT: marquer comme en cours d'enregistrement

    // Sauvegarder état dans chrome.storage (pour survivre au réveil du service worker)
    await this.saveState();

    // Sauvegarder dans Map aussi
    this.sessions.set(sessionId, this.currentSession);

    // Vérifier que content script est prêt
    const isReady = await this.ensureContentScript(tab.id);
    
    if (!isReady) {
      throw new Error('Content script not ready. Please refresh the page and try again.');
    }
    
    // Notifier content script de démarrer
    await chrome.tabs.sendMessage(tab.id, {
      action: 'START_TRACKING',
      sessionId
    });

    // Activer badge REC sur l'icône
    this.updateBadge(true);

    console.log('[Background] Session started:', sessionId);
    return this.currentSession;
  }
  
  async stopSession() {
    if (!this.currentSession) {
      console.log('[Background] No active session');
      return;
    }
    
    const sessionId = this.currentSession.sessionId;
    
    // Désactiver badge REC
    this.updateBadge(false);

    // Notifier content script
    try {
      await chrome.tabs.sendMessage(this.currentSession.tabId, {
        action: 'STOP_TRACKING'
      });
      
      // Récupérer données finales
      const response = await chrome.tabs.sendMessage(this.currentSession.tabId, {
        action: 'GET_DATA'
      });

      if (response?.success) {
        // Fusionner les events finaux (il peut y avoir des events non encore envoyés)
        this.storeEvents(response.events);
        console.log('[Background] Merged final events from content script');

        // NE PAS écraser screenshots - ils contiennent déjà les dataUrl stockés via STORE_SCREENSHOT
        // Les screenshots du content script n'ont que les metadata, pas les images
        console.log('[Background] Screenshots already stored with dataUrl, keeping existing ones');
      }
      
    } catch (error) {
      console.warn('[Background] Could not get final data:', error);
    }
    
    this.currentSession.endTime = Date.now();
    this.isRecording = false;  // IMPORTANT: marquer comme arrêté

    // Sauvegarder état dans chrome.storage
    await this.saveState();

    console.log('[Background] Session stopped:', sessionId);
    console.log('[Background] Captured:',
      this.currentSession.events.length, 'events,',
      this.currentSession.screenshots.length, 'screenshots'
    );

    // Garder en mémoire pour export (ne pas set this.currentSession = null)
    return this.currentSession;
  }
  
  storeEvents(events) {
    // Accepter events tant qu'une session existe (même si isRecording = false pendant le stop)
    if (!this.currentSession) {
      console.log('[Background] No session, rejecting events');
      return;
    }

    let newEvents = 0;
    // Fusionner avec events existants
    events.forEach(event => {
      if (!this.currentSession.events.find(e => e.id === event.id && e.timestamp === event.timestamp)) {
        this.currentSession.events.push(event);
        newEvents++;
      }
    });

    if (newEvents > 0) {
      console.log(`[Background] Stored ${newEvents} new events, total: ${this.currentSession.events.length}`);
    }
  }
  
  async storeScreenshot(screenshot) {
    if (!this.currentSession) return;

    const sessionId = this.currentSession.sessionId;

    // Utiliser la queue pour éviter les problèmes de concurrence
    try {
      await this.queueScreenshotWrite(sessionId, screenshot);

      // Ne garder que les métadonnées en RAM (pas le dataUrl)
      this.currentSession.screenshots.push({
        timestamp: screenshot.timestamp,
        trigger: screenshot.trigger,
        url: screenshot.url,
        viewport: screenshot.viewport,
        relatedEventId: screenshot.relatedEventId
        // dataUrl stocké dans IndexedDB uniquement
      });

      const count = this.currentSession.screenshots.length;
      console.log(`[Background V4.1] 📦 Screenshot queued to IndexedDB, total: ${count}`);
    } catch (error) {
      console.error('[Background] Failed to store screenshot in IndexedDB:', error);
      // Fallback: stocker en RAM (ancien comportement)
      this.currentSession.screenshots.push(screenshot);
      console.log(`[Background] Screenshot stored in RAM (fallback), total: ${this.currentSession.screenshots.length}`);
    }
  }

  /**
   * Stocke un feedback (annotations utilisateur)
   */
  storeFeedback(feedback) {
    if (!this.currentSession) return;

    // Initialiser le tableau feedbacks si nécessaire
    if (!this.currentSession.feedbacks) {
      this.currentSession.feedbacks = [];
    }

    this.currentSession.feedbacks.push(feedback);

    console.log('[Background] Feedback sauvegardé:', feedback.feedbackId);
    console.log(`[Background] Total feedbacks: ${this.currentSession.feedbacks.length}`);
  }

  async captureScreenshot(tabId) {
    // Vérifier le mode de capture de la session
    const captureMode = this.currentSession?.captureMode || 'viewport';

    // Mode viewport: capture simple et rapide
    if (captureMode === 'viewport') {
      try {
        console.log('[Background] 📸 Viewport capture...');
        const dataUrl = await chrome.tabs.captureVisibleTab(null, {
          format: 'jpeg',
          quality: 85
        });
        return { success: true, dataUrl, fullPage: false };
      } catch (error) {
        console.error('[Background] Viewport capture failed:', error);
        return { success: false, error: error.message };
      }
    }

    // Mode fullpage: capture via html2canvas
    try {
      console.log('[Background] 📸 Full page DOM capture...');

      const result = await chrome.tabs.sendMessage(tabId, {
        action: 'CAPTURE_FULL_PAGE_DOM'
      });

      if (result?.success) {
        console.log('[Background] ✅ DOM capture complete');
        return {
          success: true,
          dataUrl: result.dataUrl,
          fullPage: true,
          dimensions: result.dimensions
        };
      } else {
        throw new Error(result?.error || 'DOM capture failed');
      }
    } catch (error) {
      console.error('[Background] DOM capture failed:', error);

      // Fallback: capture simple viewport
      console.log('[Background] Falling back to simple viewport capture');
      try {
        const dataUrl = await chrome.tabs.captureVisibleTab(null, {
          format: 'jpeg',
          quality: 85
        });
        return { success: true, dataUrl, fullPage: false };
      } catch (e) {
        return { success: false, error: error.message };
      }
    }
  }

  async captureFullPage(tabId) {
    try {
      console.log('[Background] 📸 Starting full page capture...');

      // 1. Obtenir les dimensions de la page
      const pageInfo = await chrome.tabs.sendMessage(tabId, {
        action: 'GET_PAGE_INFO'
      });

      if (!pageInfo || !pageInfo.success) {
        throw new Error('Could not get page info');
      }

      const { scrollHeight, scrollWidth, viewportHeight, viewportWidth, originalScrollY } = pageInfo;

      console.log(`[Background] Page: ${scrollWidth}x${scrollHeight}, Viewport: ${viewportWidth}x${viewportHeight}`);

      // 2. Calculer le nombre de captures nécessaires
      const numCaptures = Math.ceil(scrollHeight / viewportHeight);
      const captures = [];

      console.log(`[Background] Will capture ${numCaptures} viewports`);

      // 3. Scroll et capturer chaque section
      for (let i = 0; i < numCaptures; i++) {
        const scrollY = i * viewportHeight;

        // Demander au content script de scroller
        await chrome.tabs.sendMessage(tabId, {
          action: 'SCROLL_TO',
          scrollY: scrollY
        });

        // Attendre que le scroll soit effectué et le rendu stabilisé
        // IMPORTANT: Chrome limite à 2 captures/seconde, donc minimum 600ms entre chaque
        await new Promise(resolve => setTimeout(resolve, 600));

        // Capturer le viewport visible
        const dataUrl = await chrome.tabs.captureVisibleTab(null, {
          format: 'jpeg',
          quality: 85
        });

        captures.push({
          dataUrl,
          scrollY,
          isLast: i === numCaptures - 1,
          // Pour la dernière capture, on peut avoir un overlap
          cropHeight: (i === numCaptures - 1) ? (scrollHeight - scrollY) : viewportHeight
        });

        console.log(`[Background] Captured viewport ${i + 1}/${numCaptures}`);
      }

      // 4. Restaurer la position de scroll originale
      await chrome.tabs.sendMessage(tabId, {
        action: 'SCROLL_TO',
        scrollY: originalScrollY
      });

      // 5. Assembler les captures
      const assembledImage = await this.assembleCaptures(captures, scrollWidth, scrollHeight, viewportHeight);

      console.log('[Background] ✅ Full page capture complete');

      return {
        success: true,
        dataUrl: assembledImage,
        fullPage: true,
        dimensions: {
          width: scrollWidth,
          height: scrollHeight
        }
      };

    } catch (error) {
      console.error('[Background] Full page capture failed:', error);

      // Fallback: capture simple du viewport
      console.log('[Background] Falling back to simple viewport capture');
      const dataUrl = await chrome.tabs.captureVisibleTab(null, {
        format: 'jpeg',
        quality: 85
      });

      return {
        success: true,
        dataUrl: dataUrl,
        fullPage: false
      };
    }
  }

  async assembleCaptures(captures, totalWidth, totalHeight, viewportHeight) {
    // Créer un canvas pour assembler les images
    const canvas = new OffscreenCanvas(totalWidth, totalHeight);
    const ctx = canvas.getContext('2d');

    // Charger et dessiner chaque capture
    for (let i = 0; i < captures.length; i++) {
      const capture = captures[i];

      // Convertir dataUrl en ImageBitmap
      const response = await fetch(capture.dataUrl);
      const blob = await response.blob();
      const imageBitmap = await createImageBitmap(blob);

      // Position Y où dessiner cette capture
      const y = capture.scrollY;

      // Pour la dernière capture, on peut avoir besoin de ne prendre qu'une partie
      if (capture.isLast && capture.cropHeight < viewportHeight) {
        // Dessiner seulement la partie nécessaire de la dernière capture
        const sourceY = viewportHeight - capture.cropHeight;
        ctx.drawImage(
          imageBitmap,
          0, sourceY, totalWidth, capture.cropHeight,  // source
          0, y, totalWidth, capture.cropHeight          // destination
        );
      } else {
        ctx.drawImage(imageBitmap, 0, y);
      }
    }

    // Convertir le canvas en dataUrl
    const blob = await canvas.convertToBlob({ type: 'image/jpeg', quality: 0.85 });
    const reader = new FileReader();

    return new Promise((resolve) => {
      reader.onloadend = () => resolve(reader.result);
      reader.readAsDataURL(blob);
    });
  }
  
  async exportSession() {
    if (!this.currentSession) {
      return { success: false, error: 'No active session' };
    }

    const session = this.currentSession;

    try {
      console.log('[Background] Creating ZIP export...');

      // Créer instance JSZip
      const zip = new JSZip();

      // Préparer données session (sans dataUrl dans screenshots)
      const sessionData = {
        session: {
          sessionId: session.sessionId,
          url: session.url,
          title: session.title,
          startTime: session.startTime,
          endTime: session.endTime || Date.now(),
          duration: (session.endTime || Date.now()) - session.startTime
        },
        events: session.events.map((event, index) => ({
          ...event,
          id: event.id || index + 1
        })),
        screenshots: session.screenshots.map((screenshot, index) => ({
          id: screenshot.id || index + 1,
          filename: `screenshot-${screenshot.id || index + 1}-${screenshot.timestamp}.jpg`,
          trigger: screenshot.trigger,
          timestamp: screenshot.timestamp,
          url: screenshot.url,
          viewport: screenshot.viewport,
          relatedEventId: screenshot.relatedEventId
          // dataUrl retiré - sera dans dossier screenshots/
        })),
        feedbacks: session.feedbacks || [], // Inclure les feedbacks
        stats: {
          totalEvents: session.events.length,
          totalScreenshots: session.screenshots.length,
          totalClicks: session.events.filter(e => e.type === 'click').length,
          totalNavigations: session.events.filter(e => e.type === 'navigation').length,
          totalFeedbacks: (session.feedbacks || []).length
        },
        version: '4.1.0'
      };

      // Ajouter session-data.json au ZIP
      zip.file('session-data.json', JSON.stringify(sessionData, null, 2));

      // Créer dossier screenshots et ajouter chaque image
      const screenshotsFolder = zip.folder('screenshots');

      // Récupérer les screenshots depuis IndexedDB
      const screenshotsFromDB = await this.getScreenshotsFromDB(session.sessionId);
      console.log(`[Background] 📦 Loading ${screenshotsFromDB.length} screenshots from IndexedDB for export`);

      let screenshotsExported = 0;
      for (let i = 0; i < screenshotsFromDB.length; i++) {
        const screenshot = screenshotsFromDB[i];
        const filename = `screenshot-${i + 1}-${screenshot.timestamp}.jpg`;

        // Convertir dataUrl en blob
        if (screenshot.dataUrl) {
          // Retirer préfixe "data:image/jpeg;base64,"
          const base64Data = screenshot.dataUrl.replace(/^data:image\/jpeg;base64,/, '');

          // Ajouter image au dossier screenshots
          screenshotsFolder.file(filename, base64Data, { base64: true });
          screenshotsExported++;
        } else {
          console.warn(`[Background] Screenshot #${i + 1} has no dataUrl!`);
        }
      }

      // Mettre à jour les métadonnées avec les bons IDs
      sessionData.screenshots = screenshotsFromDB.map((screenshot, index) => ({
        id: index + 1,
        filename: `screenshot-${index + 1}-${screenshot.timestamp}.jpg`,
        trigger: screenshot.trigger,
        timestamp: screenshot.timestamp,
        url: screenshot.url,
        viewport: screenshot.viewport,
        relatedEventId: screenshot.relatedEventId
      }));

      sessionData.stats.totalScreenshots = screenshotsFromDB.length;

      console.log(`[Background] Exported ${screenshotsExported}/${screenshotsFromDB.length} screenshots to ZIP`);

      console.log(`[Background] Exported ${session.events.length} events`);
      console.log('[Background] ZIP structure created, generating blob...');

      // Générer blob ZIP
      const zipBlob = await zip.generateAsync({
        type: 'blob',
        compression: 'DEFLATE',
        compressionOptions: { level: 6 }
      });

      console.log('[Background] ZIP blob generated, size:', zipBlob.size);

      // Créer nom de fichier .uxtrack
      const timestamp = new Date(session.startTime).toISOString().replace(/[:.]/g, '-').slice(0, -5);
      const filename = `sumo-session-${session.sessionId}-${timestamp}.uxtrack`;

      // Convertir blob en base64 data URL (compatible Service Worker)
      const reader = new FileReader();

      const dataUrl = await new Promise((resolve, reject) => {
        reader.onloadend = () => resolve(reader.result);
        reader.onerror = reject;
        reader.readAsDataURL(zipBlob);
      });

      console.log('[Background] Data URL created, length:', dataUrl.length);

      // Déclencher téléchargement via chrome.downloads API
      const downloadId = await chrome.downloads.download({
        url: dataUrl,
        filename: filename,
        saveAs: true
      });

      console.log('[Background] Download started with ID:', downloadId);

      return {
        success: true,
        downloadId,
        filename,
        stats: {
          totalEvents: session.events.length,
          totalScreenshots: session.screenshots.length,
          zipSize: zipBlob.size
        }
      };

    } catch (error) {
      console.error('[Background] Error creating ZIP:', error);
      return {
        success: false,
        error: error.message
      };
    }
  }
  
  handleTabActivated(activeInfo) {
    // Notifier content script si session active
    if (this.currentSession && this.currentSession.tabId === activeInfo.tabId) {
      chrome.tabs.sendMessage(activeInfo.tabId, {
        action: 'TAB_ACTIVATED'
      }).catch(() => {
        // Tab pas prêt, ignorer
      });
    }
  }

  /**
   * Met à jour le badge REC sur l'icône de l'extension
   */
  updateBadge(isRecording) {
    if (isRecording) {
      // Badge rouge "REC" pendant l'enregistrement
      chrome.action.setBadgeText({ text: 'REC' });
      chrome.action.setBadgeBackgroundColor({ color: '#EF4444' }); // Rouge vif
      chrome.action.setBadgeTextColor({ color: '#FFFFFF' }); // Texte blanc
    } else {
      // Pas de badge quand pas d'enregistrement
      chrome.action.setBadgeText({ text: '' });
    }
  }
}

// Créer instance background
const sumoBackground = new SUMOBackground();
